# Schedwi
# Copyright (C) 2011, 2012 Herve Quatremain
#
# This file is part of Schedwi.
#
# Schedwi is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
#
# Schedwi is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.


"""Module to display a calendar."""

import sys
import getopt
import datetime

import simple_queries_cal
import path_cal
import set_options
import calcomp


def usage():
    """Print a usage message on STDOUT."""
    print _("""Usage: show [OPTION]... CALENDAR
Display a calendar.  The days that match the CALENDAR formula are highlighted.

Options:
  -m, --month=MONTH month number to display (0-12).  By default, the current
                    month is displayed.  If set to 0, the whole year is showned
  -y, --year=YEAR   year to display.  By default the current year
  -h, --help        display this help
    """)


def show(sql_session, current_cwd, arguments, workload=None):
    """Display calendars.

    Arguments:
    sql_session -- SQLAlchemy session
    current_cwd -- current working directory (a path_cal.PathCal object)
    arguments -- list of arguments given to the show command (list
                 of calendars)
    workload -- workload to use

    """
    # Parse the parameters
    try:
        optlist, args = getopt.getopt(arguments, "m:y:h",
                                      [
                                          "month=",
                                          "year=",
                                          "help"
                                      ])
    except getopt.GetoptError, err:
        sys.stderr.write(_("show: ") + str(err) + "\n")
        return 1
    curr_day = datetime.date.today()
    month = curr_day.month
    year = curr_day.year
    for o, a in optlist:
        if o in ("-m", "--month"):
            try:
                m = int(a)
            except ValueError:
                sys.stderr.write(_("show: `%s': invalid month number\n") % a)
                return 1
            if m < 0 or m > 12:
                sys.stderr.write(_("show: `%s': invalid month number\n") % a)
                return 1
            if m == 0:
                month = None
            else:
                month = m
        elif o in ("-y", "--year"):
            try:
                y = int(a)
            except ValueError:
                sys.stderr.write(_("show: `%s': invalid year number\n") % a)
                return 1
            year = y
        elif o in ("-h", "--help"):
            usage()
            return 0
    paths = list()
    if args:
        for arg in args:
            paths.extend(path_cal.get_paths(sql_session, arg, current_cwd,
                                            workload=workload))
    else:
        sys.stderr.write(_("show: missing calendar name\n"))
        return 1
    if not paths:
        sys.stderr.write(_("show: no such calendar\n"))
        return 1
    first = True
    error = False
    session = sql_session.open_session()
    for p in paths:
        if first is False:
            sys.stdout.write("\n")
        else:
            first = False
        try:
            cal = simple_queries_cal.sql_get_cal(session, p.id[-1], workload,
                                                 True)
        except:
            sys.stderr.write(_("show: `%s': no such calendar\n") % p)
            error = True
            continue
        err, idx, c = calcomp.str2cal(cal.formula, year)
        set_options.print_calcomp_error(err, idx, cal.formula)
        if err != calcomp.CAL_NOERROR:
            error = True
            continue
        print "%s:" % p
        c.print_cal(month)
    sql_session.close_session(session)
    return 0 if not error else 1
