# Schedwi
# Copyright (C) 2011-2013 Herve Quatremain
#
# This file is part of Schedwi.
#
# Schedwi is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
#
# Schedwi is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.


"""Module to remove a calendar or calendar directory."""

import sys
import getopt

import simple_queries_cal
import path_cal
import whatuses
from help import print_trim


# Database ID of the root directory (/)
ROOT_DIR_ID = 1


def usage():
    """Print a usage message on STDOUT."""
    print_trim(_("""Usage: rm [OPTION]... CAL|DIR...
    Remove each specified calendar.  By default, it does not remove calendar
    directories unless the -r, -R, or --recursive options are given.

    Options:
      -f, --force          never prompt for confirmation.
      -r, -R, --recursive  remove directories and their contents recursively.
      -h, --help           display this help.
    """))


def rm_recursive_force(session, p, workload=None):
    """Recursively force delete a calendar tree.

    @param session:
                opened SQLAlchemy session.
    @param p:
                calendar/directory ro remove.
    @type p:
                L{path_cal.PathCal}
    @param workload:
                workload to consider.
    @return:    None in case of success or the full name of the directory or
                calendar which cannot be removed because it is used
                by a job/jobset.
    """
    global ROOT_DIR_ID
    try:
        cal = simple_queries_cal.sql_get_cal(session, p.id[-1], workload)
    except:
        # Cannot find the calendar or directory in the database but try
        # anyway to remove all its children (to maybe fix a database
        # incoherency)
        for child in simple_queries_cal.sql_get_cal_children(session, p.id[-1],
                                                             False, workload):
            j = path_cal.PathCal(session, id=child.id, workload=workload)
            rm_recursive_force(session, j, workload)
        return None
    if whatuses.is_used(session, cal, workload):
        return unicode(p).encode('utf-8')
    if cal.entry_type != 0:  # Directory
        cal_in_error = None
        for child in simple_queries_cal.sql_get_cal_children(session, p.id[-1],
                                                             False, workload):
            j = path_cal.PathCal(session, id=child.id, workload=workload)
            ret = rm_recursive_force(session, j, workload)
            if ret:
                cal_in_error = ret
        if cal_in_error:
            return cal_in_error
    if cal.id != ROOT_DIR_ID:
        session.delete(cal)
    else:
        return "/"
    return None


def rm_recursive(session, p, force, recursive, workload=None):
    """Remove a calendar or directory.

    Arguments:
    session -- SQLAlchemy session
    p -- calendar or directory to remove
    force -- whether or not to remove the calendar or directory without
             confirmation
    recursive -- whether or not recursively remove calendar directories.
    workload -- workload to use

    Return:
    0 -- calendar/directory removed.
    1 -- the item to remove is a directory but the recursive flag -r
         is not set.
    2 -- only some of the children calendar/directory have been removed.
    3 -- the item to remove is used in a job/jobset.

    """
    global ROOT_DIR_ID
    if force is True and recursive is True:
        ret = rm_recursive_force(session, p, workload)
        if ret:
            sys.stderr.write(_(
                "rm: `%s' is used (run `whatuses \"%s\"' for more details)\n")
                % (ret, ret))
            return 3
        else:
            return 0
    try:
        cal = simple_queries_cal.sql_get_cal(session, p.id[-1], workload)
    except:
        sys.stderr.write(_("rm: no such calendar or directory\n"))
        return 1
    if whatuses.is_used(session, cal, workload):
        sys.stderr.write(
            _("rm: `%s' is used (run `whatuses \"%s\"' for more details)\n")
            % (p, p))
        return 3
    # Calendar
    if cal.entry_type == 0:
        if force is False:
            answer = raw_input(_("rm: remove calendar `%s'? ") %
                               unicode(p).encode('utf-8'))
            if answer[0] not in (_('y'), _('Y')):
                return 2
        session.delete(cal)
        return 0
    # Directory
    if recursive is False:
        sys.stderr.write(_("rm: cannot remove `%s': Is a directory\n") % p)
        return 1
    answer = raw_input(_("rm: descend into directory `%s'? ") %
                       unicode(p).encode('utf-8'))
    if answer[0] not in (_('y'), _('Y')):
        return 2
    # Recursivly remove all the children
    error = False
    for child in simple_queries_cal.sql_get_cal_children(session, p.id[-1],
                                                         False, workload):
        j = path_cal.PathCal(session, id=child.id, workload=workload)
        ret = rm_recursive(session, j, force, recursive, workload)
        if ret:
            error = True
    if error:
        return 1
    if cal.id == ROOT_DIR_ID:
        sys.stderr.write(_("rm: cannot remove the root `/' directory\n"))
        return 1
    session.delete(cal)
    return 0


def rm(sql_session, current_cwd, arguments, workload=None):
    """Remove a calendar or a directory.

    Arguments:
    sql_session -- SQLAlchemy session
    current_cwd -- current working directory (a path_cal.PathCal object)
    arguments -- list of arguments given to the rm command (list of calendars)
    workload -- workload to use

    """
    try:
        optlist, args = getopt.getopt(arguments, "frRh",
                                      ["force", "recursive", "help"])
    except getopt.GetoptError, err:
        sys.stderr.write(_("rm: ") + str(err) + "\n")
        return 1
    force = False
    recursive = False
    for o, a in optlist:
        if o in ("-f", "--force"):
            force = True
        elif o in ("-r", "-R", "--recursive"):
            recursive = True
        elif o in ("-h", "--help"):
            usage()
            return 0
    paths = list()
    if args:
        for arg in args:
            paths.extend(path_cal.get_paths(sql_session,
                                            arg,
                                            current_cwd,
                                            workload=workload))
    else:
        if force is False:
            sys.stderr.write(_("rm: missing calendar or directory name\n"))
            return 1
        return 0
    if not paths:
        if force is False:
            sys.stderr.write(_("rm: no such calendar or directory\n"))
            return 1
        return 0
    error = False
    session = sql_session.open_session()
    for p in paths:
        try:
            ret = rm_recursive(session, p, force, recursive, workload)
        except:
            sql_session.close_session(session)
            raise
        if ret == 1:
            error = True
    sql_session.close_session(session)
    return 1 if error else 0
