/* Schedwi
   Copyright (C) 2013 Herve Quatremain

   This file is part of Schedwi.

   Schedwi is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 3 of the License, or
   (at your option) any later version.

   Schedwi is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

/* link_graph.js -- Draw the dependency graph */

/*
 * Requires:
 *  - the `graph' and `job_details' collections provided by the
 *    schedwi GUI server (see web/link_graph.py)
 *  - graph_node_positions.js for the compute_rank(), add_fake_nodes() and
 *    compute_colum_by_node() functions.
 *  - paper.js
 *  - the constant variables provided by the schedwi GUI server (see
 *    the _constants() method from web/link_graph.py)
 */

var canvas = document.getElementById(canvas_id);
paper.setup(canvas);

/*
 * Add the paper object as an attribute to the canvas object so that it can
 * be retrieved from Muntjac (Window.executeJavaScript())
 */
canvas.paper_obj = paper;

/* Size of job/jobset cells */
var cell = new paper.Size(150, 150);
var half_cell = new paper.Size();
half_cell.width = cell.width / 2.0;
half_cell.height = cell.height / 2.0;

var layer_links = new paper.Layer();
var layer_nodes = new paper.Layer();
var layer_tooltips = new paper.Layer();
var layer_zoom = new paper.Layer();


/*
 * Compute the position of a point distant from a destination point
 * of a specified distance (radius)
 *
 * @param p_source:
 *      Source point.
 * @type p_source: Point
 * @param p_dest:
 *      Destination point.
 * @type p_dest: Point
 * @param radius_dest:
 *      Distance from the destination point.
 * @return:
 *      The Point object.
 */
function get_point_radius(p_source, p_dest, radius_dest)
{
    var v1, v;

    v1 = new paper.Point(p_source.x - p_dest.x, p_source.y - p_dest.y);
    v = new paper.Point();
    v.length = radius_dest;
    v.angle = v1.angle;
    return new paper.Point(p_dest.add([v.x, v.y]));
}


/*
 * Draw an arrow image
 *
 * @param color:
 *      Color of the arrow.  If not set, the arrow is black.
 * @return:
 *      The Path object.
 */
function build_arrow(color)
{
    if (! color) {
        color = colors[NONE];
    }
    arrow = new paper.Path.RegularPolygon(new paper.Point(10, 10), 3, 10);
    arrow.fillColor = color;
    return arrow;
}


/*
 * Build and return a Paper.js Symbol for the generic arrow image (black)
 *
 * @return:
 *      The Paper.js Symbol.
 */
function build_arrow_symbol()
{
    return new paper.Symbol(build_arrow(colors[NONE]));
}


/*
 * Build and return a Paper.js Symbol for the waiting arrow (yellow)
 *
 * @return:
 *      The Paper.js Symbol.
 */
function build_arrow_waiting_symbol()
{
    return new paper.Symbol(build_arrow(colors[WAITING]));
}


/*
 * Build and return a Paper.js Symbol for the running arrow (blue)
 *
 * @return:
 *      The Paper.js Symbol.
 */
function build_arrow_running_symbol()
{
    return new paper.Symbol(build_arrow(colors[RUNNING]));
}


/*
 * Build and return a Paper.js Symbol for the failed arrow (red)
 *
 * @return:
 *      The Paper.js Symbol.
 */
function build_arrow_failed_symbol()
{
    return new paper.Symbol(build_arrow(colors[FAILED]));
}


/*
 * Build and return a Paper.js Symbol for the completed arrow (green)
 *
 * @return:
 *      The Paper.js Symbol.
 */
function build_arrow_completed_symbol()
{
    return new paper.Symbol(build_arrow(colors[COMPLETED]));
}


/*
 * Build a generic job image.  It will be used to construct symbols for
 * jobs and jobsets in their different states (waiting, running, failed, ...)
 *
 * @param color:
 *      An object with the colors to use for the image.
 * @return:
 *      The image (a Paper.js Item)
 */
function build_job(colors)
{
    if (! colors) {
        colors = {
            frame_fill_gradient_start: '#e1e1e1',
            frame_fill_gradient_end: '#828282',
            frame_stroke_gradient_start: '#666666',
            frame_stroke_gradient_end: '#c2c2c2',
            circle1_stoke_color: '#a8a8a8',
            circle2_stoke_color: '#a8a8a8',
            circle3_stoke_color: '#666666',
            circle3_fill_color: 'white'
        };
    }

    var group = new paper.Group();
    var frame = new paper.Path();
    var c1 = new paper.Path.Circle(new paper.Point(118, 163), 17.4);
    var c2 = new paper.Path.Circle(new paper.Point(118, 163), 11.4);
    var c3 = new paper.Path.Circle(new paper.Point(118, 163), 7.5);

    var fill_frame = new paper.GradientColor(
            new paper.Gradient([colors.frame_fill_gradient_start,
                                colors.frame_fill_gradient_end]),
            new paper.Point(59,98), new paper.Point(175,228));
    var stroke_frame = new paper.GradientColor(
            new paper.Gradient([colors.frame_stroke_gradient_start,
                                colors.frame_stroke_gradient_end]),
            new paper.Point(59,98), new paper.Point(175,228));

    c1.style = {
        strokeColor: colors.circle1_stoke_color,
        strokeWidth: 0.6
    };
    c2.style = {
        strokeColor: colors.circle2_stoke_color,
        strokeWidth: 1.2
    };
    c3.style = {
        strokeColor: colors.circle3_stoke_color,
        strokeWidth: 1.5,
        fillColor: colors.circle3_fill_color
    };

    frame.add(new paper.Point(100.2, 182.7));
    frame.add(new paper.Point(104.4, 179.4));
    frame.add(new paper.Point(111.6, 182.7));
    frame.add(new paper.Point(111.6, 188.4));
    frame.add(new paper.Point(119.7, 188.4));
    frame.add(new paper.Point(120, 183.3));
    frame.add(new paper.Point(127.8, 180.6));
    frame.add(new paper.Point(131.7, 184.5));
    frame.add(new paper.Point(137.4, 179.4));
    frame.add(new paper.Point(134.1, 175.2));
    frame.add(new paper.Point(137.7, 167.7));
    frame.add(new paper.Point(143.1, 167.7));
    frame.add(new paper.Point(143.1, 159.3));
    frame.add(new paper.Point(138.9, 159.3));
    frame.add(new paper.Point(135.6, 151.5));
    frame.add(new paper.Point(139.2, 147.6));
    frame.add(new paper.Point(134.1, 141.9));
    frame.add(new paper.Point(129.9, 145.5));
    frame.add(new paper.Point(122.7, 141.6));
    frame.add(new paper.Point(122.4, 135.9));
    frame.add(new paper.Point(114.6, 135.6));
    frame.add(new paper.Point(114.3, 141.3));
    frame.add(new paper.Point(106.5, 144.3));
    frame.add(new paper.Point(102.3, 140.1));
    frame.add(new paper.Point(96.9, 145.2));
    frame.add(new paper.Point(100.2, 149.4));
    frame.add(new paper.Point(96.9, 156.6));
    frame.add(new paper.Point(90.9, 156.6));
    frame.add(new paper.Point(90.9, 164.4));
    frame.add(new paper.Point(96.3, 165.3));
    frame.add(new paper.Point(98.7, 172.5));
    frame.add(new paper.Point(95.1, 177));
    frame.add(new paper.Point(100.2, 182.7));
    frame.add(new paper.Point(104.4, 179.4));
    frame.add(new paper.Point(111.6, 182.7));
    frame.add(new paper.Point(111.6, 188.4));
    frame.add(new paper.Point(119.7, 188.4));
    frame.add(new paper.Point(120, 183.3));
    frame.add(new paper.Point(127.8, 180.6));
    frame.add(new paper.Point(131.7, 184.5));
    frame.add(new paper.Point(137.4, 179.4));
    frame.add(new paper.Point(134.1, 175.2));
    frame.add(new paper.Point(137.7, 167.7));
    frame.add(new paper.Point(143.1, 167.7));
    frame.add(new paper.Point(143.1, 159.3));
    frame.add(new paper.Point(138.9, 159.3));
    frame.add(new paper.Point(135.6, 151.5));
    frame.add(new paper.Point(139.2, 147.6));
    frame.add(new paper.Point(134.1, 141.9));
    frame.add(new paper.Point(129.9, 145.5));
    frame.add(new paper.Point(122.7, 141.6));
    frame.add(new paper.Point(122.4, 135.9));
    frame.add(new paper.Point(114.6, 135.6));
    frame.add(new paper.Point(114.3, 141.3));
    frame.add(new paper.Point(106.5, 144.3));
    frame.add(new paper.Point(102.3, 140.1));
    frame.add(new paper.Point(96.9, 145.2));
    frame.add(new paper.Point(100.2, 149.4));
    frame.add(new paper.Point(96.9, 156.6));
    frame.add(new paper.Point(90.9, 156.6));
    frame.add(new paper.Point(90.9, 164.4));
    frame.add(new paper.Point(96.3, 165.3));
    frame.add(new paper.Point(98.7, 172.5));
    frame.add(new paper.Point(95.1, 177));
    frame.add(new paper.Point(100.2, 182.7));
    frame.add(new paper.Point(104.4, 179.4));
    frame.add(new paper.Point(111.6, 182.7));
    frame.add(new paper.Point(111.6, 188.4));
    frame.add(new paper.Point(119.7, 188.4));
    frame.add(new paper.Point(120, 183.3));
    frame.add(new paper.Point(127.8, 180.6));
    frame.add(new paper.Point(131.7, 184.5));
    frame.add(new paper.Point(137.4, 179.4));
    frame.add(new paper.Point(134.1, 175.2));
    frame.add(new paper.Point(137.7, 167.7));
    frame.add(new paper.Point(143.1, 167.7));
    frame.add(new paper.Point(143.1, 159.3));
    frame.add(new paper.Point(138.9, 159.3));
    frame.add(new paper.Point(135.6, 151.5));
    frame.add(new paper.Point(139.2, 147.6));
    frame.add(new paper.Point(134.1, 141.9));
    frame.add(new paper.Point(129.9, 145.5));
    frame.add(new paper.Point(122.7, 141.6));
    frame.add(new paper.Point(122.4, 135.9));
    frame.add(new paper.Point(114.6, 135.6));
    frame.add(new paper.Point(114.3, 141.3));
    frame.add(new paper.Point(106.5, 144.3));
    frame.add(new paper.Point(102.3, 140.1));
    frame.add(new paper.Point(96.9, 145.2));
    frame.add(new paper.Point(100.2, 149.4));
    frame.add(new paper.Point(96.9, 156.6));
    frame.add(new paper.Point(90.9, 156.6));
    frame.add(new paper.Point(90.9, 164.4));
    frame.add(new paper.Point(96.3, 165.3));
    frame.add(new paper.Point(98.7, 172.5));
    frame.add(new paper.Point(95.1, 177));
    frame.add(new paper.Point(100.2, 182.7));
    frame.add(new paper.Point(104.4, 179.4));
    frame.add(new paper.Point(111.6, 182.7));
    frame.add(new paper.Point(111.6, 188.4));
    frame.add(new paper.Point(119.7, 188.4));
    frame.add(new paper.Point(120, 183.3));
    frame.add(new paper.Point(127.8, 180.6));
    frame.add(new paper.Point(131.7, 184.5));
    frame.add(new paper.Point(137.4, 179.4));
    frame.add(new paper.Point(134.1, 175.2));
    frame.add(new paper.Point(137.7, 167.7));
    frame.add(new paper.Point(143.1, 167.7));
    frame.add(new paper.Point(143.1, 159.3));
    frame.add(new paper.Point(138.9, 159.3));
    frame.add(new paper.Point(135.6, 151.5));
    frame.add(new paper.Point(139.2, 147.6));
    frame.add(new paper.Point(134.1, 141.9));
    frame.add(new paper.Point(129.9, 145.5));
    frame.add(new paper.Point(122.7, 141.6));
    frame.add(new paper.Point(122.4, 135.9));
    frame.add(new paper.Point(114.6, 135.6));
    frame.add(new paper.Point(114.3, 141.3));
    frame.add(new paper.Point(106.5, 144.3));
    frame.add(new paper.Point(102.3, 140.1));
    frame.add(new paper.Point(96.9, 145.2));
    frame.add(new paper.Point(100.2, 149.4));
    frame.add(new paper.Point(96.9, 156.6));
    frame.add(new paper.Point(90.9, 156.6));
    frame.add(new paper.Point(90.9, 164.4));
    frame.add(new paper.Point(96.3, 165.3));
    frame.add(new paper.Point(98.7, 172.5));
    frame.add(new paper.Point(95.1, 177));
    frame.add(new paper.Point(100.2, 182.7));
    frame.add(new paper.Point(104.4, 179.4));
    frame.add(new paper.Point(111.6, 182.7));
    frame.add(new paper.Point(111.6, 188.4));
    frame.add(new paper.Point(119.7, 188.4));
    frame.add(new paper.Point(120, 183.3));
    frame.add(new paper.Point(127.8, 180.6));
    frame.add(new paper.Point(131.7, 184.5));
    frame.add(new paper.Point(137.4, 179.4));
    frame.add(new paper.Point(134.1, 175.2));
    frame.add(new paper.Point(137.7, 167.7));
    frame.add(new paper.Point(143.1, 167.7));
    frame.add(new paper.Point(143.1, 159.3));
    frame.add(new paper.Point(138.9, 159.3));
    frame.add(new paper.Point(135.6, 151.5));
    frame.add(new paper.Point(139.2, 147.6));
    frame.add(new paper.Point(134.1, 141.9));
    frame.add(new paper.Point(129.9, 145.5));
    frame.add(new paper.Point(122.7, 141.6));
    frame.add(new paper.Point(122.4, 135.9));
    frame.style = {
        fillColor: fill_frame,
        strokeColor: stroke_frame,
        strokeWidth: 2
    };

    group.addChild(frame);
    group.addChild(c1);
    group.addChild(c2);
    group.addChild(c3);

    return group;
}


/*
 * Build and return a Paper.js Symbol for the generic job image (grey)
 *
 * @return:
 *      The Paper.js Symbol.
 */
function build_job_symbol()
{
    return new paper.Symbol(build_job());
}


/*
 * Build and return a Paper.js Symbol for the waiting job image (yellow)
 *
 * @return:
 *      The Paper.js Symbol.
 */
function build_job_waiting_symbol()
{
    return new paper.Symbol(build_job({
            frame_fill_gradient_start: '#ffff73',
            frame_fill_gradient_end: '#bfbf30',
            frame_stroke_gradient_start: '#a6a600',
            frame_stroke_gradient_end: '#ffff40',
            circle1_stoke_color: '#ffff00',
            circle2_stoke_color: '#ffff00',
            circle3_stoke_color: '#a6a600',
            circle3_fill_color: 'white'
                                }));
}


/*
 * Build and return a Paper.js Symbol for the running job image (blue)
 *
 * @return:
 *      The Paper.js Symbol.
 */
function build_job_running_symbol()
{
    return new paper.Symbol(build_job({
            frame_fill_gradient_start: '#6997d3',
            frame_fill_gradient_end: '#274d7e',
            frame_stroke_gradient_start: '#05326d',
            frame_stroke_gradient_end: '#4282d3',
            circle1_stoke_color: '#0e51a7',
            circle2_stoke_color: '#0e51a7',
            circle3_stoke_color: '#05326d',
            circle3_fill_color: 'white'
                                }));
}


/*
 * Build and return a Paper.js Symbol for the failed job image (red)
 *
 * @return:
 *      The Paper.js Symbol.
 */
function build_job_failed_symbol()
{
    return new paper.Symbol(build_job({
            frame_fill_gradient_start: '#ff7373',
            frame_fill_gradient_end: '#bf3030',
            frame_stroke_gradient_start: '#a60000',
            frame_stroke_gradient_end: '#ff4040',
            circle1_stoke_color: '#ff0000',
            circle2_stoke_color: '#ff0000',
            circle3_stoke_color: '#a60000',
            circle3_fill_color: 'white'
                                }));
}


/*
 * Build and return a Paper.js Symbol for the completed job image (green)
 *
 * @return:
 *      The Paper.js Symbol.
 */
function build_job_completed_symbol()
{
    return new paper.Symbol(build_job({
            frame_fill_gradient_start: '#67e667',
            frame_fill_gradient_end: '#269926',
            frame_stroke_gradient_start: '#008500',
            frame_stroke_gradient_end: '#39e639',
            circle1_stoke_color: '#00cc00',
            circle2_stoke_color: '#00cc00',
            circle3_stoke_color: '#008500',
            circle3_fill_color: 'white'
                                }));
}


/*
 * Build a generic jobset image.  It will be used to construct symbols for
 * jobsets in their different states (waiting, running, failed, ...)
 *
 * @param color:
 *      An object with the colors to use for the image.
 * @return:
 *      The image (a Paper.js Item)
 */
function build_jobset(colors)
{
    var group = new paper.Group();

    var c1 = build_job(colors);
    var c2 = build_job(colors);
    c2.position = new paper.Point(164, 140);
    var c3 = build_job(colors);
    c3.position = new paper.Point(161, 188);
    c3.rotate(-20);
    group.addChild(c1);
    group.addChild(c2);
    group.addChild(c3);
    group.scale(0.7);

    return group;
}


/*
 * Build and return a Paper.js Symbol for the generic jobset image (grey)
 *
 * @return:
 *      The Paper.js Symbol.
 */
function build_jobset_symbol()
{
    return new paper.Symbol(build_jobset());
}


/*
 * Build and return a Paper.js Symbol for the waiting jobset image (yellow)
 *
 * @return:
 *      The Paper.js Symbol.
 */
function build_jobset_waiting_symbol()
{
    return new paper.Symbol(build_jobset({
            frame_fill_gradient_start: '#ffff73',
            frame_fill_gradient_end: '#bfbf30',
            frame_stroke_gradient_start: '#a6a600',
            frame_stroke_gradient_end: '#ffff40',
            circle1_stoke_color: '#ffff00',
            circle2_stoke_color: '#ffff00',
            circle3_stoke_color: '#a6a600',
            circle3_fill_color: 'white'
                                }));
}


/*
 * Build and return a Paper.js Symbol for the running jobset image (blue)
 *
 * @return:
 *      The Paper.js Symbol.
 */
function build_jobset_running_symbol()
{
    return new paper.Symbol(build_jobset({
            frame_fill_gradient_start: '#6997d3',
            frame_fill_gradient_end: '#274d7e',
            frame_stroke_gradient_start: '#05326d',
            frame_stroke_gradient_end: '#4282d3',
            circle1_stoke_color: '#0e51a7',
            circle2_stoke_color: '#0e51a7',
            circle3_stoke_color: '#05326d',
            circle3_fill_color: 'white'
                                }));
}


/*
 * Build and return a Paper.js Symbol for the failed jobset image (red)
 *
 * @return:
 *      The Paper.js Symbol.
 */
function build_jobset_failed_symbol()
{
    return new paper.Symbol(build_jobset({
            frame_fill_gradient_start: '#ff7373',
            frame_fill_gradient_end: '#bf3030',
            frame_stroke_gradient_start: '#a60000',
            frame_stroke_gradient_end: '#ff4040',
            circle1_stoke_color: '#ff0000',
            circle2_stoke_color: '#ff0000',
            circle3_stoke_color: '#a60000',
            circle3_fill_color: 'white'
                                }));
}


/*
 * Build and return a Paper.js Symbol for the completed jobset image (green)
 *
 * @return:
 *      The Paper.js Symbol.
 */
function build_jobset_completed_symbol()
{
    return new paper.Symbol(build_jobset({
            frame_fill_gradient_start: '#67e667',
            frame_fill_gradient_end: '#269926',
            frame_stroke_gradient_start: '#008500',
            frame_stroke_gradient_end: '#39e639',
            circle1_stoke_color: '#00cc00',
            circle2_stoke_color: '#00cc00',
            circle3_stoke_color: '#008500',
            circle3_fill_color: 'white'
                                }));
}


/*
 * Compute the position of the job which index is provided.
 *
 * @param node:
 *      job ID.
 * @return:
 *      The Point object that represents the job location.
 */
function job_position(node)
{
    var job_pos = new paper.Point();
    job_pos.x = column_by_node[node] * cell.width + half_cell.width;
    job_pos.y = (rank_by_node[node] - rank_min) * cell.height +
                half_cell.height;
    return job_pos;
}


/*
 * Build the tooltip Group.
 *
 * @param tooltip_message:
 *      Tooltip message to display.
 * @return:
 *      The new paper.Group object.
 */
function build_tooltip(tooltip_message)
{
    var text = new paper.PointText(new paper.Point(0, 0));
    text.content = tooltip_message;
    text.characterStyle = { fontSize: 10, fillColor: '#222222' };
    var bounds = text.bounds;

    var rect = new paper.Path.Rectangle(new paper.Point(-3, bounds.y + 3),
                                        new paper.Point(bounds.width + 3,
                                                bounds.y + bounds.height - 3));
    rect.fillColor = '#fffcdd';
    rect.strokeColor = '#b8b295';
    rect.strokeWidth = 1;
    return new paper.Group(rect, text);
}


/*
 * Display a tooltip.
 *
 * @param tooltip_message:
 *      Tooltip message to display.
 * @param position:
 *      The position (Point) of the hoovered item.
 */
var tooltip_item = null;
function tooltip_draw(tooltip_message, position)
{
    tooltip_clear();
    layer_tooltips.activate();
    tooltip_item = build_tooltip(tooltip_message);
    var bounds = tooltip_item.bounds;
    var window_size = paper.view.viewSize;
    var p = new paper.Point(position.x + bounds.width / 2,
                            position.y + bounds.height / 2 + 20);
    if ((position.x + bounds.width) > window_size.width) {
        p.x = window_size.width - bounds.width / 2;
    }
    if ((position.y + bounds.height + 20) > window_size.height) {
        p.y = window_size.height - bounds.height / 2;
    }
    tooltip_item.position = p;
}


/*
 * Hide the tooltip.
 */
function tooltip_clear()
{
    if (tooltip_item) {
        tooltip_item.remove();
        tooltip_item = null;
    }
}


/*
 * Mouse move handler (show a tooltip for links)
 *
 * @param event:
 *      The mouse event.
 */
paper.tool.onMouseMove = function(event) {
    var a = paper.project.hitTest(event.point);

    if (a && a.item.name) {
        tooltip_draw(a.item.name, event.point);
    }
    else {
        tooltip_clear();
    }
};


/*
 * Mouse drag handler (move the graph)
 *
 * @param event:
 *      The mouse event.
 */
paper.tool.onMouseDrag = function(event) {
    canvas.style.cursor = 'move';
    layer_links.translate(event.delta);
    layer_nodes.translate(event.delta);
};


/*
 * Mouse up handler (after a move)
 *
 * @param event:
 *      The mouse event.
 */
paper.tool.onMouseUp = function(event) {
    canvas.style.cursor = 'default';
};


/*
 * Zoom object
 *
 * @param layer:
 *      Layer to use to draw the zoom buttons.
 * @type layer: Layer
 * @param p:
 *      Top left corner of the zoom buttons.
 * @type p: Point
 * @param layers_to_zoom:
 *      Array of layers to scale when zoom buttons are pressed.
 * @type layers_to_zoom: Array
 */
function ZoomObject(layer, p, layers_to_zoom)
{
    var bt_size = new paper.Size(15, 15);
    var label;
    var bt;
    var r;

    this.layer = layer;
    this.layers_to_zoom = layers_to_zoom;
    this.zoom_level = 100;
    this.zoom_min = 25;
    this.zoom_max = 200;
    this.zoom_incr = 25;

    this.zoom_width_at_100 = layers_to_zoom[0].bounds.width;

    /* Draw the + and - buttons */
    layer.activate();

    label = new paper.PointText(new paper.Point(p.x + bt_size.width / 2,
                                                p.y + bt_size.height / 2 + 4));
    label.content = '+';
    label.characterStyle = { fontSize: 12, fillColor: '#777777' };
    label.paragraphStyle.justification = 'center';
    r = new paper.Rectangle({ point: p, size: bt_size });
    bt = new paper.Path.Rectangle(r, new paper.Size(5, 5));
    bt.fillColor = '#e8e8e8';
    bt.strokeColor = '#666666';
    this.zoom_in = new paper.Group([bt, label]);
    this.zoom_in.my_obj = this;
    this.zoom_in.onMouseUp = function(event) {
        var current_zoom = this.my_obj.zoom_level;
        var new_zoom = this.my_obj.zoom_level + this.my_obj.zoom_incr;

        if (new_zoom <= this.my_obj.zoom_max) {
            var scale;

            scale = (this.my_obj.zoom_width_at_100 * new_zoom) /
                    (this.my_obj.zoom_width_at_100 * current_zoom);
            for (var i = 0; i < this.my_obj.layers_to_zoom.length; i++) {
                this.my_obj.layers_to_zoom[i].scale(scale, paper.view.center);
            }
            this.my_obj.zoom_level = new_zoom;
        }
    };

    label = new paper.PointText(
                new paper.Point(p.x + bt_size.width / 2,
                           p.y + bt_size.height + bt_size.height / 2 + 4));
    label.content = '-';
    label.characterStyle = { fontSize: 12, fillColor: '#777777' };
    label.paragraphStyle.justification = 'center';
    r = new paper.Rectangle({
            point: new paper.Point(p.x, p.y + bt_size.height),
            size: bt_size });
    bt = new paper.Path.Rectangle(r, new paper.Size(5, 5));
    bt.fillColor = '#e8e8e8';
    bt.strokeColor = '#666666';
    this.zoom_out = new paper.Group([bt, label]);
    this.zoom_out.my_obj = this;
    this.zoom_out.onMouseUp = function(event) {
        var current_zoom = this.my_obj.zoom_level;
        var new_zoom = this.my_obj.zoom_level - this.my_obj.zoom_incr;

        if (new_zoom >= this.my_obj.zoom_min) {
            var scale;

            scale = (this.my_obj.zoom_width_at_100 * new_zoom) /
                    (this.my_obj.zoom_width_at_100 * current_zoom);
            for (var i = 0; i < this.my_obj.layers_to_zoom.length; i++) {
                this.my_obj.layers_to_zoom[i].scale(scale, paper.view.center);
            }
            this.my_obj.zoom_level = new_zoom;
        }
    };
}


var job_symbols = [];
job_symbols[JOBSET] = [];
job_symbols[JOBSET][NONE] = build_jobset_symbol();
job_symbols[JOBSET][WAITING] = build_jobset_waiting_symbol();
job_symbols[JOBSET][RUNNING] = build_jobset_running_symbol();
job_symbols[JOBSET][COMPLETED] = build_jobset_completed_symbol();
job_symbols[JOBSET][FAILED] = build_jobset_failed_symbol();
job_symbols[JOB] = [];
job_symbols[JOB][NONE] = build_job_symbol();
job_symbols[JOB][WAITING] = build_job_waiting_symbol();
job_symbols[JOB][RUNNING] = build_job_running_symbol();
job_symbols[JOB][COMPLETED] = build_job_completed_symbol();
job_symbols[JOB][FAILED] = build_job_failed_symbol();

var arrow_symbols = [];
arrow_symbols[NONE] = build_arrow_symbol();
arrow_symbols[WAITING] = build_arrow_waiting_symbol();
arrow_symbols[RUNNING] = build_arrow_running_symbol();
arrow_symbols[COMPLETED] = build_arrow_completed_symbol();
arrow_symbols[FAILED] = build_arrow_failed_symbol();
arrow_symbols[C_OR_F] = build_arrow_symbol();


/*
 * Main
 */

/* Compute the node positions (see graph_node_positions.js) */
compute_rank();
add_fake_nodes();
compute_colum_by_node();


/* Draw the nodes (jobs and jobsets) */
layer_nodes.activate();
var key;
for (key in job_details) {
    if (! job_details.hasOwnProperty(key)) {  /* For Lint */
        continue;
    }
    var job_pos = job_position(key);

    if (job_details[key].fake === false) {
        var g = new paper.Group();
        var img = job_symbols[job_details[key].type][job_details[key].state]
                    .place(job_pos);
        var text = new paper.PointText(job_pos.add([0,
                                             img.bounds.height / 2 + 10]));
        text.paragraphStyle.justification = 'center';
        text.content = job_details[key].name;
        text.fillColor = 'black';
        g.addChild(img);
        g.addChild(text);

        var bounds = g.strokeBounds;
        var vector = new paper.Point(bounds.topLeft.x - bounds.bottomRight.x,
                                     bounds.topLeft.y - bounds.bottomRight.y);
        job_details[key].radius = vector.length / 2.0;
    }
}


/* Draw the links */
var paths =  {};
layer_links.activate();

for (key in job_details) {
    if (! job_details.hasOwnProperty(key)) {  /* For Lint */
        continue;
    }
    if (! (key in graph)) {  /* No links */
        continue;
    }

    var job_source_pos = job_position(key);

    for (var l = 0; l < graph[key].length; l++) {
        var job_dest_id = graph[key][l].id;
        var job_dest_pos = job_position(job_dest_id);
        var p_dest;

        /* Destination point */
        if (job_details[job_dest_id].fake === true) {
            p_dest = job_dest_pos;
        }
        else {
            p_dest = get_point_radius(job_source_pos,
                                      job_dest_pos,
                                      job_details[job_dest_id].radius);
        }

        var tooltip = string_tooltip.replace('%s',
                                             job_details[job_dest_id].name);
        tooltip = tooltip.replace('%s', status_names[graph[key][l].status]);
        tooltip = tooltip.replace('%s', job_details[key].name);

        var p;
        var p_src;
        if (job_details[key].fake === false || !(key in paths)) {
            p = new paper.Path();
            p.name = tooltip;
            p.strokeColor = colors[graph[key][l].status];
            p.strokeWidth = 4;
            if (job_details[key].fake === true) {
                p_src = job_source_pos;
            }
            else {
                p_src = get_point_radius(job_dest_pos,
                                         job_source_pos,
                                         job_details[key].radius);
            }
            p.add(p_src, p_dest);
        }
        else {
            p = paths[key];
            p.add(p_dest);
            p.smooth();
        }
        paths[job_dest_id] = p;

        /* Arrows */
        if (job_details[key].fake === false) {
            var v;
            var arrow = arrow_symbols[graph[key][l].status].place(p_src);
            var s = p.firstSegment;

            arrow.name = tooltip;
            if (s.handleIn.x !== 0 && s.handleIn.y !== 0) {
                v = s.handleIn;
            }
            else {
                v = new paper.Point(p_dest.x - p_src.x,
                                    p_dest.y - p_src.y);
            }
            arrow.rotate(v.angle - 90);
        }
    }
}

var zoom = new ZoomObject(layer_zoom, new paper.Point(5, 5),
        [layer_nodes, layer_links]);

paper.view.draw();

