/* Schedwi
   Copyright (C) 2007-2010 Herve Quatremain

   This file is part of Schedwi.

   Schedwi is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 3 of the License, or
   (at your option) any later version.

   Schedwi is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

/* send_result.c -- Send the result of a job to the server */

#include <schedwi.h>

#if STDC_HEADERS
#include <stdlib.h>
#include <string.h>
#endif

#if HAVE_TIME_H
#include <time.h>
#endif

#if HAVE_UNISTD_H
#include <unistd.h>
#endif

#if HAVE_ERRNO_H
#include <errno.h>
#endif
#ifndef errno
extern int errno;
#endif

#if HAVE_ASSERT_H
#include <assert.h>
#endif

#include <lib_functions.h>

#if HAVE_DIRENT_H
# include <dirent.h>
# define NAMLEN(dirent) schedwi_strlen((dirent)->d_name)
#else
# define dirent direct
# define NAMLEN(dirent) (dirent)->d_namlen
# if HAVE_SYS_NDIR_H
#  include <sys/ndir.h>
# endif
# if HAVE_SYS_DIR_H
#  include <sys/dir.h>
# endif
# if HAVE_NDIR_H
#  include <ndir.h>
# endif
#endif

#include <lwc_linkedlist.h>
#include <lwc_log.h>
#include <conf.h>
#include <utils.h>
#include <net_utils_glob.h>
#include <net_module.h>
#include <result_mgnt.h>
#include <send_result.h>


/*
 * Send the list of completed jobs to the server
 *
 * Return:
 *   0 --> No error
 *  -1 --> Memory allocation error (a message is logged using lwc_writeLog())
 *  -2 --> Network error (a message is logged using lwc_writeLog())
 *  -3 --> The server fails to manage the request (a message is logged using
 *         lwc_writeLog())
 */
static int
net_send_list (lwc_LL *jobs_to_insert)
{
	char *txt, *server_reply;
	size_t txt_len;
	int ret;
	file_result_t *ptr;
	const char *hostname, *port_number;
	net_id sock;
#if HAVE_NANOSLEEP
	struct timespec req;

	req.tv_sec = 0;
	req.tv_nsec = 500000000; /* 0.5 sec */
#endif


	if (jobs_to_insert == NULL || lwc_getNumNode (jobs_to_insert) == 0) {
		return 0;
	}

	ret = conf_get_param_string ("SERVER", &hostname);
	ret += conf_get_param_string ("SERVER_PORT", &port_number);
#if HAVE_ASSERT_H
	assert (ret == 0);
#endif

	/* For each job, construct and send the result string */
	lwc_rewindLL (jobs_to_insert);
	txt = NULL;
	txt_len = 0;
	while ((ptr = (file_result_t *) lwc_nextLL (jobs_to_insert)) != NULL) {

		ret = result_to_str (ptr->r, &txt, &txt_len);
		if (ret != 0) {
			if (txt != NULL) {
				free (txt);
			}
			if (ret == -1) {
				lwc_writeLog (	LOG_CRIT,
						_("Memory allocation error"));
			}
			else {
				lwc_writeLog (LOG_CRIT,
					_("Internal error: buffer too small"));
			}
			return -1;
		}

		/* Connect to the server */
		if (net_client (&sock, port_number, hostname) != 0) {
			/*
			 * Maybe the server if overloaded.
			 * Throttle the client (sleep 0.5 s) and try again
			 */
#if HAVE_NANOSLEEP
			nanosleep (&req, 0);
#elif HAVE_SLEEP
			sleep (1);
#elif HAVE_USLEEP
			usleep (500000);
#endif
			if (net_client (&sock, port_number, hostname) != 0) {
				free (txt);
				return -2;
			}
		}

		/* Send the result */
		if (net_write (&sock, txt, schedwi_strlen (txt)) != 0) {
			net_close (&sock);
			free (txt);
			return -2;
		}

		/* Read the result code returned by the server */
		server_reply = NULL;
		ret = net_read_result (&sock, &server_reply);
		net_close (&sock);
		if (ret < 0) {
			/* Network error */
			free (txt);
			if (server_reply != NULL) {
				free (server_reply);
			}
			return -2;
		}
		if (ret != 0) {
			/* Server failure */
			free (txt);
			if (server_reply != NULL) {
				lwc_writeLog (LOG_ERR, _("Server failure: %s"),
						server_reply);
				free (server_reply);
			}
			else {
				lwc_writeLog (LOG_ERR, _("Server failure"));
			}
			return -3;
		}
		if (server_reply != NULL) {
			free (server_reply);
		}

		/* Success from the server, remove the result file */
		if (my_unlink (ptr->file_name) != 0) {
			lwc_writeLog (	LOG_CRIT,
				_("Cannot remove the result file %s: %s"),
					ptr->file_name,
					strerror (errno));
		}
	}
	if (txt != NULL) {
		free (txt);
	}

	return 0;
}


/*
 * Read the files in the result directory and send the list of completed jobs
 * to the server
 *
 * Return:
 *   0 --> No error
 *  -1 --> Memory allocation error (a message is logged using lwc_writeLog())
 *  -2 --> System error (a message is logged using lwc_writeLog())
 */
int
send_result (const char *result_dir, const char *prefix, const char *suffix)
{
	DIR *d;
	struct dirent *f;
	lwc_LL *list;
	size_t prefix_length, suffix_length, name_length;
	int ret;
	file_result_t *ret_ptr;

#if HAVE_ASSERT_H
	assert (result_dir != NULL && prefix != NULL);
#endif

	/* New linked list */
	list = lwc_newLL ();
	if (list == NULL) {
		lwc_writeLog (LOG_CRIT, _("Memory allocation error"));
		return -1;
	}

	/* Open the result direcory */
	d = opendir (result_dir);
	if (d == NULL) {
		lwc_writeLog (LOG_ERR,
		_("Cannot open the RESULT_DIR directory %s: opendir: %s"),
			result_dir, strerror (errno));
		lwc_delLL (list, NULL);
		return -2;
	}

	prefix_length = schedwi_strlen (prefix);
	if (suffix != NULL) {
		suffix_length = schedwi_strlen (suffix);
	}
	else {
		suffix_length = 0;
	}

	/* Read the directory */
	while ((f = readdir (d)) != NULL) {
		name_length = schedwi_strlen (f->d_name);
		if (	   prefix_length + suffix_length >= name_length
			|| schedwi_strncmp (	prefix, f->d_name,
						prefix_length) != 0)
		{
			continue;
		}
		if (	   suffix != NULL
			&& strcmp (suffix,
				f->d_name + name_length - suffix_length) != 0)
		{
			continue;
		}

		/* Read the file */
		ret = new_file_result (result_dir, f->d_name, &ret_ptr);
		if (ret != 0) {
			if (ret == -1) {
				lwc_writeLog (	LOG_CRIT,
						_("Memory allocation error"));
			}
 			else {
				lwc_writeLog (	LOG_ERR,
				_("Cannot read the result file %s in %s: %s"),
						f->d_name,
						result_dir,
						strerror (errno));
			}
			closedir (d);
			lwc_delLL (list,
				(void (*)(const void *))destroy_file_result);
			return ret;
		}

		/* Store the result in the list */
		if (ret_ptr != NULL && lwc_addEndLL (list, ret_ptr) != 0) {
			closedir (d);
			lwc_delLL (list,
				(void (*)(const void *))destroy_file_result);
			destroy_file_result (ret_ptr);
			lwc_writeLog (LOG_CRIT, _("Memory allocation error"));
			return -1;
		}
	}
	closedir (d);
	ret = net_send_list (list);
	lwc_delLL (list, (void (*)(const void *))destroy_file_result);
	return ret;
}

/*------------------------======= End Of File =======------------------------*/
