/* Schedwi
   Copyright (C) 2010, 2011 Herve Quatremain

   This file is part of Schedwi.

   Schedwi is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 3 of the License, or
   (at your option) any later version.

   Schedwi is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

/*
 * parse_json_result.c -- Parse the JSON string which contains the result
 * sent by the other end to a request.
 * The JSON string looks like:
 *     {
 *         "success" : false,
 *          "data" : "Database unavailable"
 *     }
 */

#include <schedwi.h>
#if STDC_HEADERS
#include <stdlib.h>
#include <string.h>
#endif

#if HAVE_ASSERT_H
#include <assert.h>
#endif

#include <parse_json_result.h>
#include <lwc_log.h>
#include <lib_functions.h>
#include <JSON_parser.h>


enum steps {
	S_START,
	S_DATA,
	S_KEY_SUCCESS,
	S_KEY_DATA,
	S_KEY_REASON,
	S_END
};

struct json_cb {
	enum steps step;
	char error;
	error_reason_t reason;
	int failure;
	char *message;
};


/*
 * JSON callback function
 */
static int
json_char_callback (void* ctx, int type, const JSON_value* value)
{
	struct json_cb *s = (struct json_cb *)ctx;
	char *v;

	switch (type) {
		case JSON_T_OBJECT_BEGIN:
			if (s->step == S_START) {
				s->step = S_DATA;
			}
			break;
		case JSON_T_KEY:
			if (s->step == S_DATA) {
				if (schedwi_strcasecmp (value->vu.str.value,
							"success") == 0)
				{
					s->step = S_KEY_SUCCESS;
				}
				else
				if (schedwi_strcasecmp (value->vu.str.value,
							"data") == 0)
				{
					s->step = S_KEY_DATA;
				}
				else
				if (schedwi_strcasecmp (value->vu.str.value,
							"reason") == 0)
				{
					s->step = S_KEY_REASON;
				}
			}
			break;
		case JSON_T_TRUE:
			if (s->step == S_KEY_SUCCESS) {
				s->failure = 0;
				s->step = S_DATA;
			}
			break;
		case JSON_T_FALSE:
			if (s->step == S_KEY_SUCCESS) {
				s->failure = 1;
				s->step = S_DATA;
			}
			break;
		case JSON_T_STRING:
			if (s->step == S_KEY_DATA) {
				v = (char *)malloc (value->vu.str.length + 1);
				if (v == NULL) {
					s->error = 1;
					return 1;
				}
				strncpy (v, value->vu.str.value,
						value->vu.str.length);
				v[value->vu.str.length] = '\0';
				if (s->message != NULL) {
					free (s->message);
				}
				s->message = v;
				s->step = S_DATA;
			}
			break;
		case JSON_T_INTEGER:
			if (s->step == S_KEY_REASON) {
				s->reason = int_to_reg_error (
						value->vu.integer_value);
				s->step = S_DATA;
			}
			break;
		case JSON_T_OBJECT_END:
			if (s->step == S_DATA) {
				s->step = S_END;
			}
			break;
	}

	return 1;
}


/*
 * Parse the JSON string.
 *
 * The format of the string should be as follow:
 *     {
 *         "success" : false,
 *          "data" : "Database unavailable"
 *     }
 * OR
 *     {
 *         "success" : false,
 *         "reason" : 2,
 *          "data" : "Certificate still not ready"
 *     }
 *
 * Return:
 *   0 --> The reply means the remote action was successfull. If result_msg is
 *         not NULL, it is set with the message sent by the remote socket. It
 *         must be freed by the caller.
 *   1 --> The reply means the remote action failed. If result_msg is not NULL,
 *         it is set with the message sent by the remote socket. It may be NULL
 *         if no message was sent. It must be freed by the caller.
 *  -1 --> Error (a message is displayed using lwc_writeLog())
 */
int
parse_json_result (const char *json_string, ssize_t json_string_length,
			char **result_msg, error_reason_t *reason)
{
	JSON_config config;
	struct JSON_parser_struct *jc;
	struct json_cb cb_data;
	int i;

#if HAVE_ASSERT_H
	assert (json_string != NULL && json_string_length > 0);
#endif

	/* JSON parser initialization */
	init_JSON_config (&config);

	cb_data.step                  = S_START;
	cb_data.error                 = 0;
	cb_data.reason                = REG_OK;
	cb_data.failure               = 1;
	cb_data.message               = NULL;
	config.depth                  = 20;
	config.callback               = &json_char_callback;
	config.allow_comments         = 0;
	config.handle_floats_manually = 0;
	config.callback_ctx           = &cb_data;

	jc = new_JSON_parser (&config);

	for (i = 0; i < json_string_length && cb_data.step != S_END; i++) {
		/*
		 * No need to check the return code as the JSON string has
		 * already been parsed once in net_parse.c
		 */
		JSON_parser_char (jc, json_string[i]);
		if (cb_data.error != 0) {
			delete_JSON_parser (jc);
			if (cb_data.message != NULL) {
				free (cb_data.message);
			}
			lwc_writeLog (LOG_CRIT, _("Memory allocation error"));
			return -1;
		}
	}
	delete_JSON_parser (jc);

	if (reason != NULL) {
		*reason = cb_data.reason;
	}
	if (result_msg != NULL) {
		*result_msg = cb_data.message;
	}
	else {
		if (cb_data.message != NULL) {
			free (cb_data.message);
		}
	}
	return cb_data.failure;
}

/*------------------------======= End Of File =======------------------------*/
