/* Schedwi
   Copyright (C) 2007 Herve Quatremain

   This file is part of Schedwi.

   Schedwi is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 3 of the License, or
   (at your option) any later version.

   Schedwi is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

/* workload_class.c -- Functions to run a workload */

#include <schedwi.h>

#if STDC_HEADERS
#include <stdlib.h>
#include <string.h>
#else
#if HAVE_STDLIB_H
#include <stdlib.h>
#endif
#if HAVE_STRING_H
#include <string.h>
#endif
#endif

#if HAVE_UNISTD_H
#include <unistd.h>
#endif

#if HAVE_SIGNAL_H
#include <signal.h>
#endif

#if HAVE_ERRNO_H
#include <errno.h>
#endif
#ifndef errno
extern int errno;
#endif

#if HAVE_ASSERT_H
#include <assert.h>
#endif


#include <lwc_log.h>
#include <thread_init.h>
#include <workload_class.h>

/*
 * Each minute the workload is checked.  If the sleep time between checks
 * appears to be longer than a minute, something is wrong.  The
 * MAX_DRIFT_SEC specifies in seconds what is the drift allowed.
 * If this limit is passed some actions are taken.  This is probably a
 * system time adjustement (daylight saving for instance).
 */
#define MAX_DRIFT_SEC 60


/*
 * Comparison function between a workload date and the the date of the provided
 * workload_class object.  This function is used for the lwc_searchLL()
 * function in load_workload_class_find() (in load_workload_class.c)
 */
int
workload_class_date_compar (	const schedwi_date *workload_date,
				const workload_class_ptr ptr)
{
#if HAVE_ASSERT_H
	assert (workload_date != NULL && ptr != NULL);
#endif

	return schedwi_date_compar (workload_date, &(ptr->workload_date));
}


/*
 * Thread to run the workload
 */
static void *
workload_class_run (void *obj)
{
	workload_class_ptr ptr = (workload_class_ptr) obj;
	time_t before, after;
	int sleep_duration;
#if HAVE_NANOSLEEP
	struct timespec req;

	req.tv_sec  = 60;
	req.tv_nsec = 0;
#endif

#if HAVE_ASSERT_H
	assert (ptr != NULL);
#endif

	/* Initialize the thread */
	thread_init ();

	/* Install the cleanup callback */
	pthread_cleanup_push (thread_clean, NULL);

	/* Run the job/job set tree every minute */
	time (&after);
	while (1) {

		/* Walk through the tree to start jobs */
		schedwi_jobtree_run (ptr->jobtree, after);

		time (&before);

		/* Sleep for one minute */
#if HAVE_NANOSLEEP
		nanosleep (&req, NULL);
#elif HAVE_SLEEP
		sleep (60);
#else
		/*
		 * usleep() should not be used.  On some OS it may
		 * not work correctly with pthread
		 */
		usleep (60 * 1000000);
#endif

		time (&after);
		pthread_testcancel ();

		/*
		 * Check the drift time
		 */
#if HAVE_DIFFTIME
		sleep_duration = (int)difftime (after, before);
#else
		sleep_duration = after - before;
#endif

		/* System clock has gained or lost time */
		if (	   sleep_duration > MAX_DRIFT_SEC + 60
			|| sleep_duration < 60 - MAX_DRIFT_SEC)
		{
			schedwi_jobtree_adjust (ptr->jobtree,
						sleep_duration - 60,
						after);
		}
	}
	pthread_cleanup_pop (1);
	return NULL;
}


/*
 * Create a new workload_class_ptr object.  A new thread is created by this
 * function
 *
 * Return:
 *   The new object to be freed by the caller by workload_class_destroy() or
 *   NULL in case of error (an error has been logged by lwc_writeLog())
 */
workload_class_ptr
workload_class_new (schedwi_date workload_date, calendar_list_t_ptr calendars)
{
	workload_class_ptr ptr;
	pthread_attr_t attr;

#if HAVE_ASSERT_H
	assert (calendars != NULL);
#endif

	ptr = (workload_class_ptr) malloc (sizeof (workload_class));
	if (ptr == NULL) {
		lwc_writeLog (LOG_CRIT, _("Memory allocation error"));
		return NULL;
	}

	/* Build the jobtree_ptr object */
	ptr->jobtree = new_jobtree (workload_date, calendars);
	if (ptr->jobtree == NULL) {
		free (ptr);
		return NULL;
	}

	ptr->workload_date = workload_date;

	if (pthread_attr_init (&attr) != 0) {
		lwc_writeLog (	LOG_ERR,
			_("Internal error: attribute initialization: %s"),
				strerror (errno));
		free_jobtree (ptr->jobtree);
		free (ptr);
		return NULL;
	}
	pthread_attr_setdetachstate (&attr, PTHREAD_CREATE_JOINABLE);

	if (pthread_create (	&(ptr->thread), &attr,
				workload_class_run, ptr) != 0)
	{
		lwc_writeLog (	LOG_ERR,
				_("Internal error: thread creation: %s"),
				strerror (errno));
		pthread_attr_destroy (&attr);
		free_jobtree (ptr->jobtree);
		free (ptr);
		return NULL;
	}
	pthread_attr_destroy (&attr);
	return ptr;
}


/*
 * Destroy the provided workload_class_ptr object.  The associated thread is
 * destroyed.
 */
void
workload_class_destroy (workload_class_ptr ptr)
{
	if (ptr != NULL) {
		pthread_cancel (ptr->thread);
		pthread_kill (ptr->thread, SIGUSR2);
		pthread_join (ptr->thread, NULL);
		free_jobtree (ptr->jobtree);
		free (ptr);
	}
}


/*
 * Check if the workload is completed (ie. the root node is completed)
 *
 * Return:
 *   0 --> Not completed
 *   1 --> Completed
 */
int
workload_class_is_completed (workload_class_ptr ptr)
{
	if (ptr == NULL) {
		return 1;
	}
	return schedwi_jobtree_is_completed (ptr->jobtree);
}


/*
 * Call the provided function for each jobs and jobsets starting from
 * the privided job/jobset ID.
 * If foreach_func() returns a value different from 0, the recursion is
 * stopped.
 */
void
workload_class_job_foreach (	workload_class_ptr ptr,
				unsigned long long int id,
				int (*foreach_func)(schedwi_jobtree_node_ptr,
							void *),
				void *data_user)
{
	if (ptr != NULL && foreach_func != NULL) {
		schedwi_jobtree_job_foreach (	ptr->jobtree, id, foreach_func,
						data_user);
	}
}


/*
 * Set the status of a job
 *
 * Return:
 *   0 --> No error
 *  -1 --> Error.  An error message has been logged by using lwc_writeLog()
 *  -2 --> Database error.  An error message has been logged by using
 *         lwc_writeLog()
 */
int
workload_class_job_finished (	workload_class_ptr ptr,
				unsigned long long int id,
				job_status_state status,
				long int duration,
				const char *status_message)
{
	if (ptr == NULL) {
		return 0;
	}

	return schedwi_jobtree_job_finished (	ptr->jobtree, id, status,
						duration, status_message);
}


/*
 * Print the job/jobset status for the provided workload
 */
void
workload_class_print (workload_class_ptr ptr)
{
	char *s;

	if (ptr != NULL) {
		s = schedwi_date_to_string (ptr->workload_date);
		if (s == NULL) {
			lwc_writeLog (LOG_CRIT, _("Memory allocation error"));
		}
		else {
			lwc_writeLog (LOG_INFO, _("[%s]"), s);
			free (s);
			schedwi_jobtree_print (ptr->jobtree);
		}
	}
}

/*-----------------============== End Of File ==============-----------------*/
