/* Schedwi
   Copyright (C) 2007-2013 Herve Quatremain

   This file is part of Schedwi.

   Schedwi is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 3 of the License, or
   (at your option) any later version.

   Schedwi is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

/* schedwisrv.c -- Schedwi server */

#include <schedwi.h>

#if HAVE_SYS_TYPES_H
#include <sys/types.h>
#endif

#if STDC_HEADERS
#include <stdlib.h>
#include <string.h>
#endif

#if HAVE_STDIO_H
#include <stdio.h>
#endif

#if HAVE_UNISTD_H
#include <unistd.h>
#endif

#if HAVE_TIME_H
#include <time.h>
#endif

#if HAVE_SYS_WAIT_H
# include <sys/wait.h>
#endif

#if HAVE_SIGNAL_H
#include <signal.h>
#endif

#if HAVE_GETOPT_H
#include <getopt.h>
#endif

#ifdef HAVE_LOCALE_H
#include <locale.h>
#endif

#if HAVE_ERRNO_H
#include <errno.h>
#endif
#ifndef errno
extern int errno;
#endif

#if HAVE_ASSERT_H
#include <assert.h>
#endif

#include <pthread.h>

#include <sql_common.h>
#include <utils.h>
#include <conf.h>
#include <conf_srv.h>
#include <lwc_log.h>
#include <net_utils.h>
#include <net_utils_sock.h>
#include <net_utils_ssl.h>
#include <net_parse.h>
#include <schedwi_time.h>
#include <load_workload_class.h>
#include <signal_utils.h>
#include <thread_init.h>
#include <commands_thread.h>
#include <module.h>
#include <lib_functions.h>
#include <cert_ca.h>


static const char *pid_file = NULL;
static const char *configuration_file = NULL;
static pthread_t net_server_thread;
static load_workload_class_ptr workload_list = NULL;

struct socket_set {
	fd_set sock_set;
	int sock_max;
	const char *allow_from;
};



/*
 * Print a help message to stderr
 */
void
help (const char * prog_name)
{
	int i;
	char * msg[] = {
N_("The Schedwi server."),
"",
#if HAVE_GETOPT_LONG
N_("  -c, --config=FILE  use the configuration file FILE rather than"),
N_("                     the default one"),
N_("  -t, --test         Run syntax tests for the configuration file. The"),
N_("                     program immediately exits after these syntax"),
N_("                     parsing tests with either a return code of 0 (syntax"),
N_("                     OK) or return code not equal to 0 (syntax error)"),
#if HAVE_WORKING_FORK
N_("  -f, --foreground   keep the program in foreground"),
#endif
N_("  -h, --help         display this help and exit"),
N_("  -V, --version      print version information and exit"),
#else /* HAVE_GETOPT_LONG */
N_("  -c FILE            use the configuration file FILE rather than"),
N_("                     the default one"),
N_("  -t                 Run syntax tests for the configuration file. The"),
N_("                     program immediately exits after these syntax"),
N_("                     parsing tests with either a return code of 0 (syntax"),
N_("                     OK) or return code not equal to 0 (syntax error)"),
#if HAVE_WORKING_FORK
N_("  -f                 keep the program in foreground"),
#endif
N_("  -h                 display this help and exit"),
N_("  -V                 print version information and exit"),
#endif /* ! HAVE_GETOPT_LONG */
"",
"~",
N_("Exit status is 1 on error (0 means no error)."),
NULL
	};

#if HAVE_ASSERT_H
	assert (prog_name != NULL && configuration_file != NULL);
#endif

	fprintf (stderr, _("Usage: %s [OPTION]...\n"), prog_name);

	for (i = 0; msg[i] != NULL; i++) {
		if (msg[i][0] == '~') {
			fputs (	_("The default configuration file is "),
				stderr);
			fputs (configuration_file, stderr);
		}
		else {
			fputs ((msg[i][0] == '\0')? msg[i]: _(msg[i]), stderr);
		}
		fputc ('\n', stderr);
	}

	fputc ('\n', stderr);
	fputs (_("Report bugs to "), stderr);
	fputs (PACKAGE_BUGREPORT, stderr);
	fputc ('\n', stderr);
}


/*
 * Signal handler for SIGHUP
 */
static void
sighup_action ()
{
	int ret;
	const char *log_file, *date_format;
	long int facility;

	/*
	 * Before anything else be sure that the configuration file
	 * can be read
	 */
	if (access (configuration_file, R_OK) != 0) {
		lwc_writeLog (	LOG_ERR, _("%s: %s"),
				configuration_file, strerror (errno));
		return;
	}

	close_pid_file (pid_file);
	conf_destroy_srv ();

	/*
	 * Re-read the configuration file.
	 * WARNING an error in this file (missing file or syntax error) will
	 * exit the server daemon
	 */
	ret = conf_init_srv (configuration_file);
	switch (ret) {
		case -1:
			lwc_writeLog (LOG_CRIT, _("Memory allocation error"));
			break;

		case -2:
			lwc_writeLog (LOG_CRIT,
				_("Cannot read the configuration file %s: %s"),
				configuration_file,
				strerror (errno));
			break;

		case -3:
			lwc_writeLog (LOG_CRIT,
				_("Syntax error in the configuration file %s"),
				configuration_file);
			break;

	}
	if (ret != 0) {
		exit (1);
	}

	lwc_closeLog ();

	/*
	 * Open the log file
	 */
	log_file = date_format = NULL;
	facility = -1;
	ret = conf_get_param_string ("LOG_FILE", &log_file);
	ret += conf_get_param_syslog_facility ("SYSLOG", &facility);
	ret += conf_get_param_string ("DATE_FORMAT", &date_format);
#if HAVE_ASSERT_H
	assert (ret == 0);
#endif

	lwc_newLog (    PACKAGE_NAME,
			(facility != -1) ? 1 : 0,
			(facility != -1) ? (int)facility : 0,
			log_file,
			date_format);
	lwc_writeLog (LOG_INFO, _("Restarting"));

	/*
	 * Create the PID file and check that an other process is not already
	 * running
	 */
	ret = conf_get_param_string ("PID_FILE", &pid_file);
#if HAVE_ASSERT_H
	assert (ret == 0);
#endif
	ret = write_pid_file (pid_file);
	switch (ret) {
		case -1:
			lwc_writeLog (	LOG_CRIT,
					_("Cannot create the PID file %s: %s"),
					pid_file,
					strerror (errno));
			break;
		
		case 1:
			lwc_writeLog (LOG_ERR,
				_("An other daemon is already running"));
			break;
	}
}

/*
 * Prepare the alarm to wake-up the thread at midnight
 * Every time the alarm is triggered the sigalrm_action() is called
 */
static void
prepare_alarm ()
{
	/* Wake-up at midnight (+5 sec to be sure that it will be tomorrow) */
	alarm (schedwi_time_to_midnight () + 5);
}


/*
 * Signal handler for SIGALRM
 */
static void
sigalrm_action()
{
	static schedwi_date previous = { 0, 0, 0};
	schedwi_date now;
	int ret;
	long int retention_completed, retention_failed;

	/* Retrieve the retention values */
	ret = conf_get_param_int (	"WORKLOAD_PURGE_COMPLETED",
					&retention_completed);
	ret += conf_get_param_int (	"WORKLOAD_PURGE_FAILED",
					&retention_failed);
#if HAVE_ASSERT_H
	assert (ret == 0);
#endif

	schedwi_date_now (&now);

	/* Clean the workloads (remove the completed ones) */
	load_workload_class_clean_list (workload_list, now,
					retention_completed, retention_failed);

	/* If it's really a new day (it should be) */
	if (schedwi_date_compar (&now, &previous) != 0) {

		/* Load the workload for today */
		if (load_workload_class_for_day (workload_list, now) == 0) {
			previous = now;
		}
		else {
			/*
			 * Failed to load the workload.
			 * Try again in a minute
			 */
			alarm (60);
			return;
		}
	}

	prepare_alarm ();
}


/*
 * Network connection management thread
 */
static void *
net_server_thread_func (void *data)
{
	struct socket_set *s_set = (struct socket_set *)data;

	if (s_set == NULL) {
		pthread_exit (NULL);
	}

	/* Initialize the thread */
	thread_init ();

	/* Install the cleanup callback */
	pthread_cleanup_push (thread_clean, NULL);

	/* Manage network connections */
	net_accept (	&(s_set->sock_set), s_set->sock_max, s_set->allow_from,
			net_parse, workload_list);

	pthread_cleanup_pop (1);
	pthread_exit (NULL);
}


/*
 * Create the thread to manage the agent connections
 */
static int
start_net_server (struct socket_set *s_set)
{
	pthread_attr_t attr;

	/* Initialize the thread attribute object */
	pthread_attr_init (&attr);
	pthread_attr_setdetachstate (&attr, PTHREAD_CREATE_JOINABLE);

	if (pthread_create (	&net_server_thread, &attr,
				net_server_thread_func, s_set) != 0)
	{
		lwc_writeLog (	LOG_CRIT,
	_("Cannot create a new thread to manage network connections: %s"),
				strerror (errno));
		pthread_attr_destroy (&attr);
		return -1;
	}
	pthread_attr_destroy (&attr);
	return 0;
}


/*
 * Main function
 *
 * The exit code is 1 in case of error or 0 if killed (SIGTERM)
 */
int
main (int argc, char **argv)
{
	int ret, sig, fd;
	const char *group_id, *user_id;
	const char *log_file, *date_format;
	const char *iface, *server_port, *address_family;
	const char *server_key, *server_crt, *ca_crt, *ca_key, *ca_crl;
	char quick_random;
	const char *prog_name;
	long int facility;
	char test, foreground;
	dbi_conn sql_handler;
	char *err_msg;
	schedwi_date now;
	sigset_t waitset;
	struct socket_set s_set;

#if HAVE_GETOPT_LONG
	int option_index;
	struct option long_options[] =
	{
		{"help",       0, 0, 'h'},
		{"version",    0, 0, 'V'},
		{"test",       0, 0, 't'},
		{"foreground", 0, 0, 'f'},
		{"config",     1, 0, 'c'},
		{0, 0, 0, 0}
	};
#endif

#if HAVE_SETLOCALE
	setlocale (LC_ALL, "");
#endif
#if HAVE_BINDTEXTDOMAIN
	bindtextdomain (PACKAGE, LOCALEDIR);
#endif
#if HAVE_TEXTDOMAIN
	textdomain (PACKAGE);
#endif


	/* Set default values for options */
	test = foreground = 0;
	configuration_file = SCHEDWI_DEFAULT_CONFFILE_SRV;

	prog_name = base_name (argv[0]);

	/* Parse options */
	while (1) {
#if HAVE_GETOPT_LONG
		option_index = 0;
		ret = getopt_long (argc, argv, "hVtfc:",
					long_options, &option_index);
#else
		ret = getopt (argc, argv, "hVtfc:");
#endif

		if (ret == -1) {
			break;
		}

		switch (ret) {
			case 'h':
				help (prog_name);
				return 0;
			case 'V':
				version (prog_name);
				return 0;
			case 't':
				test = 1;
				break;
			case 'f':
				foreground = 1;
				break;
			case 'c':
				configuration_file = optarg;
				break;
			default:
				help (prog_name);
				return 1;
		}
	}

	if (test == 1) {
		printf (_("Configuration file: %s\n"), configuration_file);
	}

	/*
	 * seed for the rand() function that is used through the program
	 */
	srand ((unsigned int) time (NULL));

	/*
	 * Initialize the module engine
	 */
	if (module_init () != 0) {
		return 1;
	}

	/*
	 * Read the configuration file
	 */
	ret = conf_init_srv (configuration_file);
	switch (ret) {
		case -1:
			fputs (_("Memory allocation error\n"), stderr);
			break;

		case -2:
			perror (configuration_file);
			break;
	}
	if (ret != 0) {
		module_exit ();
		return 1;
	}

	ret =  conf_get_param_string ("SSLCACertificateFile", &ca_crt);
	ret += conf_get_param_string ("SSLCACertificateKeyFile", &ca_key);
	ret += conf_get_param_string ("SSLCACRLFile", &ca_crl);
	ret += conf_get_param_string ("SSLCertificateFile", &server_crt);
	ret += conf_get_param_string ("SSLCertificateKeyFile", &server_key);
	ret += conf_get_param_bool ("SSLQuickRandom", &quick_random);
	ret += conf_get_param_string ("USER", &user_id);
	ret += conf_get_param_string ("GROUP", &group_id);
	ret += conf_get_param_string ("PID_FILE", &pid_file);
#if HAVE_ASSERT_H
	assert (ret == 0);
#endif

	/*
	 * Initialize the GnuTLS layer
         *
	 * The gnutls_global_init() function seems to open /dev/urandom.
	 * To make *almost* sure that the file descriptor to be use is 3, we
	 * start to close all the file descriptors above 2.  Later on, the
	 * clean_process() function will close all the remaining file
	 * descriptors, except 3.
	 */
	for (fd = getdtablesize () - 1; fd >= 3; fd--) {
		close (fd);
	}
	if (net_init_gnutls (quick_random) != 0) {
		conf_destroy_srv ();
		module_exit ();
		return 1;
	}

	/*
	 * Create the keys and certificates if needed
	 */
	if (	   test == 0
		&& cert_check_and_create (	ca_crt, ca_key,
						server_crt, server_key,
						user_id, group_id,
						quick_random) != 0)
	{
		net_destroy ();
		conf_destroy_srv ();
		module_exit ();
		return 1;
	}

	/*
	 * Initialize the network (before detaching the program from its
	 * terminal as a password may be asked for the SSL private key)
	 */
	if ((	   test == 0
		|| (	   access (ca_crt, F_OK) == 0
			&& access (server_crt, F_OK) == 0
			&& access (server_key, F_OK) == 0
		   )
	    ) && net_init (	server_crt, server_key, ca_crt,
				ca_crl, quick_random) != 0)
	{
		net_destroy ();
		conf_destroy_srv ();
		module_exit ();
		return 1;
	}

	/*
	 * Create the directory that will hold the PID file
	 */
	mkpath (pid_file, user_id, group_id);

	/*
	 * Change the group id and the user id of the process
	 */
	ret = change_group (group_id);
	if (ret != 0) {
		if (ret == -2) {
			fprintf (stderr,
				_("Unknown group name: %s\n"),
				group_id);
		}
		else {
			fprintf (stderr,
				_("Failed to change the group ID to %s: %s\n"),
				group_id, strerror (errno));
		}
		conf_destroy_srv ();
		net_destroy ();
		module_exit ();
		return 1;
	}

	ret = change_user (user_id);
	if (ret != 0) {
		if (ret == -2) {
			fprintf (stderr,
				_("Unknown user name: %s\n"),
				user_id);
		}
		else {
			fprintf (stderr,
				_("Failed to change the user ID to %s: %s\n"),
				user_id, strerror (errno));
		}
		conf_destroy_srv ();
		net_destroy ();
		module_exit ();
		return 1;
	}

	/* It was only a test of the configuration file */
	if (test == 1) {
		net_destroy ();

		/* Just try a connection to the database */
		err_msg = NULL;
		sql_handler = begin_sql (&err_msg);
		if (sql_handler == NULL) {
			if (err_msg != NULL) {
				fputs (err_msg, stderr);
				free (err_msg);
			}
			else {
				fputs (_("Failed to connect to database"),
					stderr);
			}
			fputc ('\n', stderr);
			conf_destroy_srv ();
			module_exit ();
			return 1;
		}
		end_sql ();

		conf_destroy_srv ();
		module_exit ();
		fputs (_("Fine."), stdout);
		fputc ('\n', stdout);
		return 0;
	}

	/*
	 * Go background
	 */
	if (foreground == 0) {
		if (go_daemon () != 0) {
			perror ("fork");
			conf_destroy_srv ();
			net_destroy ();
			module_exit ();
			return 1;
		}
	}
	else {
		clean_process ();
	}


	/*
	 * Open the log file
	 */
	log_file = date_format = NULL;
	facility = -1;
	ret = conf_get_param_string ("LOG_FILE", &log_file);
	ret += conf_get_param_syslog_facility ("SYSLOG", &facility);
	ret += conf_get_param_string ("DATE_FORMAT", &date_format);
#if HAVE_ASSERT_H
	assert (ret == 0);
#endif

	lwc_newLog (    PACKAGE_NAME,
			(facility != -1) ? 1 : 0,
			(facility != -1) ? (int)facility : 0,
			log_file,
			date_format);
	lwc_writeLog (LOG_INFO, _("Starting"));


	/*
	 * Create the PID file and check that an other process is not already
	 * running
	 */
	ret = write_pid_file (pid_file);
	switch (ret) {
		case -1:
			lwc_writeLog (	LOG_CRIT,
					_("Cannot create the PID file %s: %s"),
					pid_file,
					strerror (errno));
			break;
		
		case 1:
			lwc_writeLog (LOG_CRIT,
				_("An other daemon is already running"));
			break;
	}
	if (ret != 0) {
		lwc_closeLog ();
		conf_destroy_srv ();
		net_destroy ();
		module_exit ();
		return 1;
	}


	/*
	 * Database connection
	 */
	err_msg = NULL;
	sql_handler = begin_sql (&err_msg);
	if (sql_handler == NULL) {
		if (err_msg != NULL) {
			lwc_writeLog (	LOG_CRIT,
					_("Failed to connect to database: %s"),
					err_msg);
			free (err_msg);
		}
		else {
			lwc_writeLog (	LOG_CRIT,
					_("Failed to connect to database"));
		}
		lwc_closeLog ();
		close_pid_file (pid_file);
		conf_destroy_srv ();
		net_destroy ();
		module_exit ();
		return 1;
	}


	/*
	 * SSL setup
	 */
	if (quick_random == 0) {
		lwc_writeLog (LOG_INFO,
		_("Initializing the SSL layer (may take several minutes)..."));
	}
	if (net_init_server () != 0) {
		lwc_closeLog ();
		close_pid_file (pid_file);
		end_sql ();
		conf_destroy_srv ();
		net_destroy ();
		module_exit ();
		return 1;
	}
	if (quick_random == 0) {
		lwc_writeLog (LOG_INFO, _("Done"));
	}


	/*
	 * Network setup
	 */
	ret = conf_get_param_string ("IFACE_LISTEN", &iface);
	ret += conf_get_param_string ("SERVER_PORT", &server_port);
	ret += conf_get_param_string ("ADDRESS_FAMILY", &address_family);
	ret += conf_get_param_string ("ALLOW_FROM", &(s_set.allow_from));
#if HAVE_ASSERT_H
	assert (ret == 0);
#endif

	FD_ZERO (&(s_set.sock_set));
	s_set.sock_max = -1;
	if (net_server (server_port, iface, address_family,
			&(s_set.sock_set), &(s_set.sock_max)) != 0)
	{
		lwc_closeLog ();
		close_pid_file (pid_file);
		end_sql ();
		conf_destroy_srv ();
		net_destroy ();
		module_exit ();
		return 1;
	}

	/*
	 * Mask all the signal before starting the threads.  The threads will
	 * inherit this mask.  Later, below in this main() function the
	 * mask is removed which allow this main thread to manage the
	 * handling of signals
	 */
	signal_mask_all ();

	/*
	 * Load (and run) the pending workload from the database
	 */
	workload_list = load_workload_class_new ();
	if (workload_list == NULL) {
		lwc_closeLog ();
		close_pid_file (pid_file);
		end_sql ();
		net_close_sock_set (&(s_set.sock_set), &(s_set.sock_max));
		conf_destroy_srv ();
		net_destroy ();
		module_exit ();
		return 1;
	}

	if (load_workload_class_from_status_table (workload_list) != 0) {
		lwc_closeLog ();
		close_pid_file (pid_file);
		end_sql ();
		net_close_sock_set (&(s_set.sock_set), &(s_set.sock_max));
		load_workload_class_destroy (workload_list);
		conf_destroy_srv ();
		net_destroy ();
		module_exit ();
		return 1;
	}

	/*
	 * Load the workload for today (if it was
	 * not already in the database)
	 */
	schedwi_date_now (&now);
	if (	   load_workload_class_find (workload_list, now) == NULL
		&& load_workload_class_for_day (workload_list, now) != 0)
	{
		lwc_closeLog ();
		close_pid_file (pid_file);
		end_sql ();
		net_close_sock_set (&(s_set.sock_set), &(s_set.sock_max));
		load_workload_class_destroy (workload_list);
		conf_destroy_srv ();
		net_destroy ();
		module_exit ();
		return 1;
	}

	/*
	 * Start the server that will listen to network requests
	 */
	if (start_net_server (&s_set) != 0) {
		lwc_closeLog ();
		close_pid_file (pid_file);
		end_sql ();
		net_close_sock_set (&(s_set.sock_set), &(s_set.sock_max));
		load_workload_class_destroy (workload_list);
		conf_destroy_srv ();
		net_destroy ();
		module_exit ();
		return 1;
	}

	/* Start the thread which reply to user commands */
	if (commands_thread_init (workload_list) != 0) {
		pthread_cancel (net_server_thread);
		pthread_kill (net_server_thread, SIGUSR2);
		pthread_join (net_server_thread, NULL);
		lwc_closeLog ();
		close_pid_file (pid_file);
		end_sql ();
		net_close_sock_set (&(s_set.sock_set), &(s_set.sock_max));
		load_workload_class_destroy (workload_list);
		conf_destroy_srv ();
		net_destroy ();
		module_exit ();
		return 1;
	}

	/* Prepare to wait for signals */
	if (	  sigemptyset (&waitset) != 0
		|| sigaddset (&waitset, SIGHUP) != 0
		|| sigaddset (&waitset, SIGTERM) != 0
		|| sigaddset (&waitset, SIGALRM) != 0
		|| sigaddset (&waitset, SIGUSR1) != 0)
	{
		lwc_writeLog (	LOG_CRIT,
				_("Cannot prepare signals: %s"),
				strerror (errno));
		pthread_cancel (net_server_thread);
		pthread_kill (net_server_thread, SIGUSR2);
		pthread_join (net_server_thread, NULL);
		lwc_closeLog ();
		close_pid_file (pid_file);
		end_sql ();
		net_close_sock_set (&(s_set.sock_set), &(s_set.sock_max));
		commands_thread_destroy ();
		load_workload_class_destroy (workload_list);
		conf_destroy_srv ();
		net_destroy ();
		module_exit ();
		return 1;
	}

	/* Wake-up at midnight */
	prepare_alarm ();

	/*
	 * Main loop
	 */
	while (1) {

		/* Wait for signals */
		ret = sigwait (&waitset, &sig);
		if (ret != 0) {
			lwc_writeLog (	LOG_CRIT,
					_("Failed to wait for signals: %s"),
					strerror (errno));
			pthread_cancel (net_server_thread);
			pthread_kill (net_server_thread, SIGUSR2);
			pthread_join (net_server_thread, NULL);
			lwc_closeLog ();
			close_pid_file (pid_file);
			end_sql ();
			net_close_sock_set (	&(s_set.sock_set),
						&(s_set.sock_max));
			commands_thread_destroy ();
			load_workload_class_destroy (workload_list);
			conf_destroy_srv ();
			net_destroy ();
			module_exit ();
			return 1;
		}
		switch (sig) {
			case SIGALRM:
				sigalrm_action ();
				break;

			case SIGUSR1:
				load_workload_class_print (workload_list);
				break;

			case SIGHUP:
				sighup_action ();
				break;

			case SIGTERM:
				/* Stop listening for agent requests */
				pthread_cancel (net_server_thread);
				pthread_kill (net_server_thread, SIGUSR2);
				pthread_join (net_server_thread, NULL);
				net_close_sock_set (	&(s_set.sock_set),
							&(s_set.sock_max));

				/* Stop user commands */
				commands_thread_destroy ();

				/* Stop all the threads */
				load_workload_class_destroy (workload_list);

				/* Disconnect from the database */
				end_sql ();

				/* Destroy the configuration file object */
				conf_destroy_srv ();

				/* Clean the network layer */
				net_destroy ();

				/* Destroy the module engine */
				module_exit ();

				/* Close the log file */
				lwc_writeLog (LOG_INFO, _("Exiting"));
				lwc_closeLog ();

				/* Finaly, remove the pid file */
				close_pid_file (pid_file);
				return 0;

			default:
				lwc_writeLog (	LOG_WARNING,
				_("Internal error: unexpected signal %d"),
						sig);
		}
	}

	return 0;
}

/*------------------------======= End Of File =======------------------------*/
