/* Schedwi
   Copyright (C) 2013 Herve Quatremain

   This file is part of Schedwi.

   Schedwi is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 3 of the License, or
   (at your option) any later version.

   Schedwi is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

/* schedwi_system.c -- system() with environment variables */

#include <schedwi.h>

#if HAVE_SYS_TYPES_H
#include <sys/types.h>
#endif

#if STDC_HEADERS
#include <stdlib.h>
#include <string.h>
#else
#if HAVE_STDLIB_H
#include <stdlib.h>
#endif
#if HAVE_STRING_H
#include <string.h>
#endif
#endif

#if HAVE_STDIO_H
#include <stdio.h>
#endif

#if HAVE_SYS_WAIT_H
# include <sys/wait.h>
#endif
#ifndef WEXITSTATUS
# define WEXITSTATUS(stat_val) ((unsigned)(stat_val) >> 8)
#endif
#ifndef WIFEXITED
# define WIFEXITED(stat_val) (((stat_val) & 255) == 0)
#endif

#if HAVE_UNISTD_H
#include <unistd.h>
#endif

#if HAVE_SIGNAL_H
#include <signal.h>
#endif

#include <lib_functions.h>
#include <schedwi_system.h>
#include <schedwi_jobtree.h>


#define SHELL_PATH "/bin/sh"	/* Path of the shell.  */
#define SHELL_NAME "sh"		/* Name to give it.  */


/*
 * Run the provided command (in `line').  The environment variables in the
 * `variables' array (that contains `len' variables) are set.  The
 * following variables are also automatically set:
 *     SCHEDWI_JOBPATH
 *     SCHEDWI_JOBID
 *     SCHEDWI_START_TIME_EPOCH
 *     SCHEDWI_START_TIME (format HH:MM)
 *     SCHEDWI_TIME_LIMIT_EPOCH (if a start time limit is set for the job)
 *     SCHEDWI_TIME_LIMIT (if a start time limit is set for the job)
 *
 * Return:
 *    -1 --> System error
 *   127 --> Failed to start the command
 *   300 --> The command has not exited normally (killed, ...)
 * other --> Command return code
 */
int
schedwi_system (const char *line,
		int workload, const schedwi_jobtree_node_ptr job,
		const char * const *variables, size_t len)
{
#if HAVE_PID_T
	pid_t p;
#else
	int p;
#endif
	sigset_t mask, omask;
	struct sigaction sa;
	int status, i;
	char **new_env;
	schedwi_time t;
	size_t j;

	if (line == NULL || line[0] == '\0') {
		return 0;
	}

	/* Build the environment variables */
	new_env = (char **) malloc (sizeof (char *) * (len + 8));
	if (new_env == NULL) {
		return -1;
	}

	i = 0;
	/* SCHEDWI_JOBPATH */
	new_env[i] = (char *) malloc (schedwi_strlen (job->path) + 30);
	if (new_env[i] == NULL) {
		while (--i >= 0) {
			free (new_env[i]);
		}
		free (new_env);
		return -1;
	}
	strcpy (new_env[i], "SCHEDWI_JOBPATH=");
	strcat (new_env[i], job->path);
	i++;

	/* SCHEDWI_JOBID */
	new_env[i] = (char *) malloc (100);
	if (new_env[i] == NULL) {
		while (--i >= 0) {
			free (new_env[i]);
		}
		free (new_env);
		return -1;
	}
	sprintf (new_env[i], "SCHEDWI_JOBID=%d_%lld", workload, job->id);
	i++;

	/* SCHEDWI_START_TIME_EPOCH */
	new_env[i] = schedwi_time_strftime ("SCHEDWI_START_TIME_EPOCH=%s",
					    job->run_time);
	if (new_env[i] == NULL) {
		while (--i >= 0) {
			free (new_env[i]);
		}
		free (new_env);
		return -1;
	}
	i++;

	/* SCHEDWI_START_TIME */
	new_env[i] = schedwi_time_strftime ("SCHEDWI_START_TIME=%H:%M",
					    job->run_time);
	if (new_env[i] == NULL) {
		while (--i >= 0) {
			free (new_env[i]);
		}
		free (new_env);
		return -1;
	}
	i++;

	/* SCHEDWI_TIME_LIMIT_EPOCH and SCHEDWI_TIME_LIMIT */
	if (job->start_limit > 0) {
		if (job->run_time > 0) {
			t = job->run_time;
		}
		else {
			if (job->parent == NULL) {
				/* Should not be possible */
				t = job->run_time;
			}
			else {
				t = job->parent->start_time; 
			}
		}
		new_env[i] = schedwi_time_strftime (
			"SCHEDWI_TIME_LIMIT_EPOCH=%s",
			schedwi_time_add_seconds (t, job->start_limit));
		if (new_env[i] == NULL) {
			while (--i >= 0) {
				free (new_env[i]);
			}
			free (new_env);
			return -1;
		}
		i++;

		new_env[i] = schedwi_time_strftime (
			"SCHEDWI_TIME_LIMIT=%H:%M",
			schedwi_time_add_seconds (t, job->start_limit));
		if (new_env[i] == NULL) {
			while (--i >= 0) {
				free (new_env[i]);
			}
			free (new_env);
			return -1;
		}
		i++;
	}

	/* Copy the variables passed as parameter */
	if (variables != NULL && len > 0) {
		for (j = 0; j < len; j++) {
			new_env[i] = (char *) malloc (
					schedwi_strlen (variables[j]) + 1);
			if (new_env[i] == NULL) {
				while (--i >= 0) {
					free (new_env[i]);
				}
				free (new_env);
				return -1;
			}
			strcpy (new_env[i], variables[j]);
			i++;
		}
	}
	new_env[i] = NULL;

	if (	   sigemptyset (&mask) != 0
		|| sigaddset (&mask,  SIGCHLD) != 0
		|| sigprocmask (SIG_BLOCK, &mask, &omask) != 0)
	{
		while (--i >= 0) {
			free (new_env[i]);
		}
		free (new_env);
		return -1;
	}

	p = fork ();
	if (p < 0) {
		sigprocmask (SIG_SETMASK, &omask, NULL);
		while (--i >= 0) {
			free (new_env[i]);
		}
		free (new_env);
		return -1;
	}

	if (p == 0) {
		/* Child */
		const char *new_argv[4];

		new_argv[0] = SHELL_NAME;
		new_argv[1] = "-c";
		new_argv[2] = line;
		new_argv[3] = NULL;

		/* Close all files */
		for (i = getdtablesize () - 1; i >= 0; i--) {
			close (i);
		}

		/* Clear the environment */
#ifndef CYGWIN_IN_USE
#if HAVE_CLEARENV
		clearenv ();
#elif _GNU_SOURCE
		environ = NULL;
#endif
#endif

		/* Clear the signal actions and mask */
		schedwi_memset (&sa, 0, sizeof (struct sigaction));
		sa.sa_handler = SIG_DFL;
		sigemptyset (&sa.sa_mask);
		for (i = 0; i < 128; i++) {
			sigaction (i, &sa, NULL);
		}
		sigfillset (&mask);
		sigprocmask (SIG_UNBLOCK, &mask, NULL);

		execve (SHELL_PATH,	(char *const *) new_argv,
					(char *const *) new_env);

		exit (127);
	}

	while (--i >= 0) {
		free (new_env[i]);
	}
	free (new_env);

	/* Parent */
	if (waitpid (p, &status, 0) != p) {
		sigprocmask (SIG_SETMASK, &omask, NULL);
		return -1;
	}

	sigprocmask (SIG_SETMASK, &omask, NULL);

	if (WIFEXITED (status) != 0) {
		return WEXITSTATUS (status);
	}
	else {
		return 300;
	}
}

/*-----------------============== End Of File ==============-----------------*/
