/* Schedwi
   Copyright (C) 2007-2013 Herve Quatremain

   This file is part of Schedwi.

   Schedwi is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 3 of the License, or
   (at your option) any later version.

   Schedwi is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

/* schedwi_jobtree.c -- Function to manage the job/jobset tree */

#include <schedwi.h>

#if STDC_HEADERS
#include <stdlib.h>
#include <string.h>
#else
#if HAVE_STDLIB_H
#include <stdlib.h>
#endif
#if HAVE_STRING_H
#include <string.h>
#endif
#endif

#if HAVE_ERRNO_H
#include <errno.h>
#endif
#ifndef errno
extern int errno;
#endif

#if HAVE_ASSERT_H
#include <assert.h>
#endif

#include <lwc_log.h>
#include <sql_job.h>
#include <sql_hierarchy.h>
#include <sql_link.h>
#include <sql_status.h>
#include <sql_constraint_file.h>
#include <utils.h>
#include <job_status.h>
#include <startjob.h>
#include <stopjob.h>
#include <lib_functions.h>
#include <module.h>
#include <schedwi_jobtree_priv.h>
#include <check_waiting_can_start.h>

#define TABSTOP 4

/*
 * Error callback function for the sql_job_get_start_limit(),
 * sql_job_get_max_duration(), sql_job_get_retries(),
 * sql_job_get_retries_interval(), sql_job_get_manual(),
 * sql_job_get_manual_command() and sql_status_get () functions
 */
static void
sql_get_error_logger (void *data, const char *msg, int err_code)
{
	if (msg != NULL) {
		lwc_writeLog (LOG_ERR, msg);
	}
	else {
		lwc_writeLog (LOG_ERR,
		_("Database error while trying to retrieve a job parameter"));
	}
}


/*
 * Retrieve the parameters of a job from the database
 *
 * Return:
 *   0 --> No error
 *  -1 --> Error (an error message has already been logged by this function)
 */
static int
set_schedwi_jobtree_node (	schedwi_jobtree_node_ptr ptr,
				int workload_date,
				schedwi_date job_date,
				int job_time,
				lwc_LL *hierarchy_list)
{
	short int time_limit;
	int current_status;
	long int start_time;
	lwc_LL *rows;
	row_item_t *row;
	int n, i, j;
	char *manual_command;
	char *filename;
	schedwi_jobtree_file_ptr constraint_files;

#if HAVE_ASSERT_H
	assert (ptr != NULL && hierarchy_list != NULL);
#endif

	/* Retrieve parameters from the database */
	if (sql_job_get_start_limit (	workload_date, hierarchy_list,
					&time_limit,
					sql_get_error_logger, NULL) != 0)
	{
		return -1;
	}
	ptr->start_limit = time_limit * 60; /* The limit in DB is in min */

	if (sql_job_get_max_duration (	workload_date, hierarchy_list,
					&time_limit,
					sql_get_error_logger, NULL) != 0)
	{
	return -1;
	}
	ptr->max_duration = time_limit * 60; /* The limit in DB is in min */

	if (sql_job_get_retries (	workload_date, hierarchy_list,
					&(ptr->retries),
					sql_get_error_logger, NULL) != 0)
	{
		return -1;
	}

	if (sql_job_get_retries_interval (	workload_date,
						hierarchy_list,
						&time_limit,
						sql_get_error_logger,
						NULL) != 0)
	{
		return -1;
	}
	ptr->retries_interval = time_limit * 60; /* The value in DB is in min */

	if (sql_job_get_manual (workload_date, hierarchy_list, &(ptr->manual),
				sql_get_error_logger, NULL) != 0)
	{
		return -1;
	}
	if (ptr->manual == 0) {
		if (ptr->manual_command != NULL) {
			free (ptr->manual_command);
		}
		ptr->manual_command = NULL;
	}
	else {
		if (sql_job_get_manual_command (workload_date, hierarchy_list,
					&manual_command,
					sql_get_error_logger, NULL) != 0)
		{
			return -1;
		}
		if (ptr->manual_command != NULL) {
			free (ptr->manual_command);
		}
		ptr->manual_command = manual_command;
	}

	if (sql_status_get (	workload_date, ptr->id, &current_status,
				&start_time, &(ptr->retry_num),
				&(ptr->wait_reason), NULL,
				sql_get_error_logger, NULL) != 0)
	{
		return -1;
	}

	ptr->status = (current_status == 0)
			 	? JOB_STATUS_STATE_WAITING
		  		: job_status_state_int2status (current_status);
	if (current_status == 0) {
		/*
		 * It's a new job (waiting). Add an entry in the job_status
		 * table
		 */
		ptr->wait_reason = WAIT_REASON_MASK_TIME;
		job_status_set (ptr->id, ptr->path, workload_date,
				JOB_STATUS_STATE_WAITING, 0, 0, 0,
				ptr->wait_reason, NULL);
	}
	ptr->start_time = (ptr->status == JOB_STATUS_STATE_RUNNING)
				? (schedwi_time) start_time : 0;
	ptr->run_time   = schedwi_time_convert (job_date, job_time);


	/* Retrieve contraint files*/
	constraint_files = NULL;
	j = 0;
	if (sql_constraint_file_get (	workload_date, &rows, ptr->id,
					sql_get_error_logger, NULL) != 0)
	{
		return -1;
	}
	n = lwc_getNumNode (rows);
    if (n != 0) {
        constraint_files = (schedwi_jobtree_file_ptr) malloc (
                                            sizeof (schedwi_jobtree_file) * n);
        if (constraint_files == NULL) {
            lwc_delLL (rows, (void (*)(const void *)) sql_free_row);
            lwc_writeLog (LOG_CRIT, _("Memory allocation error"));
            return -1;
        }
        for (i = 0;
             i < n && (row = (row_item_t *) lwc_delStartLL (rows)) != NULL;
             i++)
        {
            /*
             * row[0] --> Job ID
             * row[1] --> Host ID
             * row[2] --> File name
             * row[3] --> Test on file exists (1) or not (0)
             */
            filename = row[2].value_string;
            if (filename != NULL && filename[0] != '\0') {
                constraint_files[j].file = (char *)malloc(row[2].len +1);
                if (constraint_files[j].file == NULL) {
                    while (--j >= 0) {
                        free (constraint_files[j].file);
                    }
                    free (constraint_files);
                    sql_free_row (row);
                    lwc_delLL (rows, (void (*)(const void *)) sql_free_row);
                    lwc_writeLog (LOG_CRIT, _("Memory allocation error"));
                    return -1;
                }
                strcpy (constraint_files[j].file, filename);
                constraint_files[j].host_id =
                           (unsigned long long int)sql_row_item2ll (&(row[1]));
                constraint_files[j].exist =
                           (char)sql_row_item2ll (&(row[3]));
                j++;
            }
            sql_free_row (row);
        }
    }
	if (ptr->constraint_files != NULL) {
		i = ptr->num_constraint_files;
		while (--i >= 0) {
			free ((ptr->constraint_files)[i].file);
		}
		free (ptr->constraint_files);
	}
	ptr->constraint_files = constraint_files;
	ptr->num_constraint_files = j;
	lwc_delLL (rows, (void (*)(const void *)) sql_free_row);

	return 0;
}


/*
 * Create a new node
 *
 * Return:
 *   The new node (to be freed by the caller by free_schedwi_jobtree_node()) or
 *   NULL in case of error (an error message has been logged by
 *	  using lwc_writeLog())
 */
static schedwi_jobtree_node_ptr
new_schedwi_jobtree_node (	unsigned long long int id,
				char *path,
				int workload_date,
				char node_type,
				schedwi_date job_date,
				int job_time,
				lwc_LL *hierarchy_list)
{
	schedwi_jobtree_node_ptr ptr;

#if HAVE_ASSERT_H
	assert (path != NULL && hierarchy_list != NULL);
#endif

	/* Allocate memory for the new node */
	ptr = (schedwi_jobtree_node_ptr)malloc (sizeof (schedwi_jobtree_node));
	if (ptr == NULL) {
		lwc_writeLog (LOG_CRIT, _("Memory allocation error"));
		return NULL;
	}
	ptr->parent     = NULL;
	ptr->children   = NULL;
	ptr->next       = NULL;
	ptr->prev       = NULL;
	ptr->id         = id;
	ptr->path       = path;
	ptr->start_day  = job_date;
	ptr->node_type  = node_type;
	ptr->links      = NULL;
	ptr->links_in   = NULL;
	ptr->constraint_files = NULL;
	ptr->num_constraint_files = 0;
	ptr->manual_command = NULL;

	if (set_schedwi_jobtree_node (	ptr, workload_date, job_date, job_time,
					hierarchy_list) != 0)
	{
		free (ptr);
		return NULL;
	}

	return ptr;
}


/*
 * Free a link list
 */
static void
free_schedwi_jobtree_link_list (schedwi_jobtree_link_ptr ptr)
{
	schedwi_jobtree_link_ptr tmp;

	while (ptr != NULL) {
		tmp = ptr->next;
		free (ptr);
		ptr = tmp;
	}
}


/*
 * Free a single node
 */
static void
free_schedwi_jobtree_node (schedwi_jobtree_node_ptr ptr)
{
	int i;

	if (ptr != NULL) {
		if (ptr->path != NULL) {
			free (ptr->path);
		}
		if (ptr->manual_command != NULL) {
			free (ptr->manual_command);
		}
		if (ptr->constraint_files != NULL) {
			i = ptr->num_constraint_files;
			while (--i >= 0) {
				free ((ptr->constraint_files)[i].file);
			}
			free (ptr->constraint_files);
		}
		free_schedwi_jobtree_link_list (ptr->links);
		free_schedwi_jobtree_link_list (ptr->links_in);
		free (ptr);
	}
}


/*
 * Recursively free a node tree
 */
static void
free_jobtree_all_nodes (schedwi_jobtree_node_ptr ptr)
{
	if (ptr != NULL) {
		free_jobtree_all_nodes (ptr->children);
		free_jobtree_all_nodes (ptr->next);
		free_schedwi_jobtree_node (ptr);
	}
}


/*
 * Update a node
 * Return:
 *
 *   0 --> No error
 *  -1 --> Error (an error message has already been logged by this function)
 */
static int
update_jobtree_node (schedwi_jobtree_node_ptr ptr, int workload_date)
{
	lwc_LL *hierarchy_list;
	schedwi_jobtree_node_ptr p;
	int job_time;

#if HAVE_ASSERT_H
	assert (ptr != NULL);
#endif

	/*
	 * Initialize the hierarchy working list.
	 * This list will be used to retrieve the job parameters from the
	 * database.
	 */
	hierarchy_list = lwc_newLL ();
	if (hierarchy_list == NULL) {
		lwc_writeLog (LOG_CRIT, _("Memory allocation error"));
		return -1;
	}

	for (p = ptr; p != NULL; p = p->parent) {
		if (lwc_addEndLL (hierarchy_list, (&p->id)) != 0) {
			lwc_delLL (hierarchy_list, NULL);
			lwc_writeLog (LOG_CRIT, _("Memory allocation error"));
			return -1;
		}
	}

	/* Retrieve the run time from the database */
	if (sql_job_get_start_time (	workload_date, ptr->id, &job_time,
					sql_get_error_logger, NULL) != 0)
	{
		lwc_delLL (hierarchy_list, NULL);
		return -1;
	}

	/* Update the job parameters */
	if (set_schedwi_jobtree_node (	ptr, workload_date, ptr->start_day,
					job_time, hierarchy_list) != 0)
	{
		lwc_delLL (hierarchy_list, NULL);
		return -1;
	}

	lwc_delLL (hierarchy_list, NULL);
	return 0;
}


/*
 * Comparison function used by qsort() to sort the job/jobset list by ID
 */
static int
compar_job_id (const void *a, const void *b)
{
	jobtree_jobs_ptr ptr1 = (jobtree_jobs_ptr) a;
	jobtree_jobs_ptr ptr2 = (jobtree_jobs_ptr) b;

	return ptr1->id - ptr2->id;
}


/*
 * Comparison function used by bsearch() to search a job/jobset list by ID
 */
static int
search_job_id (const void *key, const void *str)
{
	unsigned long long int *id = (unsigned long long int *)key;
	jobtree_jobs_ptr ptr = (jobtree_jobs_ptr) str;

	return *id - ptr->id;
}


/*
 * Retrieve a job/jobset node from the tree given its ID
 * Warning: the pthread mutex must be locked before calling this function
 * (by pthread_mutex_lock (&(tree->mutex)) )
 *
 * Return:
 *   The node or
 *   NULL if not found
 */
schedwi_jobtree_node_ptr
jobtree_find (jobtree_ptr tree, unsigned long long int id)
{
	jobtree_jobs_ptr ptr;

	if (tree == NULL) {
		return NULL;
	}
	ptr =  bsearch (&id, tree->jobs, tree->nb_jobs,
			sizeof (jobtree_jobs), search_job_id);
	if (ptr == NULL) {
		return NULL;
	}
	return ptr->node;
}


/*
 * Recursively walk through the tree and fill the provided index array
 */
static void
build_index_recur (   schedwi_jobtree_node_ptr top,
			unsigned int *idx_list,
			jobtree_jobs_ptr list)
{
	while (top != NULL) {
		list[*idx_list].id = top->id;
		list[*idx_list].node = top;
		(*idx_list)++;
		build_index_recur (top->children, idx_list, list);
		top = top->next;
	}
}


/*
 * Build the index (ordered array) of all the jobs/jobsets in the tree
 * The last element of the array has an ID of 0 and node is NULL.
 *
 * Return:
 *   0 --> No error (the array has been allocated by this function and
 *	   must be freed by the caller by free())
 *  -1 --> Memory allocation error
 */
static int
build_index ( schedwi_jobtree_node_ptr top, unsigned int nb,
		jobtree_jobs_ptr *list)
{
	jobtree_jobs_ptr ptr_list;
	unsigned int idx_list;

	/*
	 * Allocate memory for the arrays
	 */
	ptr_list = (jobtree_jobs_ptr) malloc (	  sizeof (jobtree_jobs)
						* (nb + 1));
	if (ptr_list == NULL) {
		return -1;
	}

	/*
	 * Fill the array
	 */
	idx_list = 0;
	build_index_recur (top, &idx_list, ptr_list);
	ptr_list[idx_list].id = 0;
	ptr_list[idx_list].node = NULL;

	/*
	 * Sort the array by ID
	 */
	qsort (ptr_list, idx_list, sizeof (jobtree_jobs), compar_job_id);

	*list = ptr_list;
	return 0;
}


/*
 * Error callback function for the sql_job_today() and sql_job_tomorrow()
 * functions
 */
static void
sql_list_error_logger (void *data, const char *msg, int err_code)
{
	if (msg != NULL) {
		lwc_writeLog (LOG_ERR, msg);
	}
	else {
		lwc_writeLog (LOG_ERR,
		_("Database error while trying to retrieve the job list"));
	}
}


/*
 * Recursively build the job/jobset tree
 *
 * Return:
 *   0 --> No error.  children contains the tree and must be freed by
 *	   free_jobtree_all_nodes(). nb_jobs contains the number of
 *	   jobs/jobsets in the returned tree.
 *  -1 --> Memory allocation error.  An error message has been logged by
 *	   using lwc_writeLog().
 *  -2 --> Database error.  An error message has been logged by
 *	   using lwc_writeLog().
 */
static int
build_jobtree_recur (	schedwi_jobtree_node_ptr jobset,
			unsigned long long int jobset_id,
			const char *jobset_path,
			unsigned int jobset_path_length,
			schedwi_date jobset_date,
			int workload_date,
			int jobset_time,
			calendar_list_t_ptr calendars,
			lwc_LL *hierarchy_list,
			schedwi_jobtree_node_ptr *children,
			unsigned int *nb_jobs)
{
    lwc_LL *job_list;
    row_item_t *row;
    unsigned int length;
    unsigned long long int *id, i;
    char jtype;
    int stime, ret;
    char *calendars_for_today, *calendars_for_tomorrow, *path;
    schedwi_date date_tomorrow;
    schedwi_jobtree_node_ptr ptr, nxt;

#if HAVE_ASSERT_H
    assert (   jobset_path != NULL && calendars != NULL
            && hierarchy_list != NULL && children != NULL && nb_jobs != NULL);
#endif

    ptr = nxt = NULL;

    /*
     * Build the tree with the jobs and jobsets that must run today
     */

    /* Get the calendar ID list for today */
    if (get_calendar_list_for_the_day (calendars, jobset_date,
                                       &calendars_for_today) != 0)
    {
        return -1;
    }

    /* Get the job/jobset list */
    ret = sql_job_today (workload_date,
                         &job_list,
                         jobset_id,
                         calendars_for_today,
                         jobset_time,
                         sql_list_error_logger,
                         NULL);
    free (calendars_for_today);
    if (ret != 0) {
        return ret;
    }

    /* Build the tree from the list */
    while ((row = (row_item_t *) lwc_delStartLL (job_list)) != NULL) {
        /*
         * row[0] --> ID
         * row[1] --> Type (0: jobset and 1: job)
         * row[2] --> Start time
         * row[3] --> Job/Jobset name
         */

        /* Get the job/jobset id */
        i = (unsigned long long int) sql_row_item2ll (&(row[0]));

        /* Build the full path name of the job/jobset */
        length = jobset_path_length + row[3].len + 1;
        path = (char *) malloc (length + 1);
        if (path == NULL) {
            sql_free_row (row);
            lwc_delLL (job_list, (void (*)(const void *)) sql_free_row);
            free_jobtree_all_nodes (nxt);
            lwc_writeLog (LOG_CRIT, _("Memory allocation error"));
            return -1;
        }
        if (i == 1) { /* Root jobset */
                path[0] = '/';
                path[1] = '\0';
                length = 1;
        }
        else {
                if (jobset_id == 1) {
                        path[0] = '/';
                        strcpy (path + 1, row[3].value_string);
                        length = row[3].len + 1;
                }
                else {
                        strcpy (path, jobset_path);
                        path[jobset_path_length] = '/';
                        strcpy (path + jobset_path_length + 1,
                                row[3].value_string);
                }
        }

        /* Add the new job/jobset ID to the hierarchy list */
        id = (unsigned long long int *) malloc (sizeof(unsigned long long int));
        if (id == NULL || lwc_addStartLL (hierarchy_list, id) != 0) {
            if (id != NULL) {
                free (id);
            }
            free (path);
            sql_free_row (row);
            lwc_delLL (job_list, (void (*)(const void *)) sql_free_row);
            free_jobtree_all_nodes (nxt);
            lwc_writeLog (LOG_CRIT, _("Memory allocation error"));
            return -1;
        }
        *id = i;
        jtype = (char) sql_row_item2ll (&(row[1]));
        stime = (int) sql_row_item2ll (&(row[2]));
        sql_free_row (row);

        /* Create the new job object */
        ptr = new_schedwi_jobtree_node (*id,
                                        path,
                                        workload_date,
                                        jtype,
                                        jobset_date,
                                        stime,
                                        hierarchy_list);
        if (ptr == NULL) {
            free (id);
            lwc_delStartLL (hierarchy_list);
            free (path);
            lwc_delLL (job_list, (void (*)(const void *)) sql_free_row);
            free_jobtree_all_nodes (nxt);
            return -1;
        }

        /* Link the new job/jobset */
        (*nb_jobs)++;
        ptr->parent = jobset;
        ptr->next = nxt;
        if (nxt != NULL) {
            nxt->prev = ptr;
        }
        nxt = ptr;

        /* If it's a jobset, recursively build the job tree */
        if (ptr->node_type == 0) {
            ret = build_jobtree_recur (ptr,
                                       *id,
                                       path,
                                       length,
                                       jobset_date,
                                       workload_date,
                                       stime,
                                       calendars,
                                       hierarchy_list,
                                       &(ptr->children),
                                       nb_jobs);
            if (ret != 0) {
                free (id);
                lwc_delStartLL (hierarchy_list);
                lwc_delLL (job_list, (void (*)(const void *)) sql_free_row);
                free_jobtree_all_nodes (nxt);
                return ret;
            }
        }

        /* Remove the job/jobset ID from the hierarchy list */
        free (id);
        lwc_delStartLL (hierarchy_list);
    }
    lwc_delLL (job_list, NULL);


    /*
     * Complete the tree with the jobs and jobsets that must run tomorrow
     */

    /* Get the calendar ID list for tomorrow */
    schedwi_date_add_days (jobset_date, &date_tomorrow, 1);
    if (get_calendar_list_for_the_day (calendars, date_tomorrow,
                                       &calendars_for_tomorrow) != 0)
    {
        free_jobtree_all_nodes (nxt);
        return -1;
    }

    /* Get the job/jobset list */
    ret = sql_job_tomorrow (workload_date,
                            &job_list,
                            jobset_id,
                            calendars_for_tomorrow,
                            jobset_time,
                            sql_list_error_logger,
                            NULL);
    free (calendars_for_tomorrow);
    if (ret != 0) {
        free_jobtree_all_nodes (nxt);
        return ret;
    }

    /* Build the tree from the list */
    while ((row = (row_item_t *) lwc_delStartLL (job_list)) != NULL) {
        /*
         * row[0] --> ID
         * row[1] --> Type (0: jobset and 1: job)
         * row[2] --> Start time
         * row[3] --> Job/Jobset name
         */

        /* Get the job/jobset id */
        i = (unsigned long long int) sql_row_item2ll (&(row[0]));

        /* Build the full path name of the job/jobset */
        length = jobset_path_length + row[3].len + 1;
        path = (char *) malloc (length + 1);
        if (path == NULL) {
            sql_free_row (row);
            lwc_delLL (job_list, (void (*)(const void *)) sql_free_row);
            free_jobtree_all_nodes (nxt);
            lwc_writeLog (LOG_CRIT, _("Memory allocation error"));
            return -1;
        }
        if (i == 1) { /* Root jobset */
                path[0] = '/';
                path[1] = '\0';
                length = 1;
        }
        else {
                if (jobset_id == 1) {
                        path[0] = '/';
                        strcpy (path + 1, row[3].value_string);
                        length = row[3].len + 1;
                }
                else {
                        strcpy (path, jobset_path);
                        path[jobset_path_length] = '/';
                        strcpy (path + jobset_path_length + 1,
                                row[3].value_string);
                }
        }

        /* Add the new job/jobset ID to the hierarchy list */
        id = (unsigned long long int *) malloc (sizeof(unsigned long long int));
        if (id == NULL || lwc_addStartLL (hierarchy_list, id) != 0) {
            if (id != NULL) {
                free (id);
            }
            free (path);
            sql_free_row (row);
            lwc_delLL (job_list, (void (*)(const void *)) sql_free_row);
            free_jobtree_all_nodes (nxt);
            lwc_writeLog (LOG_CRIT, _("Memory allocation error"));
            return -1;
        }
        *id = i;
        jtype = (char) sql_row_item2ll (&(row[1]));
        stime = (int) sql_row_item2ll (&(row[2]));
        sql_free_row (row);

        /* Create the new job object */
        ptr = new_schedwi_jobtree_node (*id,
                                        path,
                                        workload_date,
                                        jtype,
                                        date_tomorrow,
                                        stime,
                                        hierarchy_list);
        if (ptr == NULL) {
            free (id);
            lwc_delStartLL (hierarchy_list);
            free (path);
            lwc_delLL (job_list, (void (*)(const void *)) sql_free_row);
            free_jobtree_all_nodes (nxt);
            return -1;
        }

        /* Link the new job/jobset */
        (*nb_jobs)++;
        ptr->parent = jobset;
        ptr->next = nxt;
        if (nxt != NULL) {
            nxt->prev = ptr;
        }
        nxt = ptr;

        /* If it's a jobset, recursively build the job tree */
        if (ptr->node_type == 0) {
            ret = build_jobtree_recur (ptr,
                                       *id,
                                       path,
                                       length,
                                       date_tomorrow,
                                       workload_date,
                                       stime,
                                       calendars,
                                       hierarchy_list,
                                       &(ptr->children),
                                       nb_jobs);
            if (ret != 0) {
                free (id);
                lwc_delStartLL (hierarchy_list);
                lwc_delLL (job_list, (void (*)(const void *)) sql_free_row);
                free_jobtree_all_nodes (nxt);
                return ret;
            }
        }

        /* Remove the job/jobset ID from the hierarchy list */
        free (id);
        lwc_delStartLL (hierarchy_list);
    }
    lwc_delLL (job_list, NULL);

    *children = ptr;
    return 0;
}


/*
 * Build the job/jobset tree
 *
 * Return:
 *   0 --> No error.  top contains the tree and must be freed by
 *	   free_jobtree_all_nodes(). nb_jobs contains the number of
 *	   jobs/jobsets in the returned tree.
 *  -1 --> Memory allocation error.  An error message has been logged by
 *	   using lwc_writeLog().
 *  -2 --> Database error.  An error message has been logged by
 *	   using lwc_writeLog().
 */
static int
build_jobtree (	schedwi_date jobset_date,
		calendar_list_t_ptr calendars,
		schedwi_jobtree_node_ptr *top,
		unsigned int *nb_jobs)
{
	lwc_LL *hierarchy_list;
	unsigned long long int *id;
	int ret, workload_date;

	/*
	 * Initialize the hierarchy working list with the root job (ID=1).
	 * This list will be used to retrieve the job parameters from the
	 * database.
	 */
	hierarchy_list = lwc_newLL ();
	if (hierarchy_list == NULL) {
		lwc_writeLog (LOG_CRIT, _("Memory allocation error"));
		return -1;
	}
	id = (unsigned long long int *) malloc (sizeof(unsigned long long int));
	if (id == NULL || lwc_addEndLL (hierarchy_list, id) != 0) {
		if (id != NULL) {
			free (id);
		}
		lwc_delLL (hierarchy_list, (void (*)(const void *))free);
		lwc_writeLog (LOG_CRIT, _("Memory allocation error"));
		return -1;
	}
	*id = 1;

	/* Build the job/jobset tree */
	*nb_jobs = 0;
	workload_date = schedwi_date_to_int (jobset_date);
	ret = build_jobtree_recur (	NULL, 0, "", 0, jobset_date,
					workload_date, 0, calendars,
					hierarchy_list,
					top, nb_jobs);
	lwc_delLL (hierarchy_list, (void (*)(const void *))free);
	return ret;
}


/*
 * Error callback function for the sql_link_get_list() function
 */
static void
sql_link_get_list_logger (void *data, const char *msg, int err_code)
{
	if (msg != NULL) {
		lwc_writeLog (LOG_ERR, msg);
	}
	else {
		lwc_writeLog (LOG_ERR,
		_("Database error while trying to retrieve the job links"));
	}
}


/*
 * Add the links between jobs/jobsets in the tree
 *
 * Return:
 *   0 --> No error
 *  -1 --> Memory allocation error.  An error message has been logged by
 *	   using lwc_writeLog().
 *  -2 --> Database error.  An error message has been logged by
 *	   using lwc_writeLog().
 */
static int
build_links (jobtree_ptr ptr)
{
	unsigned long long int id;
	unsigned int i, n, idx;
	int ret;
	char *job_id_list;
	lwc_LL *rows;
	row_item_t *row;
	schedwi_jobtree_node_ptr node_src, node_dst;
	schedwi_jobtree_link_ptr new_link_out, new_link_in;


	if (ptr == NULL) {
		return 0;
	}

	job_id_list = (char *)malloc (ptr->nb_jobs * 25 + 1);
	if (job_id_list == NULL) {
		lwc_writeLog (LOG_CRIT, _("Memory allocation error"));
		return -1;
	}

	/* Build the job ID string */
	for (i = idx = 0; i < ptr->nb_jobs; i++) {
		n = copy_ulltostr ((ptr->jobs)[i].id, job_id_list + idx);
		job_id_list[idx + n] = ',';
		idx += n + 1;
	}

	/* Remove the last `,' */
	if (idx == 0) {
		job_id_list[0] = '\0';
	}
	else {
		job_id_list[idx - 1] = '\0';
	}

	/* Retrieve the links from the database */
	ret = sql_link_get_list (&rows,
				schedwi_date_to_int (ptr->date),
				job_id_list,
				sql_link_get_list_logger, NULL);
	free (job_id_list);
	if (ret != 0) {
		return ret;
	}

	node_src = NULL;
	while ((row = (row_item_t *) lwc_delStartLL (rows)) != NULL) {
		/*
		 * row[0] --> Source job ID
		 * row[1] --> Destination job ID
		 * row[2] --> Required status
		 */

		/* Find the source job/jobset in the tree */
		id = (unsigned long long int) sql_row_item2ll (&(row[0]));
		if (node_src == NULL || node_src->id != id) {
			node_src = jobtree_find (ptr, id);
			if (node_src == NULL) {
				/* Should not be possible! */
				sql_free_row (row);
				continue;
			}
		}

		/* Find the destination job/jobset in the tree */
		id = (unsigned long long int) sql_row_item2ll (&(row[1]));
		node_dst = jobtree_find (ptr, id);
		if (node_dst == NULL) {
			/*
			 * The destination job is not in the current
			 * workload tree. The link is then ignored
			 */
			sql_free_row (row);
			continue;
		}

		/* Allocate memory for the the new link */
		new_link_out = (schedwi_jobtree_link_ptr) malloc (
						sizeof (schedwi_jobtree_link));
		if (new_link_out == NULL) {
			sql_free_row (row);
			lwc_delLL (	rows,
					(void (*)(const void *)) sql_free_row);
			lwc_writeLog (LOG_CRIT, _("Memory allocation error"));
			return -1;
		}

		new_link_in = (schedwi_jobtree_link_ptr) malloc (
						sizeof (schedwi_jobtree_link));
		if (new_link_in == NULL) {
			free (new_link_out);
			sql_free_row (row);
			lwc_delLL (	rows,
					(void (*)(const void *)) sql_free_row);
			lwc_writeLog (LOG_CRIT, _("Memory allocation error"));
			return -1;
		}

		/* Link the new object to the list */
		new_link_out->next = node_src->links;
		node_src->links = new_link_out;
		new_link_out->linked_node = node_dst;
		new_link_out->required_status = job_status_state_int2status (
				(int)sql_row_item2ll (&(row[2])));

		new_link_in->next = node_dst->links_in;
		node_dst->links_in = new_link_in;
		new_link_in->linked_node = node_src;
		new_link_in->required_status = new_link_out->required_status;

		sql_free_row (row);
	}
	lwc_delLL (rows, NULL);
	return 0;
}


/*
 * Create a new jobtree object
 *
 * Return:
 *   The new object (to be freed by the caller by free_jobtree()) or
 *   NULL in case of error (an error message has been logged by
 *	  using lwc_writeLog())
 */
jobtree_ptr
new_jobtree (schedwi_date date, calendar_list_t_ptr calendars)
{
	jobtree_ptr ptr;
	unsigned int nb_jobs;

#if HAVE_ASSERT_H
	assert (calendars != NULL);
#endif

	/* Allocate memory for the new object */
	ptr = (jobtree_ptr) malloc (sizeof (jobtree));
	if (ptr == NULL) {
		lwc_writeLog (LOG_CRIT, _("Memory allocation error"));
		return NULL;
	}

	/* Initialize the mutex */
	if (pthread_mutex_init (&(ptr->mutex), NULL) != 0) {
		lwc_writeLog (	LOG_ERR,
				_("Internal error: mutex initialization: %s"),
				strerror (errno));
		free (ptr);
	}


	/* Build the job/jobset tree */
	if (build_jobtree (date, calendars, &(ptr->top), &nb_jobs) != 0) {
		pthread_mutex_destroy (&(ptr->mutex));
		free (ptr);
		return NULL;
	}

	/* Walk in the job/jobset tree and fill the provided array */
	if (build_index (ptr->top, nb_jobs, &(ptr->jobs)) != 0) {
		free_jobtree_all_nodes (ptr->top);
		pthread_mutex_destroy (&(ptr->mutex));
		free (ptr);
		lwc_writeLog (LOG_CRIT, _("Memory allocation error"));
		return NULL;
	}
	ptr->nb_jobs = nb_jobs;
	ptr->date = date;

	/* Build the link lists for each job/jobset */
	if (build_links (ptr) != 0) {
		free_jobtree (ptr);
		return NULL;
	}

	return ptr;
}


/*
 * Free a job/jobset tree
 */
void
free_jobtree (jobtree_ptr ptr)
{
	if (ptr != NULL) {
		pthread_mutex_lock (&(ptr->mutex));
		if (ptr->jobs != NULL) {
			free (ptr->jobs);
		}
		free_jobtree_all_nodes (ptr->top);
		ptr->jobs    = NULL;
		ptr->top     = NULL;
		ptr->nb_jobs = 0;
		pthread_mutex_unlock (&(ptr->mutex));
		pthread_mutex_destroy (&(ptr->mutex));
		free (ptr);
	}
}


/*
 * Check if the workload is completed (ie. the root node is completed)
 *
 * Return:
 *   0 --> Not completed
 *   1 --> Completed
 */
int
schedwi_jobtree_is_completed (jobtree_ptr ptr)
{
	int ret;

	if (ptr == NULL) {
		return 1;
	}

	pthread_mutex_lock (&(ptr->mutex));
	if (ptr->top == NULL || ptr->top->status == JOB_STATUS_STATE_COMPLETED)
	{
		ret = 1;
	}
	else {
		ret = 0;
	}
	pthread_mutex_unlock (&(ptr->mutex));

	return ret;
}

/* Declarations */
static int schedwi_jobtree_node_check OF((	schedwi_date workload_date,
						schedwi_jobtree_node_ptr ptr,
						schedwi_time current_time));
static int schedwi_jobtree_node_set_status OF((	schedwi_date workload_date,
						schedwi_jobtree_node_ptr ptr,
						job_status_state status,
						schedwi_time current_time,
						long int duration,
						char fast,
						const char *status_message));

/*
 * Do the transition from a status to an other.  For this version, this
 * function starts the job on the remote client
 *
 * Return:
 *	0 --> No error
 *  other --> Error. A message has been logged by lwc_writeLog()
 */
static int
do_transition (	schedwi_date workload_date,
		schedwi_jobtree_node_ptr ptr,
		schedwi_time current_time)
{
	int ret;
	char *reply;

#if HAVE_ASSERT_H
	assert (ptr != NULL);
#endif

	switch (ptr->status) {
		case JOB_STATUS_STATE_RUNNING:
			/* Start the job */
			if (ptr->node_type != 0) {
				reply = NULL;
				ret = startjob (schedwi_date_to_int
							(workload_date),
						ptr->id, &reply);
				if (ret != 0) {
					/*
					 * Failed to start the job, or job
					 * completed or detached (if ret == 2)
					 */
	  				ptr->wait_reason = 0;
					ret = schedwi_jobtree_node_set_status (
						workload_date,
						ptr,
						(ret == 2)
						   ? JOB_STATUS_STATE_COMPLETED
						   : JOB_STATUS_STATE_FAILED,
						current_time,
						0, 0,
						reply);
					if (reply != NULL) {
						free (reply);
					}
					return ret;
				}
				if (reply != NULL) {
					free (reply);
				}
			}
			else {
				/* If its a jobset without jobs */
				if (	   ptr->node_type == 0
					&& ptr->children == NULL)
				{
	  				ptr->wait_reason = 0;
					return schedwi_jobtree_node_set_status(
						workload_date,
						ptr,
						JOB_STATUS_STATE_COMPLETED,
						current_time, 0, 0, NULL);
				}
			}
			return 0;

		default:
			/*TODO : add transition actions for other status */
			return 0;
	}
}


/*
 * Change the status of a job/jobset. The fast parameter specifies if the
 * status must be propagated to the other jobs (0) or not (1).
 * This flag is used to avoid dead locks with clients and is used when a
 * agent has sent the result status of a job.
 *
 * Return:
 *   0 --> No error
 *  -1 --> Error.  An error message has been logged by using lwc_writeLog()
 *  -2 --> Database error.  An error message has been logged by using
 *	   lwc_writeLog()
 */
static int
schedwi_jobtree_node_set_status (	schedwi_date workload_date,
					schedwi_jobtree_node_ptr ptr,
					job_status_state status,
					schedwi_time current_time,
					long int duration,
					char fast,
					const char *status_message)
{
	int ret, d;
	schedwi_jobtree_link_ptr link;
	schedwi_jobtree_node_ptr child, sibling;
	job_status_state new_parent_status, old_status;

	if (ptr == NULL) {
		return 0;
	}

	/* The job/jobset has already the requested status */
	if (ptr->status == status) {
		return 0;
	}

	/* Check the number of retries if the job has failed */
	if (	   ptr->node_type == 1
		&& status == JOB_STATUS_STATE_FAILED
		&& ptr->retries > ptr->retry_num)
	{
		ptr->retry_num++;
		status = JOB_STATUS_STATE_WAITING;
		ptr->run_time = schedwi_time_add_seconds (current_time,
							ptr->retries_interval);
	}

	/* Update the status in the database */
	ret = job_status_set (ptr->id, ptr->path,
			schedwi_date_to_int (workload_date), status,
			current_time,
			(duration >= 0)	? duration
					: current_time - ptr->start_time,
			ptr->retry_num,
			ptr->wait_reason,
			status_message);
	if (ret != 0) {
		return ret;
	}

	/* Change the status of the job/jobset */
	old_status = ptr->status;
	ptr->status = status;

	/* Call the `schedwi_status' function of the modules */
	d = schedwi_date_to_int (workload_date);
	module_status (d, ptr, old_status, status);

	/* Set the start time */
	if (status == JOB_STATUS_STATE_RUNNING) {
		ptr->start_time = current_time;
	}

	if (fast == 0) {
		/* Tell all the jobs/jobsets waiting to check their status */
		for (	link = ptr->links_in;
			   link != NULL
			&& (   link->required_status == ptr->status
			    || (link->required_status ==
					JOB_STATUS_STATE_COMPLETED_OR_FAILED
				&& ( ptr->status == JOB_STATUS_STATE_COMPLETED
				  || ptr->status == JOB_STATUS_STATE_FAILED)));
			link = link->next)
		{
			ret = schedwi_jobtree_node_check (workload_date,
							link->linked_node,
							current_time);
			if (ret != 0) {
				return ret;
			}
		}

		/* Start all the children */
		if (ptr->status == JOB_STATUS_STATE_RUNNING) {
			for (	child = ptr->children;
				child != NULL;
				child = child->next)
			{
				ret = schedwi_jobtree_node_check (workload_date,
								child,
								current_time);
				if (ret != 0) {
					return ret;
				}
			}
		}
	}

	/* Update the parent jobset */
	if (ptr->status == JOB_STATUS_STATE_RUNNING) {
	  	ptr->wait_reason = 0;
		ret =  schedwi_jobtree_node_set_status (
						workload_date, ptr->parent,
						JOB_STATUS_STATE_RUNNING,
						current_time, -1, fast, NULL);
		if (ret != 0) {
			return ret;
		}
	}
	else {
		if (	   ptr->status == JOB_STATUS_STATE_COMPLETED
			|| ptr->status == JOB_STATUS_STATE_FAILED)
		{
			new_parent_status = ptr->status;
			/* Check if all the next nodes are finished */
			for (	sibling = ptr->next;
				sibling != NULL;
				sibling = sibling->next)
			{
				if (sibling->status == JOB_STATUS_STATE_FAILED)
				{
					new_parent_status =
						JOB_STATUS_STATE_FAILED;
					continue;
				}
				if (sibling->status !=
						JOB_STATUS_STATE_COMPLETED)
				{
					break;
				}
			}
			if (sibling == NULL) {
				/* Now check all the previous nodes */
				for (	sibling = ptr->prev;
					sibling != NULL;
					sibling = sibling->prev)
				{
					if (sibling->status ==
						JOB_STATUS_STATE_FAILED)
					{
						new_parent_status =
						JOB_STATUS_STATE_FAILED;
						continue;
					}
					if (sibling->status !=
						JOB_STATUS_STATE_COMPLETED)
					{
						break;
					}
				}
				if (sibling == NULL) {
	   				ptr->wait_reason = 0;
					ret = schedwi_jobtree_node_set_status (
							workload_date,
							ptr->parent,
							new_parent_status,
							current_time,
							-1, fast, NULL);
					if (ret != 0) {
						return ret;
					}
				}
			}
		}
	}

	/* Do the transition to the new state */
	return do_transition (workload_date, ptr, current_time);
}


/*
 * Check if the provided job/jobset must be started
 *
 * Return:
 *	0 --> No error
 *  other --> Error.  An error message has been logged by lwc_writeLog()
 */
static int
schedwi_jobtree_node_check (	schedwi_date workload_date,
				schedwi_jobtree_node_ptr ptr,
				schedwi_time current_time)
{
	int ret, d;
	char *reply;

	if (ptr == NULL) {
		return 0;
	}

	d = schedwi_date_to_int (workload_date);
	update_jobtree_node (ptr, d);

	switch (ptr->status) {

		case JOB_STATUS_STATE_RUNNING:
			/*
			 * Be sure that the current job/jobset has not
			 * run for too long
			 */
			if (	   ptr->max_duration > 0
				&& schedwi_time_add_seconds (ptr->start_time,
							ptr->max_duration)
					< current_time)
			{
				/* Too long - Stop the job */
				lwc_writeLog (	LOG_INFO,
	_("Workload %d: %s (id %lld): running for too long. Stopping..."),
						d, ptr->path, ptr->id);
				reply = NULL;
				ret = stopjob (d, ptr->id, &reply);
				if (ret == 0) {
					if (reply != NULL) {
						free (reply);
					}
					return 0;
				}
				if (ret > 0) {
					if (reply != NULL) {
						lwc_writeLog (LOG_ERR,
		_("Workload %d: %s (id %lld): failed to stop: %s"),
							d, ptr->path,
							ptr->id, reply);
					}
					else {
						lwc_writeLog (LOG_ERR,
			_("Workload %d: %s (id %lld): failed to stop"),
							d, ptr->path, ptr->id);
					}
				}
				if (reply != NULL) {
					free (reply);
				}
				return -1;
			}
			return 0;

		case JOB_STATUS_STATE_WAITING:
			ret = check_waiting_can_start (d, ptr, current_time);
			if (ret == 0) {
				return schedwi_jobtree_node_set_status (
						workload_date, ptr,
						JOB_STATUS_STATE_RUNNING,
						current_time, -1, 0,
						NULL);
			}
			if (ret == 2) {
				return schedwi_jobtree_node_set_status (
						workload_date, ptr,
						JOB_STATUS_STATE_FAILED,
						current_time, 0, 0,
						_("Start time limit reached"));
			}
			return (ret < 0) ? ret: 0;

		default:
			return 0;
	}
}


/*
 * Walk through the tree to start jobs
 *
 * Return:
 *    0 --> No error
 *   -1 --> Error.  An error message has been logged by lwc_writeLog()
 */
int
schedwi_jobtree_run (jobtree_ptr ptr, schedwi_time current_time)
{
	char error;
	unsigned int i;

	if (ptr == NULL) {
		return 0;
	}

	error = 0;

	pthread_mutex_lock (&(ptr->mutex));
	for (i = 0; i < ptr->nb_jobs; i++) {
		/* If there is an error, continue anyway with the other jobs */
		if (schedwi_jobtree_node_check (	ptr->date,
							(ptr->jobs)[i].node,
							current_time) != 0)
		{
			error = 1;
		}
	}
	pthread_mutex_unlock (&(ptr->mutex));
	return (error != 0) ? -1 : 0;
}


/*
 * Set the status of a job
 *
 * Return:
 *   0 --> No error
 *  -1 --> Error.  An error message has been logged by using lwc_writeLog()
 *  -2 --> Database error.  An error message has been logged by using
 *	   lwc_writeLog()
 */
int
schedwi_jobtree_job_finished (	jobtree_ptr ptr,
				unsigned long long int id,
				job_status_state status,
				long int duration,
				const char *status_message)
{
	int ret;
	schedwi_jobtree_node_ptr node_ptr;

	if (ptr == NULL) {
		return 0;
	}

	pthread_mutex_lock (&(ptr->mutex));

	/* Find the job for which the status must be changed */
	node_ptr = jobtree_find (ptr, id);
	if (node_ptr == NULL) {
		/* Not found.  Shouldn't be possible */
		pthread_mutex_unlock (&(ptr->mutex));
		lwc_writeLog (LOG_ERR, "Could not find job %lld", id);
		return 0;
	}

	/*
	 * Change the status of the job (and do not check the
	 * other jobs for now as the agent network connection is waiting)
	 */
	if (status != JOB_STATUS_STATE_WAITING) {
		node_ptr->wait_reason = 0;
	}
	ret = schedwi_jobtree_node_set_status (	ptr->date,
						node_ptr,
						status,
						schedwi_time_now (),
						duration,
						1,
						status_message);
	pthread_mutex_unlock (&(ptr->mutex));
	return ret;
}


/*
 * Recursively call the provided function for the provided job/jobset ID
 * and its children
 */
static void
schedwi_jobtree_job_foreach_recur (schedwi_jobtree_node_ptr node_ptr,
				int (*foreach_func)(schedwi_jobtree_node_ptr,
							void *),
				void *data_user)
{
	schedwi_jobtree_node_ptr child;

	if (node_ptr == NULL) {
		return;
	}

	if (foreach_func (node_ptr, data_user) != 0) {
		return;
	}
	for (child = node_ptr->children; child != NULL; child = child->next) {
		schedwi_jobtree_job_foreach_recur (	child, foreach_func,
							data_user);
	}
}


/*
 * Run the provided function for the provided job/jobset ID and all its
 * children.
 * If foreach_func() returns a value different from 0, the recursion is
 * stopped.
 */
void
schedwi_jobtree_job_foreach (	jobtree_ptr ptr,
				unsigned long long int id,
				int (*foreach_func)(schedwi_jobtree_node_ptr,
							void *),
				void *data_user)
{
	schedwi_jobtree_node_ptr node_ptr;

	if (ptr == NULL || foreach_func == NULL) {
		return;
	}

	pthread_mutex_lock (&(ptr->mutex));

	/* Find the job from which the recusrsion should start */
	node_ptr = jobtree_find (ptr, id);
	if (node_ptr == NULL) {
		/* Not found.  Shouldn't be possible */
		pthread_mutex_unlock (&(ptr->mutex));
		lwc_writeLog (LOG_ERR, "Could not find job %lld", id);
		return;
	}

	schedwi_jobtree_job_foreach_recur (node_ptr, foreach_func, data_user);
	pthread_mutex_unlock (&(ptr->mutex));
}


/*
 * Recursively print the job/jobset status
 */
static void
schedwi_jobtree_print_recur (schedwi_jobtree_node_ptr ptr, unsigned int depth)
{
	char *spaces;
	unsigned int space_len;

	if (ptr == NULL) {
		return;
	}

	space_len = depth * TABSTOP;
	spaces = (char *) malloc (space_len + 1);
	if (spaces != NULL) {
		schedwi_memset (spaces, ' ', space_len);
		spaces[space_len] = '\0';
	}

	lwc_writeLog (	LOG_INFO, "%s%s: %s (id %lld): %s",
			(spaces != NULL) ? spaces : " ",
			(ptr->node_type == 0) ? _("Jobset"): _("Job"),
			ptr->path,
			ptr->id,
			job_status_state2str (ptr->status));

	if (spaces != NULL) {
		free (spaces);
	}

	schedwi_jobtree_print_recur (ptr->children, depth + 1);
	schedwi_jobtree_print_recur (ptr->next, depth);
}


/*
 * Print the job/jobset status for the provided tree
 */
void
schedwi_jobtree_print (jobtree_ptr ptr)
{
	if (ptr != NULL) {
		pthread_mutex_lock (&(ptr->mutex));
		schedwi_jobtree_print_recur (ptr->top, 1);
		pthread_mutex_unlock (&(ptr->mutex));
	}
}


/*
 * Adjust the jobs/jobsets max duration or start time according
 * to the new clock
 */
void
schedwi_jobtree_adjust (jobtree_ptr ptr, int drift, schedwi_time current_time)
{
	unsigned int i;
	schedwi_jobtree_node_ptr node;
	int workload_date;

	if (ptr == NULL || drift == 0) {
		return;
	}

	workload_date = schedwi_date_to_int (ptr->date);
	lwc_writeLog (	LOG_INFO,
	_("Workload %d: clock drift (%+d seconds): countermeasures engaged"),
			workload_date, drift);
	pthread_mutex_lock (&(ptr->mutex));
	for (i = 0; i < ptr->nb_jobs; i++) {
		node = (ptr->jobs)[i].node;
		if (	   node->status == JOB_STATUS_STATE_RUNNING
				&& node->max_duration > 0)
		{
			node->start_time += drift;
			/* Update the database */
			sql_status_adjust_start_time (	workload_date,
							node->id,
							node->start_time,
							NULL, NULL);
		}
		else if (   drift > 0
			 && node->status == JOB_STATUS_STATE_WAITING
			 && node->start_limit > 0
			 && node->run_time < current_time)
		{
			node->start_limit += drift;
			/* Update the database */
			sql_job_set_start_limit (	workload_date,
							node->id,
							node->start_limit,
							NULL, NULL);
		}
	}
	pthread_mutex_unlock (&(ptr->mutex));
}

/*-----------------============== End Of File ==============-----------------*/
