/* Schedwi
   Copyright (C) 2007 Herve Quatremain

   This file is part of Schedwi.

   Schedwi is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 3 of the License, or
   (at your option) any later version.

   Schedwi is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

/* module.c -- Modules management functions */

#include <schedwi.h>

#if STDC_HEADERS
#include <stdlib.h>
#include <string.h>
#endif

#if HAVE_STDIO_H
#include <stdio.h>
#endif


#if HAVE_ASSERT_H
#include <assert.h>
#endif

#include <ltdl.h>
#include <pthread.h>

#include <lib_functions.h>
#include <lwc_linkedlist.h>
#include <lwc_log.h>
#include <utils.h>
#include <module.h>


static pthread_mutex_t lock = PTHREAD_MUTEX_INITIALIZER;
static lwc_LL *module_list = NULL;
static char initialized = 0;

struct module_node {
	char *name;
	char *filename;
	lt_dlhandle handle;
	int (*mod_init)();
	int (*mod_conf)(const char *, const char *);
	int (*mod_status)(	int, const schedwi_jobtree_node_ptr,
				job_status_state, job_status_state);
	int (*mod_check)(int, const schedwi_jobtree_node_ptr);
	void (*mod_exit)();
};


/*
 * Free the provided module node
 */
static void
free_module_list (struct module_node *ptr)
{
	if (ptr != NULL) {
		if (ptr->name != NULL) {
			free (ptr->name);
		}
		if (ptr->filename != NULL) {
			free (ptr->filename);
		}
		if (ptr->mod_exit != NULL) {
			(*(ptr->mod_exit)) ();
		}
		if (ptr->handle != NULL) {
			lt_dlclose (ptr->handle);
		}
		free (ptr);
	}
}


/*
 * Compare a module name with a module object
 *
 * Return:
 *    0 --> the name and the module object name are the same
 *   -1 --> the name is lower than the module object name
 *    1 --> the name is greater than the module object name
 */
static int
compar_module_node1 (const char *a, const struct module_node *b)
{
	if (a == NULL) {
		if (b == NULL) {
			return 0;
		}
		return -1;
	}
	if (b == NULL || b->name == NULL) {
		return 1;
	}

	return strcmp (a, b->name);
}


/*
 * Compare two module objects
 *
 * Return:
 *    0 --> the two module objects have the same name
 *   -1 --> the first module object name is lower than the second one
 *    1 --> the first module object name is greater than the second one
 */
static int
compar_module_node2 (const struct module_node *a, const struct module_node *b)
{
	if (a == NULL || a->name == NULL) {
		if (b == NULL || b->name == NULL) {
			return 0;
		}
		return -1;
	}
	if (b == NULL || b->name == NULL) {
		return 1;
	}

	return strcmp (a->name, b->name);
}


/*
 * Initialize the module engine
 *
 * Return:
 *   0 --> No error
 *  -1 --> Error (an error message has been printed by fprintf())
 */
int
module_init ()
{
	if (initialized != 0) {
		return 0;
	}

	LTDL_SET_PRELOADED_SYMBOLS ();

	if (lt_dlinit () != 0) {
		fprintf (	stderr, _("Module initialization: %s\n"),
				lt_dlerror ());
		return -1;
	}

	if (lt_dladdsearchdir (PKGLIBDIR) != 0) {
		fprintf (	stderr, _("Module path initialization: %s\n"),
				lt_dlerror ());
		lt_dlexit ();
		return -1;
	}

	initialized = 1;
	return 0;
}


/*
 * Destroy the module engine
 */
void
module_exit ()
{
	pthread_mutex_lock (&lock);
	lwc_delLL (module_list, (void (*)(const void *))free_module_list);
	module_list = NULL;
	pthread_mutex_unlock (&lock);
	initialized = 0;
	lt_dlexit ();
}


/*
 * Add a path to the search directory list
 *
 * Return:
 *   0 --> No error
 *  -1 --> Error (an error message has been printed by fprintf())
 */
int
module_path (const char *path)
{
	if (path != NULL && lt_dladdsearchdir (path) != 0) {
		fprintf (	stderr, _("Module path initialization: %s\n"),
				lt_dlerror ());
		return -1;
	}
	return 0;
}


/*
 * Load a module
 *
 * Return:
 *   0 --> No error
 *  -1 --> Error (a message has been printed by fprintf())
 *   1 --> The module init function returned a non-zero code
 */
int
module_load (const char *file)
{
	const char *name;
	char *s, *n;
	struct module_node *ptr;


	if (file == NULL) {
		return 0;
	}

	/*
	 * Build the module name (file without the path and
	 * without the extension)
	 */
	name = base_name (file);
	n = (char *)malloc (schedwi_strlen (name) + 1);
	if (n == NULL) {
		fprintf (stderr, _("Memory allocation error\n"));
		return -1;
	}
	strcpy (n, name);
	s = strchr (n, '.');
	if (s != NULL && s != n) {
		*s = '\0';
	}

	/* Check if the module is already loaded */
	pthread_mutex_lock (&lock);
	ptr = (struct module_node *) lwc_searchSortLL (module_list, n,
		(int (*)(const void *, const void *)) compar_module_node1);
	if (ptr != NULL) {
		free (n);
		if (ptr->filename == NULL) {
			fprintf (stderr,
				_("Module %s already loaded\n"),
				file);
		}
		else {
			fprintf (stderr,
				_("Module %s already loaded from %s\n"),
				file, ptr->filename);
		}
		pthread_mutex_unlock (&lock);
		return 0;
	}

	/* Create the list if need be */
	if (module_list == NULL) {
		module_list = lwc_newLL ();
		if (module_list == NULL) {
			pthread_mutex_unlock (&lock);
			free (n);
			fprintf (stderr, _("Memory allocation error\n"));
			return -1;
		}
	}
	pthread_mutex_unlock (&lock);

	ptr = (struct module_node *)malloc (sizeof (struct module_node));
	if (ptr == NULL) {
		free (n);
		fprintf (stderr, _("Memory allocation error\n"));
		return -1;
	}

	ptr->filename = (char *)malloc (schedwi_strlen (file) + 1);
	if (ptr->filename == NULL) {
		free (ptr);
		free (n);
		fprintf (stderr, _("Memory allocation error\n"));
		return -1;
	}
	strcpy (ptr->filename, file);

	/* Open the module */
	ptr->handle = lt_dlopenext (file);
	if (ptr->handle == NULL) {
		fprintf (	stderr, _("Module load: error: %s: %s\n"),
				file, lt_dlerror ());
		free (ptr->filename);
		free (ptr);
		free (n);
		return -1;
	}

	/* Load the functions */
	ptr->mod_init = (int (*)()) lt_dlsym (ptr->handle, "schedwi_init");
	ptr->mod_conf = (int (*)(const char *, const char *))
				lt_dlsym (ptr->handle, "schedwi_conf");
	ptr->mod_status = (int (*)(int,const schedwi_jobtree_node_ptr,job_status_state,job_status_state)) lt_dlsym (ptr->handle, "schedwi_status");
	ptr->mod_check = (int (*)(int,const schedwi_jobtree_node_ptr))
				lt_dlsym (ptr->handle, "schedwi_check");
	ptr->mod_exit = (void (*)()) lt_dlsym (ptr->handle, "schedwi_exit");

	/* Call the module init function */
	if (ptr->mod_init && (*(ptr->mod_init)) () != 0) {
		fprintf (stderr, _("Module %s: init function error\n"), n);
		lt_dlclose (ptr->handle);
		free (ptr->filename);
		free (ptr);
		free (n);
		return 1;
	}

	/* Link the new module object to the list of modules */
	ptr->name = n;
	pthread_mutex_lock (&lock);
	if (lwc_addLL (module_list, ptr, (int (*)(const void *, const void *))
			compar_module_node2) != 0)
	{
		pthread_mutex_unlock (&lock);
		if (ptr->mod_exit != NULL) {
			(*(ptr->mod_exit)) ();
		}
		lt_dlclose (ptr->handle);
		free (ptr->filename);
		free (ptr);
		free (n);
		fprintf (stderr, _("Memory allocation error\n"));
		return -1;
	}
	pthread_mutex_unlock (&lock);
	return 0;
}


/*
 * Forward a configuration parameter to the corresponding module
 *
 * Return:
 *   0 --> No error
 *  -1 --> Error: unknown module or the module configuration function
 *         return a non-zero value.  A message has been printed by fprintf()
 */
int
module_conf (const char *name, const char *key, const char *value)
{
	struct module_node *ptr;

	if (name == NULL || key == NULL || value == NULL) {
		return 0;
	}

	/* Retrieve the module object */
	pthread_mutex_lock (&lock);
	ptr = (struct module_node *) lwc_searchSortLL (module_list, name,
		(int (*)(const void *, const void *)) compar_module_node1);
	if (ptr == NULL) {
		pthread_mutex_unlock (&lock);
		fprintf (stderr,
			_("Unknown module `%s' for parameter `%s=%s'\n"),
			name, key, value);
		return -1;
	}

	if (ptr->mod_conf != NULL && (*(ptr->mod_conf)) (key, value) != 0) {
		pthread_mutex_unlock (&lock);
		fprintf (	stderr,
				_("Module %s: error: parameter `%s=%s'\n"),
				name, key, value);
		return -1;
	}
	pthread_mutex_unlock (&lock);
	return 0;
}


/*
 * Call all the module `schedwi_status' functions for the provided job with
 * the provided status
 *
 * Return:
 *   0 --> No error
 *  -1 --> At least a `schedwi_status' function returned non-zero.  An error
 *         message is logged by lwc_writeLog().
 */
int
module_status (	int workload, const schedwi_jobtree_node_ptr job,
		job_status_state old_status, job_status_state new_status)
{
	int ret;
	struct module_node *ptr;

#if HAVE_ASSERT_H
	assert (job != NULL);
#endif

	ret = 0;
	pthread_mutex_lock (&lock);
	lwc_rewindLL (module_list);
	while ((ptr = (struct module_node *)lwc_nextLL (module_list)) != NULL)
	{
		if (	   ptr->mod_status != NULL
			&& (*(ptr->mod_status)) (	workload, job,
							old_status,
							new_status) != 0)
		{
			lwc_writeLog (	LOG_ERR,
_("Module %s: schedwi_status: Workload %d: %s (id %lld): from %s to %s: error"),
					ptr->name,
					workload,
					job->path,
					job->id,
					job_status_state2str (old_status),
					job_status_state2str (new_status));
			ret = -1;
		}
	}
	pthread_mutex_unlock (&lock);
	return ret;
}


/*
 * Call all the module `schedwi_check' functions for the provided job
 *
 * Return:
 *   0 --> No error. The job can be run (all the `schedwi_check' functions
 *         returned 0)
 *   1 --> No error. The job cannot be started yet (a `schedwi_check' function
 *         returned a code greater than 0)
 *  -1 --> Error.  A `schedwi_check' function returned a code lower than 0.
 *         A message has been logged by lwc_writeLog().
 */
int
module_check (int workload, const schedwi_jobtree_node_ptr job)
{
	int ret;
	struct module_node *ptr;

#if HAVE_ASSERT_H
	assert (job != NULL);
#endif

	pthread_mutex_lock (&lock);
	lwc_rewindLL (module_list);
	while ((ptr = (struct module_node *)lwc_nextLL (module_list)) != NULL)
	{
		if (ptr->mod_check != NULL) {
			ret = (*(ptr->mod_check)) (workload, job);
			if (ret > 0) {
				pthread_mutex_unlock (&lock);
				return 1;
			}
			if (ret < 0) {
				lwc_writeLog (	LOG_ERR,
	_("Module %s: schedwi_check: Workload %d: %s (id %lld): error"),
					ptr->name,
					workload,
					job->path,
					job->id);
				pthread_mutex_unlock (&lock);
				return -1;
			}
		}
	}
	pthread_mutex_unlock (&lock);
	return 0;
}

/*-----------------============== End Of File ==============-----------------*/
