/* Schedwi
   Copyright (C) 2007-2013 Herve Quatremain

   This file is part of Schedwi.

   Schedwi is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 3 of the License, or
   (at your option) any later version.

   Schedwi is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

/* job_status.c -- Manage the jobs/jobsets status */

#include <schedwi.h>

#if STDC_HEADERS
#include <stdlib.h>
#else
#if HAVE_STDLIB_H
#include <stdlib.h>
#endif
#endif

#include <lwc_log.h>
#include <utils.h>
#include <sql_status.h>
#include <sql_stat.h>
#include <job_status.h>

/*
 * Error callback function for the sql_status_set() function
 */
static void
sql_status_set_error_logger (void *data, const char *msg, int err_code)
{
	if (msg != NULL) {
		lwc_writeLog (LOG_ERR, msg);
	}
	else {
		lwc_writeLog (LOG_ERR,
		    _("Database error while trying to set a job status"));
	}
}


/*
 * Error callback function for the sql_stat_success() and
 * sql_stat_fail() functions
 */
static void
sql_stat_error (void *data, const char *msg, int err_code)
{
	if (msg != NULL) {
		lwc_writeLog (LOG_ERR, msg);
	}
	else {
		lwc_writeLog (LOG_ERR,
		_("Database error while trying to update the statistics"));
	}
}


/*
 * Set the status of the provided job/jobset.  The error_message is the
 * associated message that will be added in the database with the new job
 * status.  It may be NULL.  status_time is the date/time of the new
 * status.  If it is 0, it will be set to the current date/time by this
 * function.
 *
 * Return:
 *   0 --> No error
 *  -1 --> Memory allocation error.  An error message has been logged by
 *         using lwc_writeLog().
 *  -2 --> Database error.  An error message has been logged by
 *         using lwc_writeLog().
 */
int
job_status_set (unsigned long long int id,
		const char *path,
		int workload_date,
		job_status_state status,
		unsigned int status_time,
		long int duration,
		int retry_num,
		int wait_reason,
		const char *error_message)
{
	int j_status;
	char id_str[25];
	int ret;

	/* Update the job_status database table */
	j_status = job_status_state_status2int (status);
	copy_ulltostr (id, id_str);
	ret = sql_status_set (	workload_date,
				id_str, j_status,
				(status_time == 0) ? time (NULL): status_time,
				retry_num,
				wait_reason,
				error_message,
				sql_status_set_error_logger, NULL);
	if (ret != 0) {
		return (ret == 1) ? -1: -2;
	}

	/* Update the job_stat database table */
	if (status == JOB_STATUS_STATE_COMPLETED) {
		sql_stat_success (id_str, duration, sql_stat_error, NULL);

	}
	else {
		if (status == JOB_STATUS_STATE_FAILED) {
			sql_stat_fail (id_str, duration, sql_stat_error, NULL);
		}
	}

	lwc_writeLog (	LOG_INFO, _("Workload %d: %s (id %s): %s"),
			workload_date,
			path,
			id_str,
			job_status_state2str (status));
	return 0;
}

/*-----------------============== End Of File ==============-----------------*/
