/* Schedwi
   Copyright (C) 2007-2013 Herve Quatremain

   This file is part of Schedwi.

   Schedwi is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 3 of the License, or
   (at your option) any later version.

   Schedwi is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

/* commands_thread.c -- Thread which runs user commands */

#include <schedwi.h>

#if STDC_HEADERS
#include <stdlib.h>
#include <string.h>
#else
#if HAVE_STDLIB_H
#include <stdlib.h>
#endif
#if HAVE_STRING_H
#include <string.h>
#endif
#endif

#if HAVE_SIGNAL_H
#include <signal.h>
#endif

#if HAVE_STDIO_H
#include <stdio.h>
#endif

#if HAVE_TIME_H
#include <time.h>
#endif
#if TM_IN_SYS_TIME
#include <sys/time.h>
#endif

#if HAVE_UNISTD_H
#include <unistd.h>
#endif

#if HAVE_ERRNO_H
#include <errno.h>
#endif
#ifndef errno
extern int errno;
#endif

#if HAVE_ASSERT_H
#include <assert.h>
#endif

#include <pthread.h>

#include <lwc_log.h>
#include <utils.h>
#include <commands.h>
#include <stopjob.h>
#include <job_status_state.h>
#include <thread_init.h>
#include <lib_functions.h>
#include <commands_thread.h>

/* Number of second to sleep between each scan of the user command table */
#define COMMANDS_SCAN_SLEEP 20


static pthread_t commands_thread;


/*
 * Stop the provided job
 */
static int
commands_thread_stop_job (	schedwi_jobtree_node_ptr ptr,
				void *data_user)
{
	int *workload_date = (int *)data_user;
	char *reply = NULL;

	if (	   ptr == NULL || ptr->node_type == 0
		|| ptr->status != JOB_STATUS_STATE_RUNNING
		|| workload_date == NULL || *workload_date == 0)
	{
		return 0;
	}

	if (stopjob (*workload_date, ptr->id, &reply) != 0) {
		if (reply != NULL) {
			lwc_writeLog (LOG_ERR,
		_("Workload %d: %s (id %lld): failed to stop: %s"),
				*workload_date, ptr->path, ptr->id, reply);
		}
		else {
			lwc_writeLog (	LOG_ERR,
			_("Workload %d: %s (id %lld): failed to stop"),
					*workload_date, ptr->path, ptr->id);
		}
	}
	if (reply != NULL) {
		free (reply);
	}
	return 0;
}


/*
 * Run the provided user command
 *
 * Return:
 *       0 --> No error. Command successful
 *       1 --> Command failed
 *   other --> Error.  An error message has been logged by lwc_writeLog()
 */
static int
commands_thread_action (command_action_ptr cmd, load_workload_class_ptr ptr)
{
	workload_class_ptr workload_obj;
	char *s, *fmt;
	char id_str[25];
	int ret, d;
	unsigned int len;
	job_status_state status;

#if HAVE_ASSERT_H
	assert (ptr != NULL);
#endif

	if (cmd == NULL || cmd->cmd == COMMAND_UNKNOWN) {
		return 0;
	}

	/* Retrieve the workload */
	workload_obj = load_workload_class_find (ptr, cmd->workload);
	if (workload_obj == NULL) {
		return 0;
	}

	/* Convert the workload date to an int */
	d = schedwi_date_to_int (cmd->workload);

	/* Convert the job ID to a string */
	copy_ulltostr (cmd->job_id, id_str);

	switch (cmd->cmd) {

		/* Stop the job/jobset */
		case COMMAND_STOP:
			lwc_writeLog (	LOG_INFO,
		_("Workload %d: Job/Jobset Id %s: stop request by user (%s)"),
					d, id_str,
					(cmd->user_details == NULL) ? "-"
							: cmd->user_details);
			workload_class_job_foreach (workload_obj,
						cmd->job_id,
						commands_thread_stop_job,
						&d);
			return 0;

		/* Change the status of the job/jobset */
		case COMMAND_CHANGE_STATUS:
			status = job_status_state_int2status (cmd->parameter);
			lwc_writeLog (	LOG_INFO,
_("Workload %d: Job/Jobset Id %s: new status %s requested by user (%s)"),
					d, id_str,
					job_status_state2str (status),
					(cmd->user_details == NULL) ? "-"
							: cmd->user_details);
			if (cmd->user_details == NULL) {
				ret = workload_class_job_finished (
							workload_obj,
							cmd->job_id, status,
							0,
							_("Forced by user"));
			}
			else {
				fmt = _("Forced by %s");
				len =	  schedwi_strlen (fmt)
					+ schedwi_strlen (cmd->user_details);
				s = (char *) malloc (len);
				if (s == NULL) {
					lwc_writeLog (LOG_CRIT,
						_("Memory allocation error"));
					ret = workload_class_job_finished (
							workload_obj,
							cmd->job_id, status,
							0,
							_("Forced by user"));
				}
				else {
					snprintf (	s, len, fmt,
							cmd->user_details);
					ret = workload_class_job_finished (
							workload_obj,
							cmd->job_id, status,
							0, s);
					free (s);
				}
			}

			return ret;

		default:
			return 0;
	}
}


/*
 * Thread main function
 */
static void *
commands_thread_run (void *obj)
{
	load_workload_class_ptr ptr = (load_workload_class_ptr) obj;
	lwc_LL *command_list;
	command_action_ptr command_ptr;
#if HAVE_NANOSLEEP
	struct timespec req;

	req.tv_sec = COMMANDS_SCAN_SLEEP;
	req.tv_nsec = 0;
#endif

#if HAVE_ASSERT_H
	assert (ptr != NULL);
#endif

	/* Initialize the thread */
	thread_init ();

	/* Install the cleanup callback */
	pthread_cleanup_push (thread_clean, NULL);

	/* Run the scan of commands every COMMANDS_SCAN_SLEEP */
	while (1) {

		/* Retrieve the command list from the database */
		command_list = command_get_list ();
		pthread_cleanup_push ((void (*) (void *))command_destroy_list,
					(void *)command_list);

		/* Run each command */
		lwc_rewindLL (command_list);
		while ((command_ptr = (command_action_ptr)
					lwc_nextLL (command_list)) != NULL)
		{
			if (commands_thread_action (command_ptr, ptr) != 0) {
				command_failed (command_ptr);
			}
			else {
				command_done (command_ptr);
			}
		}
		/* command_destroy_list (command_list); */
		pthread_cleanup_pop (1);

		pthread_testcancel ();

		/* Sleep until the next run */
#if HAVE_NANOSLEEP
		nanosleep (&req, NULL);
#elif HAVE_SLEEP
		sleep (COMMANDS_SCAN_SLEEP);
#else
		/*
		 * usleep() should not be used.  On some OS it may
		 * not work correctly with pthread
		 */
		usleep (COMMANDS_SCAN_SLEEP * 1000000);
#endif
		pthread_testcancel ();
	}
	pthread_cleanup_pop (1);
	return NULL;
}


/*
 * Start the new thread for the management of commands
 *
 * Return:
 *   0 --> No error
 *  -1 --> Error.  An error message has been logged by lwc_writeLog()
 */
int
commands_thread_init (load_workload_class_ptr ptr)
{
	pthread_attr_t attr;

#if HAVE_ASSERT_H
	assert (ptr != NULL);
#endif

	if (pthread_attr_init (&attr) != 0) {
		lwc_writeLog (	LOG_ERR,
			_("Internal error: attribute initialization: %s"),
				strerror (errno));
		return -1;
	}
	pthread_attr_setdetachstate (&attr, PTHREAD_CREATE_JOINABLE);

	if (pthread_create (	&commands_thread, &attr,
				commands_thread_run, ptr) != 0)
	{
		lwc_writeLog (	LOG_ERR,
				_("Internal error: thread creation: %s"),
				strerror (errno));
		pthread_attr_destroy (&attr);
		return -1;
	}
	pthread_attr_destroy (&attr);
	return 0;
}


/*
 * Destroy the thread
 */
void
commands_thread_destroy ()
{
	pthread_cancel (commands_thread);
	pthread_kill (commands_thread, SIGUSR2);
	pthread_join (commands_thread, NULL);
}

/*-----------------============== End Of File ==============-----------------*/
