# Schedwi
# Copyright (C) 2013 Herve Quatremain
#
# This file is part of Schedwi.
#
# Schedwi is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
#
# Schedwi is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

"""The trigger/acknowledge manual job/jobset window"""

from datetime import date
from babel import dates

from muntjac.api import (Window, VerticalLayout, PasswordField, Button, Label,
                         Alignment, Panel, GridLayout)
from muntjac.ui.button import IClickListener
from muntjac.ui.window import Notification
from muntjac.data.property import IValueChangeListener
from muntjac.terminal.theme_resource import ThemeResource

from web.topbar import TopBar
from simple_queries_ack import (
                            sql_get_acknowledge_request_by_urn_and_password,
                            sql_update_acknowledge_request)
from simple_queries_job import sql_get_job
import locale_utils
import path


class TriggerManualJobWindow(Window, IClickListener, IValueChangeListener):

    """The trigger/acknowledge manual job/jobset window."""

    _TITLE = 'Schedwi Acknowledge'

    def __init__(self, app, urn):
        """Initialize the Activity window.

        @param app:
                    the L{web.main.Main} object.
        @param urn:
                    the urn specified (to check from the database)
        """
        super(TriggerManualJobWindow, self).__init__()
        self._app = app
        self._urn = urn

        mainExpand = VerticalLayout()
        mainExpand.setSizeFull()
        self._mainExpand = mainExpand
        self.setContent(mainExpand)
        self.setSizeFull()
        self.setCaption(self._TITLE)
        self.setTheme('schedwi')

        nav = TopBar(self, app, app, None, None, False)
        mainExpand.addComponent(nav)

        p = Panel()
        p.setSizeUndefined()
        mainExpand.addComponent(p)
        mainExpand.setComponentAlignment(p, Alignment.TOP_CENTER)
        self._panel = p

        password_box = VerticalLayout()
        password_box.setMargin(True)
        password_box.setSpacing(True)
        self._panel.addComponent(password_box)

        p = PasswordField(_("Password:"))
        p.setImmediate(True)
        p.setWidth('100%')
        p.addListener(self, IValueChangeListener)
        password_box.addComponent(p)
        self._password_widget = p

        b = Button(_("OK"))
        b.addListener(self, IClickListener)
        password_box.addComponent(b)
        self._ok_button = b

    def buttonClick(self, event):
        password = self._password_widget.getValue()
        try:
            ack = sql_get_acknowledge_request_by_urn_and_password(self._app,
                                                        self._urn, password)
        except:
            self.showNotification(_("Incorrect URL or password."),
                        _("<br/>Letters must be typed in the correct case."),
                        Notification.TYPE_WARNING_MESSAGE)
            return
        try:
            job = sql_get_job(self._app, ack.job_id, ack.workload_date)
        except:
            self.showNotification(_("Internal error."),
                    _("<br/>Job/jobset (%d_%d) not found in the database.") %
                        (ack.workload_date, ack.job_id),
                    Notification.TYPE_ERROR_MESSAGE)
            return
        self._panel.removeAllComponents()
        t = GridLayout(2, 5)
        t.setMargin(True)
        t.setSpacing(True)
        t.setColumnExpandRatio(1, 1.0)

        # Job/jobset name
        l = Label('<b>' +
                  (_("Jobset:") if job.type == 0 else _("Job")) +
                  '</b>')
        l.setContentMode(Label.CONTENT_XHTML)
        t.addComponent(l, 0, 0)
        t.setComponentAlignment(l, Alignment.MIDDLE_RIGHT)

        p = path.id2path(self._app, ack.job_id, ack.workload_date)
        l = Label(p.path.encode('utf-8'))
        t.addComponent(l, 1, 0)

        # Workload date
        l = Label('<b>' + _("Workload:") + '</b>')
        l.setContentMode(Label.CONTENT_XHTML)
        t.addComponent(l, 0, 1)
        t.setComponentAlignment(l, Alignment.MIDDLE_RIGHT)

        d = date(ack.workload_date / 10000,
                 ack.workload_date % 10000 / 100,
                 ack.workload_date % 100)
        l = Label(dates.format_date(d, format='short',
                                    locale=locale_utils.get_locale()))
        t.addComponent(l, 1, 1)

        # Status
        l = Label('<b>' + _("Status:") + '</b>')
        l.setContentMode(Label.CONTENT_XHTML)
        t.addComponent(l, 0, 2)
        t.setComponentAlignment(l, Alignment.MIDDLE_RIGHT)

        l = Label(_("already acknowledged") if ack.status
                    else _("waiting for acknowledgement"))
        t.addComponent(l, 1, 2)

        # Time
        if ack.status:
            l = Label('<b>' + _("At:") + '</b>')
        else:
            l = Label('<b>' + _("Since:") + '</b>')
        l.setContentMode(Label.CONTENT_XHTML)
        t.addComponent(l, 0, 3)
        t.setComponentAlignment(l, Alignment.MIDDLE_RIGHT)

        l = Label(dates.format_datetime(ack.ack_time, format='full',
                                        locale=locale_utils.get_locale()))
        t.addComponent(l, 1, 3)

        # Username
        if ack.status:
            l = Label('<b>' + _("By:") + '</b>')
            l.setContentMode(Label.CONTENT_XHTML)
            t.addComponent(l, 0, 4)
            t.setComponentAlignment(l, Alignment.MIDDLE_RIGHT)

            l = Label(ack.username.encode('utf-8'))
            t.addComponent(l, 1, 4)

        self._panel.addComponent(t)
        if ack.status == 0:
            b = Button(_("Start"))
            b.setIcon(ThemeResource('icons/start.png'))
            b.setDescription(_("Trigger the start of the job/jobset"))
            b.addListener(StartCallback(ack, self, self._app), IClickListener)
            self._panel.addComponent(b)
        self._error_label = Label()
        self._panel.addComponent(self._error_label)

    def valueChange(self, event):
        self.buttonClick(event)


class StartCallback(IClickListener):

    """Start/Acknowledge button callback."""

    def __init__(self, ack_obj, trigger_manual_jobwindow_obj, sql_session):
        """Initialize the Activity window.

        @param ack_obj:
                    The L{tables.acknowledge_manual.acknowledge_manual} object
        @param trigger_manual_jobwindow_obj:
                    the L{TriggerManualJobWindow} object.
        @param sql_session:
                    SQLAlchemy session.
        """
        super(StartCallback, self).__init__()
        self._ack_obj = ack_obj
        self._trigger_manual_jobwindow_obj = trigger_manual_jobwindow_obj
        self._sql_session = sql_session

    def buttonClick(self, event):
        try:
            sql_update_acknowledge_request(self._sql_session,
                                           self._ack_obj.job_id,
                                           self._ack_obj.workload_date)
        except:
            self._trigger_manual_jobwindow_obj.showNotification(
                    _("Internal error."),
                    _("<br/>Job/jobset (%d_%d) not found in the database.") %
                        (self._ack_obj.workload_date, self._ack_obj.job_id),
                    Notification.TYPE_ERROR_MESSAGE)
            return
        n = Notification(
                    _("Acknowledgement taken into account."),
                    _("<br/>This may take up to a minute to be taken into \
                       account.<br/>You may close this window."))
        n.setDelayMsec(5000)
        self._trigger_manual_jobwindow_obj.showNotification(n)
        # Replace the panel by a already acknowledged message
        self._trigger_manual_jobwindow_obj.buttonClick(None)
