# Schedwi
# Copyright (C) 2013 Herve Quatremain
#
# This file is part of Schedwi.
#
# Schedwi is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
#
# Schedwi is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

"""Topbar."""

from muntjac.api import (HorizontalLayout, Label, Alignment,
                         NativeButton, Button)
from muntjac.terminal.theme_resource import ThemeResource
from muntjac.terminal.sizeable import ISizeable
from muntjac.ui.themes import BaseTheme
from muntjac.ui.button import IClickListener
from muntjac.addon.refresher.refresher import Refresher
from muntjac.addon.refresher.refresher import RefreshListener

from web.envwindow import EnvWindow
from web.hostwindow import HostWindow
from web.calendarwindow import CalendarWindow
import status_utils
import simple_queries_wl


class TopBar(HorizontalLayout, RefreshListener):
    """The top bar."""

    def __init__(self, parent_window, main_obj, sql_session, workload=None,
                 workload_str=None, tool_buttons=False):
        super(TopBar, self).__init__()
        self._parent_window = parent_window
        self._main_obj = main_obj
        self._sql_session = sql_session
        self._workload = workload
        self._workload_str = workload_str
        self._tool_buttons = tool_buttons

        # Main top/expanded-bottom layout
        self.setHeight('30px')
        self.setWidth('100%')
        self.setStyleName('topbar')
        self.setSpacing(False)
        self.setMargin(False, True, False, False)

        logo = self.logo()
        self.addComponent(logo)
        self.setComponentAlignment(logo, Alignment.MIDDLE_LEFT)

        button_box = HorizontalLayout()
        button_box.setSpacing(False)
        button_box.setMargin(False)
        self.addComponent(button_box)
        self.setComponentAlignment(button_box, Alignment.MIDDLE_RIGHT)

        if self._workload is not None:

            if self._workload_str is not None:
                l = Label(self._workload_str)
                button_box.addComponent(l)
                button_box.setComponentAlignment(l, Alignment.MIDDLE_RIGHT)

            h = HorizontalLayout()
            h.setSpacing(False)
            h.setMargin(False, True, False, True)
            button_box.addComponent(h)
            button_box.setComponentAlignment(h, Alignment.MIDDLE_RIGHT)

            self._label_waiting = Label("1")
            self._label_waiting.setStyleName("label-" +
                        status_utils.status2colorname(status_utils.WAITING))
            h.addComponent(self._label_waiting)

            self._label_running = Label("2")
            self._label_running.setStyleName("label-" +
                        status_utils.status2colorname(status_utils.RUNNING))
            h.addComponent(self._label_running)

            self._label_completed = Label("3")
            self._label_completed.setStyleName("label-" +
                        status_utils.status2colorname(status_utils.COMPLETED))
            h.addComponent(self._label_completed)

            self._label_failed = Label("4")
            self._label_failed.setStyleName("label-" +
                        status_utils.status2colorname(status_utils.FAILED))
            h.addComponent(self._label_failed)

            self._refresher = Refresher()
            self._refresher.setRefreshInterval(60000)   # Every minute
            self._refresher.addListener(self, RefreshListener)
            h.addComponent(self._refresher)
            self.refresh()

        if tool_buttons:
            b = NativeButton()
            b.setIcon(ThemeResource('icons/agents.png'))
            b.setDescription(_("Hosts"))
            b.setStyleName('btleft')
            b.addListener(HostPopup(self), IClickListener)
            button_box.addComponent(b)
            button_box.setComponentAlignment(b, Alignment.MIDDLE_RIGHT)

            b = NativeButton()
            b.setIcon(ThemeResource('icons/environments.png'))
            b.setDescription(_("Environment groups"))
            b.setStyleName('btmiddle')
            b.addListener(EnvPopup(self), IClickListener)
            button_box.addComponent(b)
            button_box.setComponentAlignment(b, Alignment.MIDDLE_RIGHT)

            b = NativeButton()
            b.setIcon(ThemeResource('icons/calendar.png'))
            b.setDescription(_("Calendars"))
            b.setStyleName('btmiddle')
            b.addListener(CalPopup(self), IClickListener)
            button_box.addComponent(b)
            button_box.setComponentAlignment(b, Alignment.MIDDLE_RIGHT)

            b = NativeButton()
            b.setIcon(ThemeResource('icons/exit.png'))
            b.setDescription(_("Exit"))
            b.setStyleName('btright')
            b.addListener(Exit(main_obj), IClickListener)
            button_box.addComponent(b)
            button_box.setComponentAlignment(b, Alignment.MIDDLE_RIGHT)

        user_box = HorizontalLayout()
        user_box.setSpacing(False)
        user_box.setMargin(False, False, False, True)
        button_box.addComponent(user_box)
        button_box.setComponentAlignment(user_box, Alignment.MIDDLE_RIGHT)

        user = main_obj.getUser()
        if user is not None:
            l = Label(user.encode('utf-8') + '/')
            user_box.addComponent(l)
            user_box.setComponentAlignment(l, Alignment.MIDDLE_RIGHT)

            b = Button(_("logout"))
            b.setStyleName(BaseTheme.BUTTON_LINK)
            b.addListener(Logout(main_obj), IClickListener)
            user_box.addComponent(b)
            user_box.setComponentAlignment(b, Alignment.MIDDLE_RIGHT)

    def logo(self):
        h = HorizontalLayout()
        h.setStyleName("logo")
        h.setHeight('100%')
        h.setSpacing(False)
        h.setMargin(False, True, False, True)
        logo = Label("Sched")
        logo.setStyleName("logopart1")
        h.addComponent(logo)
        h.setComponentAlignment(logo, Alignment.MIDDLE_LEFT)
        logo = Label("wi")
        logo.setStyleName("logopart2")
        h.addComponent(logo)
        h.setComponentAlignment(logo, Alignment.MIDDLE_LEFT)
        return h

    def refresh(self, source=None):
        TOTAL_WIDTH = 280
        num_waiting, num_running, num_completed, num_failed = \
                simple_queries_wl.num_jobs_by_status(self._sql_session,
                                                     self._workload)
        total = num_waiting + num_running + num_completed + num_failed

        pct_waiting = 100.0 * num_waiting / total
        if pct_waiting >= 20.0:
            self._label_waiting.setValue('%d (%.1f%%)' %
                                         (num_waiting, pct_waiting))
        else:
            self._label_waiting.setValue(int(num_waiting))
        self._label_waiting.setDescription(_("Waiting (%d - %.1f%%)") %
                                           (num_waiting, pct_waiting))
        self._label_waiting.setWidth(int(TOTAL_WIDTH * pct_waiting / 100),
                                     ISizeable.UNITS_PIXELS)

        pct_running = 100.0 * num_running / total
        if pct_running >= 20.0:
            self._label_running.setValue('%d (%.1f%%)' %
                                     (num_running, pct_running))
        else:
            self._label_running.setValue(int(num_running))
        self._label_running.setDescription(_("Running (%d - %.1f%%)") %
                                           (num_running, pct_running))
        self._label_running.setWidth(int(TOTAL_WIDTH * pct_running / 100),
                                     ISizeable.UNITS_PIXELS)

        pct_completed = 100.0 * num_completed / total
        if pct_completed >= 20.0:
            self._label_completed.setValue('%d (%.1f%%)' %
                                       (num_completed, pct_completed))
        else:
            self._label_completed.setValue(int(num_completed))
        self._label_completed.setDescription(_("Completed (%d - %.1f%%)") %
                                             (num_completed, pct_completed))
        self._label_completed.setWidth(int(TOTAL_WIDTH * pct_completed / 100),
                                       ISizeable.UNITS_PIXELS)

        pct_failed = 100.0 * num_failed / total
        if pct_failed >= 20.0:
            self._label_failed.setValue('%d (%.1f%%)' %
                                    (num_failed, pct_failed))
        else:
            self._label_failed.setValue(int(num_failed))
        self._label_failed.setDescription(_("Failed (%d - %.1f%%)") %
                                          (num_failed, pct_failed))
        self._label_failed.setWidth(int(TOTAL_WIDTH * pct_failed / 100),
                                    ISizeable.UNITS_PIXELS)


class Exit(IClickListener):
    def __init__(self, main_obj):
        super(Exit, self).__init__()
        self._main_obj = main_obj

    def buttonClick(self, event):
        self._main_obj.state_select_activity()


class Logout(IClickListener):
    def __init__(self, main_obj):
        super(Logout, self).__init__()
        self._main_obj = main_obj

    def buttonClick(self, event):
        self._main_obj.logout()


class EnvPopup(IClickListener):
    def __init__(self, topbar_obj):
        super(EnvPopup, self).__init__()
        self._c = topbar_obj

    def buttonClick(self, event):
        EnvWindow(self._c._parent_window, self._c._sql_session,
                  self._c._workload)


class HostPopup(IClickListener):
    def __init__(self, topbar_obj):
        super(HostPopup, self).__init__()
        self._c = topbar_obj

    def buttonClick(self, event):
        HostWindow(self._c._parent_window, self._c._parent_window,
                   self._c._sql_session, self._c._workload)


class CalPopup(IClickListener):
    def __init__(self, topbar_obj):
        super(CalPopup, self).__init__()
        self._c = topbar_obj

    def buttonClick(self, event):
        CalendarWindow(self._c._parent_window, self._c._parent_window,
                       self._c._sql_session, self._c._workload)
