# Schedwi
# Copyright (C) 2012 Herve Quatremain
#
# This file is part of Schedwi.
#
# Schedwi is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
#
# Schedwi is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

"""Host selection widget."""

import sqlalchemy.orm.session

from muntjac.api import NativeSelect
from muntjac.data.property import IValueChangeListener
from muntjac.data.util.indexed_container import IndexedContainer

from tables.hosts import hosts
from web.autorefresh import AutoRefresh


# Memorize the last selected item to select the same host next time
_LAST_SELECTED = None


class SelectHost(NativeSelect, IValueChangeListener):

    """Host selection widget."""

    def __init__(self, sql_session, label=None, descr=None, host_id=None):
        """Build the job/jobset selection widget.

        @param sql_session:
                    SQLAlchemy session (it can be an opened session)
        @param label:
                    Selection box caption.
        @param descr:
                    Tooltip.
        @param host_id:
                    Database ID of the host to select by default.
        """
        global _LAST_SELECTED
        super(SelectHost, self).__init__()

        # Container
        c = IndexedContainer()
        c.addContainerProperty('host', str, None)
        if not isinstance(sql_session, sqlalchemy.orm.session.Session):
            session = sql_session.open_session()
            session.expire_on_commit = False
        else:
            session = sql_session
        query = session.query(hosts)
        query = query.order_by(hosts.hostname)
        for h in query.all():
            item = c.addItem(h.id)
            hostname = h.hostname.encode('utf-8')
            if isinstance(h.portnum, int):
                portnum = str(h.portnum)
            else:
                portnum = h.portnum.encode('utf-8')
            s = '%s (%s/SSL)' if h.sslenable else '%s (%s)'
            item.getItemProperty('host').setValue(s % (hostname, portnum))
        if not isinstance(sql_session, sqlalchemy.orm.session.Session):
            sql_session.close_session(session)

        self.setContainerDataSource(c)
        if label:
            self.setCaption(label)
        if descr:
            self.setDescription(descr)
        self.setNullSelectionAllowed(False)
        self.setItemCaptionPropertyId('host')
        if host_id is not None and c.containsId(host_id):
            self.setValue(host_id)
        elif _LAST_SELECTED is not None and c.containsId(_LAST_SELECTED):
            self.setValue(_LAST_SELECTED)
        else:
            self.setValue(c.firstItemId())
        self.addListener(self, IValueChangeListener)

    def get_selected_name(self):
        host_id = self.getValue()
        if not host_id:
            return ''
        else:
            c = self.getContainerDataSource()
            return c.getItem(host_id).getItemProperty('host').getValue()

    def valueChange(self, event):
        """Callback when host is selected from the list."""
        AutoRefresh.reset()
        global _LAST_SELECTED
        _LAST_SELECTED = self.getValue()
