# Schedwi
# Copyright (C) 2013 Herve Quatremain
#
# This file is part of Schedwi.
#
# Schedwi is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
#
# Schedwi is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

"""Login window."""

import logging

from muntjac.api import (Window, Panel, VerticalLayout, HorizontalLayout,
                         Button, Alignment, Label)
from muntjac.ui.button import IClickListener
from muntjac.ui.login_form import LoginForm, ILoginListener
from muntjac.ui.window import Notification
from muntjac.ui.themes import BaseTheme

from babel.core import Locale

from web.schedwipam import pam_check_user
import locale_utils
import parse_config


class LoginWindow(Window, ILoginListener):

    """Login form."""

    def __init__(self, application):
        """Create the login window.

        @param application:
                    the L{web.main.Main} object.
        """
        super(LoginWindow, self).__init__()
        self._c = application
        self.setCaption(_("Schedwi Login"))
        v = VerticalLayout()
        v.setSizeFull()
        self.addComponent(v)
        self._vertical = v
        self.login_form()

    def login_form(self):
        """Create/update the login form."""
        self._vertical.removeAllComponents()

        p = Panel()
        p.setWidth('200px')
        self._vertical.addComponent(p)
        self._vertical.setComponentAlignment(p, Alignment.MIDDLE_CENTER)

        # Login form
        login = LoginForm()
        login.setHeight('110px')
        login.setUsernameCaption(_("Username:"))
        login.setPasswordCaption(_("Password:"))
        login.setLoginButtonCaption(_("Login"))
        login.addListener(self, ILoginListener)
        self._login_form = login
        p.addComponent(login)

        # Languages selection
        h = HorizontalLayout()
        h.setSpacing(True)
        self._vertical.addComponent(h)
        self._vertical.setComponentAlignment(h, Alignment.MIDDLE_CENTER)

        for trans in locale_utils.available_translations:
            try:
                l = Locale.parse(trans)
            except:
                caption = trans
            else:
                caption = l.display_name.encode('utf-8')
            b = Button(caption)
            b.setStyleName(BaseTheme.BUTTON_LINK)
            b.addListener(LanguageButton(self, trans), IClickListener)
            h.addComponent(b)
        b = Button(_("more..."))
        b.setStyleName(BaseTheme.BUTTON_LINK)
        b.addListener(MoreLanguagesButton(self), IClickListener)
        h.addComponent(b)

    def _authenticate(self, username, password):
        if (username == parse_config.GUI_USERNAME and
            password == parse_config.GUI_PASSWORD):
            logging.info(_("%s authenticated (local)") % username)
            self._c.setUser(username)
            return True
        st, msg = pam_check_user(username, password)
        if msg:
            logging.info("%s (%s)" % (msg, username))
        if st is True:
            self._c.setUser(username)
        return st

    def onLogin(self, event):
        """Callback to the L{muntjac.ui.login_form.LoginForm} object."""
        username = event.getLoginParameter('username')
        password = event.getLoginParameter('password')
        if self._authenticate(username, password):
            self._c.state_select_activity()
        else:
            self.showNotification(_("Login incorrect"),
                                  Notification.TYPE_ERROR_MESSAGE)


class LanguageButton(IClickListener):

    """Callback for the languages buttons."""

    def __init__(self, login_window_obj, lang, window_to_close=None):
        """Initialize the callback.

        @param login_window_obj:
                    the associated L{LoginWindow} object.
        @param lang:
                    locale associated with the button.
        @param window_to_close:
                    popup window to close.
        """
        super(LanguageButton, self).__init__()
        self._c = login_window_obj
        self._lang = lang
        self._window_to_close = window_to_close

    def buttonClick(self, event):
        locale_utils.set_locale(self._lang)
        self._c.login_form()
        if self._window_to_close is not None:
            self._c.removeWindow(self._window_to_close)


class MoreLanguagesButton(IClickListener):

    """Callback for the more... button."""

    def __init__(self, login_window_obj):
        """Initialize the callback.

        @param login_window_obj:
                    the associated L{LoginWindow} object.
        """
        super(MoreLanguagesButton, self).__init__()
        self._c = login_window_obj

    def buttonClick(self, event):
        TerritoriesPopup(self._c)


class TerritoriesPopup(Window):

    """Territory/language selection."""

    _COLS = 3

    def __init__(self, login_window_obj):
        """Create the popup.

        @param login_window_obj:
                    the associated L{LoginWindow} object.
        """
        super(TerritoriesPopup, self).__init__()
        self._c = login_window_obj

        # List of available translations.  If the locale of a country
        # matches a language in this list then the language will be
        # displayed for selection.
        available = dict()
        for trans in locale_utils.available_translations:
            l = Locale.parse(trans)
            available[l.language] = l.languages[l.language].encode('utf-8')

        # Territories are organized in columns.  GridLayout cannot be used
        # because an HorizontalLayout cannot be set in a cell.
        nb = len(locale_utils.territories)
        rows = nb / self._COLS + 1
        g = [VerticalLayout() for x in range(self._COLS * 2)]

        for i in range(nb):
            # Country
            l = Label(locale_utils.territories[i][0])
            g[i / rows * 2].addComponent(l)

            # Languages
            lst = list()
            for n in locale_utils.territories[i][1:]:
                loc = Locale.parse(n)
                if loc.language in available:
                    lst.append([available[loc.language], n])
            if not lst:
                lst.append([available['en'], locale_utils.territories[i][1]])
            h = HorizontalLayout()
            h.setMargin(False)
            h.setSpacing(True)
            h.addComponent(Label(':'))
            first = True
            for lang in lst:
                if not first:
                    h.addComponent(Label('|'))
                else:
                    first = False
                b = Button(lang[0])
                b.setStyleName(BaseTheme.BUTTON_LINK)
                b.addListener(LanguageButton(login_window_obj, lang[1], self),
                              IClickListener)
                h.addComponent(b)
            g[i / rows * 2 + 1].addComponent(h)

        # Add all the columns in an HorizontalLayout
        h = HorizontalLayout()
        h.setMargin(False)
        h.setSpacing(True)
        for v in g:
            v.setMargin(True)
            v.setSpacing(False)
            h.addComponent(v)

        self.setCaption(_("Choose your Country"))
        self.setContent(h)
        login_window_obj.addWindow(self)
        self.center()
