# Schedwi
# Copyright (C) 2012 Herve Quatremain
#
# This file is part of Schedwi.
#
# Schedwi is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
#
# Schedwi is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

"""Calendar selection UI."""

from muntjac.api import Button, Window, HorizontalLayout, Alignment
from muntjac.ui.button import IClickListener
from muntjac.ui.window import Notification, ICloseListener

import sql_hierarchy
import path_cal
from web.jobwidgets import JobWidget
from web.selectcalwidget import SelectCalendar
from web.autorefresh import AutoRefresh


class Calendar(JobWidget):

    """Calendar selection button."""

    def __init__(self, jobset_ids, application_window,
                 sql_session, workload=None, job_id=None):
        """Build the widget.

        @param jobset_ids:
                    IDs (list) of the parent jobsets.
        @param application_window:
                    application window which is used to display the "select
                    calendar" popup.
        @param sql_session:
                    SQLAlchemy session.
        @param workload:
                    workload to consider.
        @param job_id:
                    ID of the edited job/jobset.
        """
        super(Calendar, self).__init__(jobset_ids, sql_session,
                                       workload, job_id)
        self._application_window = application_window

        # Retrieve the current and default value
        ids = list(jobset_ids)
        if job_id is not None:
            ids.append(job_id)
        from_obj, obj = sql_hierarchy.get_calendar_id(sql_session, ids,
                                                      workload)
        if job_id is None or not from_obj:
            self._obj = None
            self._default_obj = obj
        else:
            self._obj = obj
            foo, self._default_obj = sql_hierarchy.get_calendar_id(
                                                             sql_session,
                                                             jobset_ids,
                                                             workload)
        # self._default_obj is None if the root jobset is the one edited
        if self._default_obj:
            p = path_cal.PathCal(sql_session, workload=workload,
                                 id=self._default_obj.cal_id)
            self._default_path = unicode(p).encode('utf-8')
        self._savedValue = None
        self._w = Button()
        self._w.setWidth('100%')
        self._w.setDescription(_("Calendar name"))
        self._w.setImmediate(True)
        self._w.addListener(LaunchPopup(self), IClickListener)
        self._addInheritedWidget(self._w)

        if self._obj is not None:
            p = path_cal.PathCal(sql_session, workload=workload,
                                 id=self._obj.cal_id)
            #self._w.setCaption(str(p))
            self._w.setCaption(unicode(p).encode('utf-8'))
            self._displayed_cal_id = self._obj.cal_id
        else:
            self._savedValue = self._default_path
            self._saved_displayed_cal_id = self._default_obj.cal_id
            self._setDefaultValue()
            self._inherited.setValue(True)
            self._doGreyOut()

    def getFieldWidget(self):
        return self._w

    def _setDefaultValue(self):
        self._w.setCaption(self._default_path)
        self._displayed_cal_id = self._default_obj.cal_id

    def _saveValue(self):
        self._savedValue = self._w.getCaption()
        self._saved_displayed_cal_id = self._displayed_cal_id

    def _restoreValue(self):
        self._w.setCaption(self._savedValue)
        self._displayed_cal_id = self._saved_displayed_cal_id

    def setNewValue(self, path):
        """Set the button caption."""
        self._w.setCaption(unicode(path).encode('utf-8'))
        self._displayed_cal_id = path.id[-1]

    def get_value(self):
        if self.isInherited():
            return 0
        else:
            return self._displayed_cal_id

    def updateJob(self, job):
        job.cal_id = self.get_value()
        return None


class LaunchPopup(IClickListener):

    """Callback for the calendar button."""

    def __init__(self, calendar_obj):
        super(LaunchPopup, self).__init__()
        self._c = calendar_obj

    def buttonClick(self, event):
        AutoRefresh.reset()
        #SelectPopup(self._c, event.getClientX(), event.getClientY())
        SelectPopup(self._c)
        # Disable the calendar button
        self._c._w.setEnabled(False)


class SelectPopup(IClickListener, ICloseListener):

    """Calendar selection popup."""

    _bt_captions = [_('Cancel'), _('OK')]

    def __init__(self, calendar_obj, x=None, y=None):
        """Create the popup.

        @param calendar_obj:
                the assosiated L{Calendar} object.
        @param x:
                the X coordinate of the mouse which is used to position
                the popup.  If not set, the popup is centered.
        @param y:
                the Y coordinate of the mouse which is used to position
                the popup.  If not set, the popup is centered.
        """
        super(SelectPopup, self).__init__()

        self._c = calendar_obj

        self._w = Window(_("Calendars"))
        self._w.setWidth('250px')
        self._w.setHeight('404px')
        self._w.addListener(self, ICloseListener)

        # VerticalLayout as content by default
        v = self._w.getContent()
        v.setSizeFull()
        v.setMargin(True)
        v.setSpacing(True)

        t = SelectCalendar(self._c._sql_session, self._c._workload,
                           self._c._displayed_cal_id)
        t.setSizeFull()
        self._select_cal = t
        v.addComponent(t)
        v.setExpandRatio(t, 1.0)

        # Button box
        h_bt = HorizontalLayout()
        h_bt.setMargin(False)
        h_bt.setSpacing(True)

        for caption in self._bt_captions:
            b = Button(_(caption))
            b.addListener(self, IClickListener)
            h_bt.addComponent(b)

        v.addComponent(h_bt)
        v.setComponentAlignment(h_bt, Alignment.BOTTOM_RIGHT)

        self._c._application_window.addWindow(self._w)
        if x and y:
            self._w.setPositionX(x)
            self._w.setPositionY(y)
        else:
            self._w.center()

    def buttonClick(self, event):
        AutoRefresh.reset()
        # First button is Cancel
        if event.getButton().getCaption() != _(self._bt_captions[0]):
            v = self._select_cal.getSelected()
            if v is None:
                self._c._application_window.showNotification(
                            _("A calendar must be selected"),
                            _("<br/>Make sure that a calendar has been \
                             selected (and not a directory)"),
                            Notification.TYPE_ERROR_MESSAGE)
                return
            p = path_cal.PathCal(self._c._sql_session,
                                 workload=self._c._workload,
                                 id=v)
            self._c.setNewValue(p)
        # Close the window
        self._c._application_window.removeWindow(self._w)

    def windowClose(self, event):
        """Callback when the popup window is closed."""
        AutoRefresh.reset()
        # Re-anable the calendar button
        self._c._w.setEnabled(True)
