# Schedwi
# Copyright (C) 2012, 2013 Herve Quatremain
#
# This file is part of Schedwi.
#
# Schedwi is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
#
# Schedwi is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

"""Message boxes."""

from muntjac.api import Window, HorizontalLayout, Alignment, Label, Button
from muntjac.terminal.theme_resource import ThemeResource
from muntjac.ui.embedded import Embedded
from muntjac.ui.button import IClickListener


class _MessageBox(IClickListener):

    def __init__(self, parent, title, icon, buttons,
                 primary_text, secondary_text=None):
        """Create a new Message popup.

        @param parent:
                     the parent window (required to display the popup)
        @param title:
                     the title of the windows displayed in the header.
        @param icon:
                     icon to show near the text. The provided name is used
                     to locate the icon image file inthe theme directory:
                     icons/I{icon}.png
        @param buttons:
                     The list of button captions.
        @type buttons: list
        @param primary_text:
                     primary message to be displayed (in bold)
        @param secondary_text:
                     other text message to be displayed.
        """
        super(_MessageBox, self).__init__()

        self._parent = parent
        self._buttons = buttons
        self._listener = None

        self._w = Window(title)
        self._w.setModal(True)

        # VerticalLayout as content by default
        v = self._w.getContent()
        v.setSizeUndefined()
        v.setMargin(True)
        v.setSpacing(True)

        # For the icon and the text message
        h = HorizontalLayout()
        h.setWidth('300px')
        h.setMargin(False)
        h.setSpacing(True)

        e = Embedded(source=ThemeResource('icons/' + icon + '.png'))
        h.addComponent(e)
        h.setComponentAlignment(e, Alignment.MIDDLE_LEFT)

        s = '<h3>%s</h3>%s' % (primary_text,
                               secondary_text if secondary_text else '')
        l = Label(s)
        l.setContentMode(Label.CONTENT_XHTML)
        l.setWidth('100%')
        h.addComponent(l)
        h.setExpandRatio(l, 1.0)

        v.addComponent(h)
        v.setExpandRatio(h, 1.0)

        # Button box
        h_bt = HorizontalLayout()
        h_bt.setMargin(False)
        h_bt.setSpacing(True)

        for bt_label in self._buttons:
            b = Button(bt_label)
            b.addListener(self, IClickListener)
            h_bt.addComponent(b)

        v.addComponent(h_bt)
        v.setComponentAlignment(h_bt, Alignment.BOTTOM_RIGHT)

        self._parent.getWindow().addWindow(self._w)

    def buttonClick(self, event):
        """Callback when one of the button is clicked.

        @param event:
               fired event
        """
        # Close the window first
        self._parent.getWindow().removeWindow(self._w)
        if event.getButton().getCaption() != self._buttons[0] and \
                self._listener:
            self._listener.boxClose(event.getButton().getCaption())

    def addListener(self, listener):
        """Add the popup close listener.

        @param listener:
                    the Listener to be added.
        @type listener: BoxCloseListener
        """
        self._listener = listener


class BoxCloseListener(object):

    """Interface for listening for a button of the popup."""

    def boxClose(self, event):
        """Called when a popup button has been clicked.  The caption of
           the button is given in event.

           @param event:
                    the caption of the clicked button.
           @type event: string
        """
        raise NotImplementedError


class DeleteBox(_MessageBox):

    """Popup to ask confirmation of a delete operation."""

    def __init__(self, parent, title, primary_text, secondary_text=None):
        """Create a new Delete popup.

        @param parent:
                    the parent window (required to display the popup)
        @param title:
                    the title of the windows displayed in the header.
        @param primary_text:
                     primary message to be displayed (in bold)
        @param secondary_text:
                     other text message to be displayed.
        """
        super(DeleteBox, self).__init__(parent, title, 'warning',
                                        [_("Cancel"), _("Delete")],
                                        primary_text, secondary_text)


class QuestionBox(_MessageBox):

    """Popup to ask something to the user."""

    def __init__(self, parent, title, primary_text, secondary_text=None):
        """Create a new Question popup.

        @param parent:
                    the parent window (required to display the popup)
        @param title:
                    the title of the windows displayed in the header.
        @param primary_text:
                     primary message to be displayed (in bold)
        @param secondary_text:
                     other text message to be displayed.
        """
        super(QuestionBox, self).__init__(parent, title, 'info',
                                        [_("Cancel"), _("Continue")],
                                        primary_text, secondary_text)
