# Schedwi
# Copyright (C) 2011, 2012 Herve Quatremain
#
# This file is part of Schedwi.
#
# Schedwi is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
#
# Schedwi is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.


"""Module to remove jobs and jobsets."""

import sys
import getopt

import path
from tables.job_main import job_main
from tables.link import link
from help import print_trim


def usage():
    """Print a usage message on STDOUT."""
    print_trim(_("""Usage: rm [OPTION]... JOB|JOBSET...
        Remove each specified job.  By default, it does not remove jobsets
        unless the -r, -R, or --recursive options are given.

        Options:
          -f, --force          never prompt for confirmation.
          -r, -R, --recursive  remove jobsets and their contents recursively.
          -h, --help           display this help.
    """))


def rm_recursive(session, p, force, recursive, cwd=None):
    """Remove a job/jobset.

    Arguments:
    session -- SQLAlchemy session
    p -- job or jobset to remove
    force -- whether or not to remove the job or jobset without confirmation
    recursive -- whether or not recursively remove jobsets.

    Return:
    0 -- job/jobset removed.
    1 -- the item to remove is a jobset and the recursive (-r) flag is not set.
    2 -- only some of the children job/jobset have been removed.

    """
    # Get the list of all the jobs/jobsets that have links to this job/jobset
    q = session.query(job_main.id).filter(job_main.id == link.job_id_source)
    q = q.filter(link.job_id_destination == p.id[-1])
    links = list()
    for id, in q.all():
        links.append(path.id2path(session, id))
    try:
        job = session.query(job_main).filter_by(id=p.id[-1]).one()
    except:
        sys.stderr.write(_("rm: no such job or jobset\n"))
        return 1
    # Job
    if job.type == 1:
        if force == False:
            if links:
                print _("The following jobs/jobsets have links to this job:")
                print "\n".join(map(lambda i: i.path, links))
            answer = raw_input(_("rm: remove job `%s'? ") %
                               unicode(p).encode('utf-8'))
            if answer[0] not in (_('y'), _('Y')):
                return 2
        # Remove the links
        if links:
            q = session.query(link).filter(link.job_id_destination == p.id[-1])
            map(lambda i: session.delete(i), q.all())
        # Remove the job
        session.delete(job)
        return 0
    # Jobset
    if recursive == False:
        sys.stderr.write(_("rm: cannot remove `%s': Is a jobset\n") % p)
        return 1
    if force == False:
        if links:
            print _("The following jobs/jobsets have links to this jobset:")
            print "\n".join(map(lambda i: i.path, links))
        answer = raw_input(_("rm: descend into jobset `%s'? ") %
                           unicode(p).encode('utf-8'))
        if answer[0] not in (_('y'), _('Y')):
            return 2
    # Remove the links
    if links:
        q = session.query(link).filter(link.job_id_destination == p.id[-1])
        map(lambda i: session.delete(i), q.all())
    # Recursivly remove all the children
    error = False
    empty = True
    for child in session.query(job_main).filter_by(parent=p.id[-1]):
        ids = list(p.id)
        ids.append(child.id)
        j = path.Path(session, id=ids)
        ret = rm_recursive(session, j, force, recursive)
        if ret == 1:
            error = True
        elif ret == 2:
            empty = False
    if error:
        return 1
    if empty:
        try:
            idx = cwd.id.index(p.id[-1])
        except:
            pass
        else:
            pass
        session.delete(job)
        return 0
    return 2


def rm(sql_session, current_cwd, arguments):
    """Remove jobs and jobsets.

    Arguments:
    sql_session -- SQLAlchemy session
    current_cwd -- current working jobset (a path.Path object)
    arguments -- list of arguments given to the rm command (list
                 of jobs/jobsets)

    """
    try:
        optlist, args = getopt.getopt(arguments, "frRh",
                                      ["force", "recursive", "help"])
    except getopt.GetoptError, err:
        sys.stderr.write(_("rm: ") + str(err) + "\n")
        return 1
    force = False
    recursive = False
    for o, a in optlist:
        if o in ("-f", "--force"):
            force = True
        elif o in ("-r", "-R", "--recursive"):
            recursive = True
        elif o in ("-h", "--help"):
            usage()
            return 0
    paths = list()
    if args:
        for arg in args:
            paths.extend(path.get_paths(sql_session, arg, current_cwd))
    else:
        if force == False:
            sys.stderr.write(_("rm: missing job or jobset name\n"))
            return 1
        return 0
    if not paths:
        if force == False:
            sys.stderr.write(_("rm: no such job or jobset\n"))
            return 1
        return 0
    error = False
    session = sql_session.open_session()
    for p in paths:
        try:
            ret = rm_recursive(session, p, force, recursive)
        except:
            sql_session.close_session(session)
            raise
        if ret == 1:
            error = True
    sql_session.close_session(session)
    return 1 if error else 0
