# Schedwi
# Copyright (C) 2012, 2013 Herve Quatremain
#
# This file is part of Schedwi.
#
# Schedwi is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
#
# Schedwi is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.


"""Module to find jobs and jobsets."""

import getopt
import time
import sys
import datetime

from sqlalchemy import or_

import path
import cmd_cal.calcomp
from tables.job_main import job_main
from tables.job_main_s import job_main_s
from tables.hosts import hosts
from tables.job_host import job_host
from tables.job_host_s import job_host_s
from tables.environments import environments
from tables.environments_s import environments_s
from tables.job_environment import job_environment
from tables.job_environment_s import job_environment_s
from tables.calendars import calendars
from tables.calendars_s import calendars_s
from tables.job_command import job_command
from tables.job_command_s import job_command_s
from tables.job_username import job_username
from tables.job_username_s import job_username_s
from tables.job_file_out import job_file_out
from tables.job_file_out_s import job_file_out_s
from tables.job_file_err import job_file_err
from tables.job_file_err_s import job_file_err_s
from tables.job_control_group import job_control_group
from tables.job_control_group_s import job_control_group_s
from tables.job_max_duration import job_max_duration
from tables.job_max_duration_s import job_max_duration_s
from tables.job_retries_interval import job_retries_interval
from tables.job_retries_interval_s import job_retries_interval_s
from tables.job_start_limit import job_start_limit
from tables.job_start_limit_s import job_start_limit_s
from tables.job_success_return_code import job_success_return_code
from tables.job_success_return_code_s import job_success_return_code_s
from tables.job_retries import job_retries
from tables.job_retries_s import job_retries_s
from tables.job_arguments import job_arguments
from tables.job_arguments_s import job_arguments_s
from tables.job_loadenv import job_loadenv
from tables.job_loadenv_s import job_loadenv_s
from tables.job_detach import job_detach
from tables.job_detach_s import job_detach_s
from tables.job_manual import job_manual
from tables.job_manual_s import job_manual_s
from tables.job_manual_command import job_manual_command
from tables.job_manual_command_s import job_manual_command_s
from tables.job_stat import job_stat
from tables.constraint_file import constraint_file
from tables.constraint_file_s import constraint_file_s
from tables.link import link
from tables.link_s import link_s
from tables.calendars import calendars
from tables.calendars_s import calendars_s
from help import print_trim


def usage():
    """Print a usage message on STDOUT."""
    print_trim(_("""Usage: find [OPTION]...
Search for jobs and jobsets.

Options:
  --name=PATTERN        job or jobset name to search. See PATTERN at the end.
  --id=N                job or jobset internal database ID.
  --type=C              limit the search to jobs (if C is `j') or jobsets (`d')
  --enable=True|False   search for Enabled (True) or Disabled (False) objects.
  --host=PATTERN        search by host name.
  --calendar=PATTERN    search for jobs and jobsets with the specified
                        calendar.
  --start=TIME          search by start time (HH:MM).
  --command=PATTERN     search by command name.
  --parameter=PATTERN   search in the command parameters.
  --return=N            search by the return code (equal or below means
                        success)
  --user=PATTERN        search by the username under which the command is run.
  --stdout=PATTERN      search by the output file.
  --stderr=PATTERN      search by the error file.
  --detach=True|False   search by whether the job must be run in detach mode,
                        to start a daemon process for instance, or not.
  --manual=True|False   search by whether a confirmation is asked before
                        starting the job, or not.
  --manual-cmd=PATTERN  search by the alert command for manual jobs.
  --cgroup=PATTERN      search by the Linux Control Group name.
  --duration=MINUTES    search by the maximum allowed duration.
                        0 means no limit.
  --retries=N           search by the number of retries.
  --retries-int=MINUTES search by the number of minutes between retries.
  --limit=MINUTES       search by the start limit (number of minutes for the
                        job to be started past its start time)
                        0 means no limit.
  --user-env=True|False search by whether the user account environment must be
                        loaded (True) or not (False) before running the
                        command.
  --has-env=True|False  environment groups are enabled (True) or not (False)
  --env=PATTERN         search by environment group name.
  --file=PATTERN        search by constraint file (file name to check before
                        starting a job)
  --description=PATTERN search in the job/jobset description.
  --link=JOB|JOBSET     search for jobs and jobsets that depend of the provided
                        job/jobset.
  --success=N           search by the number of successful run.
  --failed=N            search by the number of failed run.
  --avg-duration=MINUTES search by average duration.
  --day=MM/DD[/[YY]YY]  search jobs and jobsets that are planned for the
                        specified day.
  -h, --help            display this help.

PATTERN  this is the parameter string to look for. The metacharacters `*'
         and `?' can be used.
N        numeric value to search.  N can be specified as
              +N  for greater that N,
              -N  for less than N,
               N  for exactly N.
MINUTES  number of minutes.  As for numeric values, MINUTES can be specified
         as +MINUTES (more than), -MINUTES (less than) or MINUTES (exactly)
         Also the unit (minute) can be changed with the following suffixes:
              `h'  for hours,
              `d'  for days.

    """))


def _get_day(val):
    """Parse the provided DD/MM/YYYY value.

    @param val:
                the MM/DD/YYYY value to parse.
    @return:
                the tuple (DD, MM, YYYY)
    @raise ValueError:
                the provided value is not a date
    """
    try:
        m, d, y = val.strip().split('/')
    except:
        m, d = val.strip().split('/')
        day = int(d)
        month = int(m)
        now = datetime.datetime.now()
        year = now.year
        if month < now.month or (month == now.month and day < now.day):
            year += 1
    else:
        day = int(d)
        month = int(m)
        year = int(y)
        if year < 100:
            year += 2000
    if month < 1 or month > 12 or day < 1 or day > 31:
        raise ValueError
    return (day, month, year)


def _filter_minutes(query, attr, val, zero_means_no_limit=False):
    """Filter a number of minutes.

    @param query:
                the SQLAlchemy query object to filter.
    @param attr:
                the attribute to filter.
    @param val:
                the value to parse and filter.
    @param zero_means_no_limit:
                if val is zero, it means no limit.
    @return:
                the new filtered query.
    @raise ValueError:
                the provided value is not a number
    """
    val = val.strip().lower()
    multiply = 1
    try:
        i = val.index('h')
    except:
        try:
            i = val.index('d')
        except:
            i = len(val)
        else:
            multiply = 1440
    else:
        multiply = 60
    val = val[:i]
    if val:
        v = int(val)
        if val[0] == '+':
            if zero_means_no_limit:
                if v == 0:
                    return query.filter(attr == 0)
                else:
                    return query.filter(or_(attr == 0, attr > v * multiply))
            return query.filter(attr > v * multiply)
        if val[0] == '-':
            if zero_means_no_limit:
                if v == 0:
                    return query.filter(attr != 0)
                else:
                    return query.filter(attr < -v * multiply).filter(attr != 0)
            return query.filter(attr < -v * multiply)
        return query.filter(attr == v * multiply)
    return query


def _filter_time(query, attr, val):
    """Filter the time.

    @param query:
                the SQLAlchemy query object to filter.
    @param attr:
                the attribute to filter.
    @param val:
                the value to parse and filter.
    @return:
                the new filtered query.
    @raise ValueError:
                the provided value is not a proper time.
    """
    plus = minus = False
    val = val.strip()
    if val[0] == '+':
        plus = True
        val = val[1:]
    elif val[0] == '-':
        minus = True
        val = val[1:]
    for fmt in ("%H:%M", "%I:%M %p", "%Hh%M", "%Ih%M %p"):
        try:
            tstr = time.strptime(val, fmt)
        except ValueError:
            continue
        else:
            break
    else:
        raise ValueError
    if plus:
        return query.filter(attr > tstr.tm_hour * 100 + tstr.tm_min)
    if minus:
        return query.filter(attr < tstr.tm_hour * 100 + tstr.tm_min)
    return query.filter(attr == tstr.tm_hour * 100 + tstr.tm_min)


def _filter_boolean(query, attr, val):
    """Filter a boolean.

    @param query:
                the SQLAlchemy query object to filter.
    @param attr:
                the attribute to filter.
    @param val:
                the value to parse and filter.
    @return:
                the new filtered query.
    """
    val = val.strip()
    if val and val[0] in (_('Y'), _('y'), '1', _('t'), _('T')):
        return query.filter(attr != 0)
    else:
        return query.filter(attr == 0)


def _filter_type(query, attr, val):
    """Filter by type (job or jobset).

    @param query:
                the SQLAlchemy query object to filter.
    @param attr:
                the attribute to filter.
    @param val:
                the value to parse and filter.
    @return:
                the new filtered query.
    @raise ValueError:
                the provided value is not `j' nor `d'.
    """
    if 'j' in val.lower():
        # Job
        return query.filter(attr == 1)
    elif 'd' in val.lower():
        # Jobset
        return query.filter(attr == 0)
    else:
        raise ValueError


def _filter_pattern(query, attr, val):
    """Filter a pattern.

    @param query:
                the SQLAlchemy query object to filter.
    @param attr:
                the attribute to filter.
    @param val:
                the value to parse and filter.
    @return:
                the new filtered query.
    """
    val = val.strip().replace("*", "%").replace("?", "_")
    return query.filter(attr.like(val)) if val else query


def _filter_numeric(query, attr, val):
    """Filter a numeric value.

    @param query:
                the SQLAlchemy query object to filter.
    @param attr:
                the attribute to filter.
    @param val:
                the value to parse and filter.
    @return:
                the new filtered query.
    @raise ValueError:
                the provided value is not a number
    """
    val = val.strip()
    if val:
        v = int(val)
        if val[0] == '+':
            return query.filter(attr > v)
        if val[0] == '-':
            return query.filter(attr < -v)
        return query.filter(attr == v)
    return query


def find(sql_session, current_cwd, arguments, workload=None):
    """Find job and jobset.

    @param sql_session:
                SQLAlchemy session
    current_cwd -- current working jobset (a path.Path object)
    arguments -- list of arguments given to the tree command (list
                 of jobs/jobsets)
    workload -- workload to use

    """
    try:
        optlist, args = getopt.getopt(arguments, "h",
                        [
                            "name=",
                            "id=",
                            "type=",
                            "enable=",
                            "host=",
                            "calendar=",
                            "start=",
                            "command=",
                            "parameter=",
                            "return=",
                            "user=",
                            "stdout=",
                            "stderr=",
                            "detach=",
                            "manual=",
                            "manual-cmd=",
                            "cgroup=",
                            "duration=",
                            "retries=",
                            "retries-int=",
                            "limit=",
                            "user-env=",
                            "has-env=",
                            "env=",
                            "file=",
                            "description=",
                            "link=",
                            "success=",
                            "failed=",
                            "avg-duration=",
                            "day=",
                            "help"
                        ])
    except getopt.GetoptError, err:
        sys.stderr.write(_("find: ") + str(err) + "\n")
        return 1
    if not optlist:
        return 0
    session = sql_session.open_session()
    if workload is None:
        query = session.query(job_main)
    else:
        query = session.query(job_main_s)
        query = query.filter(job_main_s.workload_date == workload)

    for o, a in optlist:
        a = a.decode('utf-8')
        if o == "--name":
            if workload is None:
                query = _filter_pattern(query, job_main.name, a)
            else:
                query = _filter_pattern(query, job_main_s.name, a)

        elif o == "--id":
            try:
                if workload is None:
                    query = _filter_numeric(query, job_main.id, a)
                else:
                    query = _filter_numeric(query, job_main_s.id, a)
            except:
                sql_session.cancel_session(session)
                sys.stderr.write(_("find: invalid number (--id=%s)\n") % a)
                return 1

        elif o == "--type":
            try:
                if workload is None:
                    query = _filter_type(query, job_main.type, a)
                else:
                    query = _filter_type(query, job_main_s.type, a)
            except:
                sql_session.cancel_session(session)
                sys.stderr.write(
                        _("find: type must be `j' or `d' (--type=%s)\n") % a)
                return 1

        elif o == "--enable":
            if workload is None:
                query = _filter_boolean(query, job_main.enabled, a)
            else:
                query = _filter_boolean(query, job_main_s.enabled, a)

        elif o == "--host":
            query = _filter_pattern(query, hosts.hostname, a)
            if workload is None:
                query = query.filter(job_host.job_id == job_main.id)
                query = query.filter(job_host.host_id == hosts.id)
            else:
                query = query.filter(job_host_s.job_id == job_main_s.id)
                query = query.filter(job_host_s.host_id == hosts.id)
                query = query.filter(job_host_s.workload_date == workload)

        elif o == "--calendar":
            if workload is None:
                query = _filter_pattern(query, calendars.name, a)
                query = query.filter(calendars.id == job_main.cal_id)
            else:
                query = _filter_pattern(query, calendars_s.name, a)
                query = query.filter(calendars_s.id == job_main_s.cal_id)
                query = query.filter(calendars_s.workload_date == workload)

        elif o == "--start":
            try:
                if workload is None:
                    query = _filter_time(query, job_main.start_time, a)
                else:
                    query = _filter_time(query, job_main_s.start_time, a)
            except:
                sql_session.cancel_session(session)
                sys.stderr.write(_("find: unknown time format (--start=%s). \
Should be `H:M' or `H:M AM/PM'\n") % a)
                return 1

        elif o == "--command":
            if workload is None:
                query = _filter_pattern(query, job_command.command, a)
                query = query.filter(job_command.job_id == job_main.id)
            else:
                query = _filter_pattern(query, job_command_s.command, a)
                query = query.filter(job_command_s.job_id == job_main_s.id)
                query = query.filter(job_command_s.workload_date == workload)

        elif o == "--parameter":
            if workload is None:
                query = _filter_pattern(query, job_arguments.argument, a)
                query = query.filter(job_arguments.job_id == job_main.id)
            else:
                query = _filter_pattern(query, job_arguments_s.argument, a)
                query = query.filter(job_arguments_s.job_id == job_main_s.id)
                query = query.filter(job_arguments_s.workload_date == workload)

        elif o == "--return":
            try:
                if workload is None:
                    query = _filter_numeric(query,
                                        job_success_return_code.success_ret,
                                        a)
                    query = query.filter(job_success_return_code.job_id ==
                                         job_main.id)
                else:
                    query = _filter_numeric(query,
                                        job_success_return_code_s.success_ret,
                                        a)
                    query = query.filter(job_success_return_code_s.job_id ==
                                         job_main_s.id)
                    query = query.filter(
                        job_success_return_code_s.workload_date == workload)
            except:
                sql_session.cancel_session(session)
                sys.stderr.write(_("find: invalid number (--return=%s)\n") % a)
                return 1

        elif o == "--user":
            if workload is None:
                query = _filter_pattern(query, job_username.username, a)
                query = query.filter(job_username.job_id == job_main.id)
            else:
                query = _filter_pattern(query, job_username_s.username, a)
                query = query.filter(job_username_s.job_id == job_main_s.id)
                query = query.filter(job_username_s.workload_date == workload)

        elif o == "--stdout":
            if workload is None:
                query = _filter_pattern(query, job_file_out.file_out, a)
                query = query.filter(job_file_out.job_id == job_main.id)
            else:
                query = _filter_pattern(query, job_file_out_s.file_out, a)
                query = query.filter(job_file_out_s.job_id == job_main_s.id)
                query = query.filter(job_file_out_s.workload_date == workload)

        elif o == "--stderr":
            if workload is None:
                query = _filter_pattern(query, job_file_err.file_err, a)
                query = query.filter(job_file_err.job_id == job_main.id)
            else:
                query = _filter_pattern(query, job_file_err_s.file_err, a)
                query = query.filter(job_file_err_s.job_id == job_main_s.id)
                query = query.filter(job_file_err_s.workload_date == workload)

        elif o == "--cgroup":
            if workload is None:
                query = _filter_pattern(query,
                                        job_control_group.control_group, a)
                query = query.filter(job_control_group.job_id == job_main.id)
            else:
                query = _filter_pattern(query,
                                        job_control_group_s.control_group, a)
                query = query.filter(
                                job_control_group_s.job_id == job_main_s.id)
                query = query.filter(
                                job_control_group_s.workload_date == workload)

        elif o == "--duration":
            try:
                if workload is None:
                    query = _filter_minutes(query,
                                            job_max_duration.max_duration,
                                            a, True)
                    query = query.filter(job_max_duration.job_id ==
                                         job_main.id)
                else:
                    query = _filter_minutes(query,
                                            job_max_duration_s.max_duration,
                                            a, True)
                    query = query.filter(job_max_duration_s.job_id ==
                                         job_main_s.id)
                    query = query.filter(job_max_duration_s.workload_date ==
                                         workload)
            except:
                sql_session.cancel_session(session)
                sys.stderr.write(_("find: invalid number (--duration=%s)\n")
                                 % a)
                return 1

        elif o == "--retries":
            try:
                if workload is None:
                    query = _filter_numeric(query, job_retries.retries, a)
                    query = query.filter(job_retries.job_id == job_main.id)
                else:
                    query = _filter_numeric(query, job_retries_s.retries, a)
                    query = query.filter(job_retries_s.job_id == job_main_s.id)
                    query = query.filter(
                        job_retries_s.workload_date == workload)
            except:
                sql_session.cancel_session(session)
                sys.stderr.write(_("find: invalid number (--retries=%s)\n")
                                 % a)
                return 1

        elif o == "--retries-int":
            try:
                if workload is None:
                    query = _filter_minutes(query,
                                        job_retries_interval.retries_interval,
                                        a)
                    query = query.filter(job_retries_interval.job_id ==
                                         job_main.id)
                else:
                    query = _filter_minutes(query,
                                     job_retries_interval_s.retries_interval,
                                     a)
                    query = query.filter(job_retries_interval_s.job_id ==
                                         job_main_s.id)
                    query = query.filter(
                        job_retries_interval_s.workload_date == workload)
            except:
                sql_session.cancel_session(session)
                sys.stderr.write(_("find: invalid number (--retries-int=%s)\n")
                                 % a)
                return 1

        elif o == "--limit":
            try:
                if workload is None:
                    query = _filter_minutes(query, job_start_limit.start_limit,
                                            a, True)
                    query = query.filter(job_start_limit.job_id == job_main.id)
                else:
                    query = _filter_minutes(query,
                                            job_start_limit_s.start_limit,
                                            a, True)
                    query = query.filter(job_start_limit_s.job_id ==
                                         job_main_s.id)
                    query = query.filter(job_start_limit_s.workload_date ==
                                         workload)
            except:
                sql_session.cancel_session(session)
                sys.stderr.write(_("find: invalid number (--limit=%s)\n") % a)
                return 1

        elif o == "--user-env":
            if workload is None:
                query = _filter_boolean(query, job_loadenv.loadenv, a)
                query = query.filter(job_loadenv.job_id == job_main.id)
            else:
                query = _filter_boolean(query, job_loadenv_s.loadenv, a)
                query = query.filter(job_loadenv_s.job_id == job_main_s.id)
                query = query.filter(job_loadenv_s.workload_date == workload)

        elif o == "--detach":
            if workload is None:
                query = _filter_boolean(query, job_detach.detach, a)
                query = query.filter(job_detach.job_id == job_main.id)
            else:
                query = _filter_boolean(query, job_detach_s.detach, a)
                query = query.filter(job_detach_s.job_id == job_main_s.id)
                query = query.filter(job_detach_s.workload_date == workload)

        elif o == "--manual":
            if workload is None:
                query = _filter_boolean(query, job_manual.manual, a)
                query = query.filter(job_manual.job_id == job_main.id)
            else:
                query = _filter_boolean(query, job_manual_s.manual, a)
                query = query.filter(job_manual_s.job_id == job_main_s.id)
                query = query.filter(job_manual_s.workload_date == workload)

        elif o == "--manual-cmd":
            if workload is None:
                query = _filter_pattern(query,
                                        job_manual_command.manual_command, a)
                query = query.filter(job_manual_command.job_id == job_main.id)
            else:
                query = _filter_pattern(query,
                                        job_manual_command_s.manual_command, a)
                query = query.filter(job_manual_command_s.job_id ==
                                        job_main_s.id)
                query = query.filter(job_manual_command_s.workload_date ==
                                        workload)

        elif o == "--has-env":
            if workload is None:
                query = _filter_boolean(query, job_main.has_env, a)
            else:
                query = _filter_boolean(query, job_main_s.has_env, a)

        elif o == "--env":
            if workload is None:
                query = _filter_pattern(query, environments.name, a)
                query = query.filter(job_environment.job_id == job_main.id)
                query = query.filter(job_environment.env_id == environments.id)
            else:
                query = _filter_pattern(query, environments_s.name, a)
                query = query.filter(environments_s.workload_date ==
                                     workload)
                query = query.filter(job_environment_s.job_id == job_main_s.id)
                query = query.filter(job_environment_s.env_id ==
                                     environments_s.id)
                query = query.filter(job_environment_s.workload_date ==
                                     workload)

        elif o == "--file":
            if workload is None:
                query = _filter_pattern(query, constraint_file.filename, a)
                query = query.filter(constraint_file.job_id == job_main.id)
            else:
                query = _filter_pattern(query, constraint_file_s.filename, a)
                query = query.filter(constraint_file_s.job_id == job_main_s.id)
                query = query.filter(constraint_file_s.workload_date ==
                                     workload)

        elif o == "--description":
            if workload is None:
                query = _filter_pattern(query, job_main.description, a)
            else:
                query = _filter_pattern(query, job_main_s.description, a)

        elif o == "--link":
            try:
                p = path.Path(session, a, current_cwd, workload=workload)
            except:
                sql_session.cancel_session(session)
                sys.stderr.write(_("find: no such job or jobset (--link=%s)\n")
                                 % a)
                return 1
            if workload is None:
                query = query.filter(link.job_id_destination == p.id[-1])
                query = query.filter(link.job_id_source == job_main.id)
            else:
                query = query.filter(link_s.job_id_destination == p.id[-1])
                query = query.filter(link_s.job_id_source == job_main_s.id)
                query = query.filter(link_s.workload_date == workload)

        elif o == "--success":
            try:
                query = _filter_numeric(query, job_stat.num_success, a)
                if workload is None:
                    query = query.filter(job_stat.job_id == job_main.id)
                else:
                    query = query.filter(job_stat.job_id == job_main_s.id)
            except:
                sql_session.cancel_session(session)
                sys.stderr.write(_("find: invalid number (--success=%s)\n")
                                 % a)
                return 1

        elif o == "--failed":
            try:
                query = _filter_numeric(query, job_stat.num_failed, a)
                if workload is None:
                    query = query.filter(job_stat.job_id == job_main.id)
                else:
                    query = query.filter(job_stat.job_id == job_main_s.id)
            except:
                sql_session.cancel_session(session)
                sys.stderr.write(_("find: invalid number (--failed=%s)\n")
                                 % a)
                return 1

        elif o == "--avg-duration":
            try:
                query = query.filter(job_stat.num_success > 0)
                query = _filter_minutes(query,
                            job_stat.total_duration / job_stat.num_success, a)
                if workload is None:
                    query = query.filter(job_stat.job_id == job_main.id)
                else:
                    query = query.filter(job_stat.job_id == job_main_s.id)
            except:
                sql_session.cancel_session(session)
                sys.stderr.write(
                    _("find: invalid number (--avg-duration=%s)\n") % a)
                return 1

        elif o == "--day":
            try:
                d, m, y = _get_day(a)
            except:
                sql_session.cancel_session(session)
                sys.stderr.write(_("find: invalid date (--day=%s)\n") % a)
                return 1
            # Retrieve all the calendar formulas
            if workload is None:
                q = session.query(calendars)
            else:
                q = session.query(calendars_s)
                q = q.filter(calendars_s.workload_date == workload)
            cals = list()
            for c in q.all():
                err, idx, cal = cmd_cal.calcomp.str2cal(c.formula, y)
                if (err == cmd_cal.calcomp.CAL_NOERROR and
                    cal.calmatch(d, m, y) == cmd_cal.calcomp.CAL_NOERROR):
                    cals.append(c.id)
            if workload is None:
                query = query.filter(job_main.cal_id.in_(cals))
            else:
                query = query.filter(job_main_s.cal_id.in_(cals))

        elif o in ("-h", "--help"):
            sql_session.cancel_session(session)
            usage()
            return 0

    if workload is None:
        query = query.order_by(job_main.parent, job_main.type, job_main.name)
    else:
        query = query.order_by(job_main_s.parent, job_main_s.type,
                               job_main_s.name)
    for job in query.all():
        p = path.id2path(session, job.id, workload)
        print unicode(p).encode('utf-8')
    sql_session.close_session(session)
    return 0
