# Schedwi
# Copyright (C) 2011-2013 Herve Quatremain
#
# This file is part of Schedwi.
#
# Schedwi is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
#
# Schedwi is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.


"""Module to manage calendars.

The str2cal and calcomp functions are used to compile a calendar string and
return a cal_t object.  Then, the calmatch method can be used to check a date
with the compiled calendar.
str2cal accept a multi-line calendar string.

The calendar string is inspired by several tools such as calendar(1) or
crontab(5).
Its format is one of the following:

   + month/month_day/option
     with month
        - An asterisk (*), which stands for all month
        - A month name.  The first three characters (case insensitive) are
          scanned so the following month names are all valid: Jan, february,
          DeCemBer, ocTob
        - A month number between 1 and 12
        - A range of month.  Ranges are two month name or number separated
          with a hyphen.  The specified range is inclusive. For example,
          6-10 specifies June, July, August, September and October.  The
          range Oct-Fev (or 10-2) specifies October, November, December,
          January and February.
          Other examples: JAN-MAR (or 1-3 or Jan-3 or 1-march),
          April-August, sep-sep
        - A list.  A list is a set of month (or ranges) separated by commas.
          Examples:
              Jan,sep-nov
              1,2,3,5-8,december
              Jan-4,Nov
     with month_day
        - An asterisk (*), which stands for all the days of the month
        - A day number between 1 and 31
        - A negative day number from the end if the month.  -1 stands for
          the last day of the month (30, 31, 28 or 29 depending of the
          month), -2 for the day before the last day of the month, ...
        - A range of days.  Ranges are two day numbers separated
          with a hyphen.  The specified range is inclusive. For example,
          6-10 specifies the 6th, 7th, 8th, 9th and 10th.  The range
          15--1 specifies all the days between the 15th and the last day
          of the month included. -1-7 specifies all the days between the
          last day of the month and the 7th (therefore from the 1st to the
          7th, plus the last day of the month)
        - A list.  A list is a set of days (or ranges) separated by commas.
          For example: 1-15,17,20--1
     with option (optional) used to specify that if a day fall on a
     specific week day, it must be moved.  Option is the list of week day
     names and increment (or decrement).  Items in the list are separated
     by commas.  An item is composed of a week day name followed by + (or
     -) and a number of days.
     Examples:
          Sat+2         All days that fall on Saturday are moved to the next
                        Monday (Saturday + 2 days = Monday)
          Sat-1         All days that fall on Saturday are moved to the
                        previous Friday
          Sat+2,Sun+1   All the days that fall the weekend are moved to
                        the next Monday
     Warning, the option string is parsed from left to right.  For instance,
          Sat+1,Sun-2   means that all Saturdays and Sundays will be moved
                        to the previous Friday (first, Saturdays are moved
                        to Sundays and then Sundays are moved to Fridays)
     The correct order for this list is then
          Sun-2,Sat+1   All days that fall on Sundays are moved to Fridays
                        and all days the fall on Saturdays are moved to the
                        next Monday

   + month/week_day/option
     with month the same as the previous format
     with week_day
        - A day name.  The first three characters (case insensitive) are
          scanned so the following day names are all valid: mon, Friday,
          Saturdays, sUnDAy
        - A range of days.  Ranges are two day names separated
          with a hyphen.  The specified range is inclusive.  For example,
          thu-saturday specifies Thusday, Friday and Saturday.  Sun-tue
          specifies Sunday, Monday and Tuesday.  Saturday-Tuesday
          specifies Saturday, Sunday, Monday and Tuesday.
        - A list.  A list is a set of day names (or ranges) separated by
          commas.  Example: MON,wed,Fri-sun
     with option used to specifies the week in the month (first, second,
     last, ...).  If option is not set, all the days of the `week_day'
     field applies to all the weeks of the month.  Valid values for
     option are: First, Second, Third, Fourth, Fifth, Last, 1, 2, 3, 4, 5,
     -1 (for last week of the month), -2, -3, -4, -5.  The option field is
     case insensitive.

   + month/Open/option
     with month the same as the previous formats.
     Open is a special keyword which stands for the days between
     Monday and Friday (included)
     Option used to specifies the day in the month (first, second,
     last, ...).  If option is not set, all the open days of the month
     are selected.  Valid values for option are: First, Second, Third,
     Fourth, Fifth, Last, 1, 2, 3, 4, 5, 6, 7, 8, ..., -1 (for last open
     day of the month), -2, -3, -4, -5, ....  The option field is
     case insensitive.  For instance 1 (or first) means the first open
     day of the month, -1 (or last) stands for the last open day of the
     month.

   + Easter or Easter+n or Easter-n
     for Easter plus or less a number of days. For example, Easter-2 stands
     for 2 days before Easter.

   + Paskha or Paskha+n or Paskha-n
     for Paskha (Orthodox Easter) plus or less a number of days. For
     example, Paskha+21 stands for 21 days after Paskha.

Not or ! at the beginning of the calendar string means `Every day except
the ones defined by the following calendar string'

Full examples:
     1/Thur
           Every Thursday of January
     Jan-March/open
           Every open days (Monday to Friday) in January, February and March
     1-12/Saturday,Sunday
           Every week-end
     7/Fri/First
           First Friday in July
     05/01
           1st May
     Jan/1/Sat+2,Sun+1
           1st January.  If the 1st January occurs a week-end, moves it to
           the following Monday (Sat+2 = Monday and Sun+1 = Monday)
     Easter+49
           49 days after Easter (Pentecost-Whitsunday)
     11/Sun/-1
           Last Sunday of November
     12/open/last
           Last open day of the year
     Jan/OPEN/1
           First open day of the year
     2/2,3
           2nd and 3rd of February
     not 1/sat,sun
           Every day except the Saturdays and Sundays in January
     ! 1-6/open/last
           Every day except the last open days in January to June
"""

import datetime
import calendar
import sys

import locale_utils
import colors

(CAL_MALLOC,        # Memory allocation error
 CAL_NOERROR,       # Success
 CAL_NOMATCH,       # The day doesn't match the calendar (calmatch)
 CAL_BADMONTHNAME,  # Unknown month name (must be Jan...Dec)
 CAL_BADDAYNAME,    # Unknown day name (must be Sun...Sat)
 CAL_MONTHOOR,      # Month number out of range (1...12)
 CAL_DAYOOR,        # Day out of range (1...31)
 CAL_BADOPTION,     # Unknown option (must be Last, first, -1, ...)
 CAL_EMPTYFIELD     # A part is empty (1-6//Last)
) = (-1, 0, -2, -3, -4, -5, -6, -7, -8)


def is_leap_year(year):
    """Check if the provided year is a leap year."""
    return ((year % 4 == 0) and (year % 100)) or (year % 400 == 0)


def get_last_day(year, month):
    """Return the last day number of the provided month and year.

    Return:
      The last day number or
      -1 if the given month or year is out of range

    """
    if month < 1 or month > 12 or year < 0:
        return -1

    if  month == 1 or \
        month == 3 or \
        month == 5 or \
        month == 7 or \
        month == 8 or \
        month == 10 or \
        month == 12:
        return 31
    if month == 2:
        if is_leap_year(year):
            return 29
        return 28
    return 30


def day_of_week(day, month, year):
    """Compute and return the week day (0=Sunday, 1=Monday, ..., 6=Saturday)

    Return -1 if the day, the month or the year is out of range
    """
    if month < 1 or month > 12 or year < 0 or day < 1 or day > 31:
        return -1
    if month < 3:
        y = year - 1
        m = 0
    else:
        y = year
        m = 2
    return ((23 * month / 9) + day + 4 + year + (y / 4) \
            - (y / 100) + (y / 400) - m) % 7


def get_open_day(month, year, inc):
    """Return the day number in the provided month that is not a week-end day.

    """
    if inc > 0:
        inc -= 1
        mday = 1
        wday = day_of_week(mday, month, year)
        if wday == 0:
            wday = 1
            mday = 2
        elif wday == 6:
            wday = 1
            mday = 3

        nb_week = inc / 5
        if wday + (inc % 5) > 5:
            nb_week += 1
        return mday + inc + nb_week * 2
    else:
        inc += 1
        mday = get_last_day(year, month)
        wday = day_of_week(mday, month, year)
        if wday == 0:
            wday = 5
            mday -= 2
        elif wday == 6:
            wday = 5
            mday -= 1
        nb_week = -inc / 5
        if wday + (inc % 5) < 1:
            nb_week += 1
        return mday + inc - nb_week * 2


def get_easter(year):
    """Return the month and day number of Easter for the given year.

    Return the tuple (easter_month, easter_day) with easter_month between 1
    and 12 and easter_day between 1 and 31.

    """
    a = year % 19
    b = year / 100
    c = year % 100
    d = b >> 2
    e = b & 3
    f = (b + 8) / 25
    g = (b - f + 1) / 3
    h = (19 * a + b - d - g + 15) % 30
    i = c >> 2
    k = c & 3
    l = (32 + (e << 1) + (i << 1) - h - k) % 7
    m = (a + 11 * h + 22 * l) / 451
    easter_month = (h + l - 7 * m + 114) / 31
    p = (h + l - 7 * m + 114) % 31
    easter_day = p + 1
    return (easter_month, easter_day)


def get_paskha(year):
    """Return the number of days in the year for Paskha (Orthodox Easter)."""
    a = year % 19
    b = year % 4
    c = year % 7
    d = (19 * a + 15) % 30
    e = (2 * b + 4 * c + 6 * d + 6) % 7
    cumdays = 31 + 28
    if is_leap_year(year):
            cumdays += 1
    return cumdays + 22 + d + e + 13


def is_bit_set(bit_field, val):
    return bit_field & (1 << (val - 1))


def add_range(bit_field, min, max, length):
    if min <= 0:
        min = 1
    if max <= 0:
        max = length

    if min > max:
        bit_field |= ~(~0 << max)
        bit_field |= (~(~0 << (length - min + 1))) << (min - 1)
    else:
        bit_field |= (~(~0 << (max - min + 1))) << (min - 1)
    return bit_field


def add_range_week(user_data, min, max):
    user_data["day"] = add_range(user_data["day"], min, max, 7)


def add_range_month(user_data, min, max):
    user_data.month = add_range(user_data.month, min, max, 31)


def add_range_all_month(user_data, min, max):
    for m in range(12):
        if is_bit_set(user_data.month, m + 1):
            i = convert_mday(min, user_data.cal_year, m + 1)
            j = convert_mday(max, user_data.cal_year, m + 1)
            user_data.year[m] = add_range(user_data.year[m], i, j, 31)


def convert_mday(val, year, month):
    """Convert a negative value to a day number in the month (1-31).

    For example, -1 is the last day of the month (31 for August for
    instance), -2 is the day before the last day of the month, ...

    """
    # If the value is positive, nothing has to be done
    if val > 0:
        return val
    last_day = get_last_day(year, month)
    ret = last_day + val + 1
    if ret <= 0:
        return last_day + 1
    return ret


def get_month_number(month_name):
    """Convert a month name to a number (1 to 12).

    Return the month number (between 1 and 12) or 0 if month_name is invalid.

    """
    m = month_name[0:3].lower()
    if m == "jan":
        return 1
    if m == "feb":
        return 2
    if m == "mar":
        return 3
    if m == "apr":
        return 4
    if m == "may":
        return 5
    if m == "jun":
        return 6
    if m == "jul":
        return 7
    if m == "aug":
        return 8
    if m == "sep":
        return 9
    if m == "oct":
        return 10
    if m == "nov":
        return 11
    if m == "dec":
        return 12
    return 0


def parse_month(month_str):
    """Parse a month (1-12 or Jan-Dec).

    @return:
            the tuple (error_code, month_number) with month_number the
            month number (between 1 and 12). month_number is 0 if month_str
            is invalid and the error_code is set.
    """
    try:
        m = int(month_str)
    except ValueError:
        ret = get_month_number(month_str.strip())
        if ret == 0:
            return (CAL_BADMONTHNAME, 0)
        return (CAL_NOERROR, ret)
    if m < 1 or m > 12:
        return (CAL_MONTHOOR, 0)
    return (CAL_NOERROR, m)


def get_day_number(day_name):
    """Convert a day name to a number (1 to 7).

    Return 1 for Sunday, 2 for Monday, ... or 0 if day_name is invalid.

    """
    d = day_name[0:3].lower()
    if d == "sun":
        return 1
    if d == "mon":
        return 2
    if d == "tue":
        return 3
    if d == "wed":
        return 4
    if d == "thu":
        return 5
    if d == "fri":
        return 6
    if d == "sat":
        return 7
    return 0


def parse_wday(day_str):
    """Parse a weekday name.

    @return:
            the tuple (error_code, weekday_number) with weekday_number the
            weekday number (between 1 and 7 with 1 for Sunday).
            weekday_number is 0 if day_str is invalid and error_code is set.
    """
    ret = get_day_number(day_str.strip())
    if ret == 0:
        return (CAL_BADDAYNAME, 0)
    return (CAL_NOERROR, ret)


def parse_mday(day_str):
    """Convert the given `day_str' string to an integer.

    `day_str' if the day offset in the month. It can be between -1 to -31
    or 1 to 31.  A negative value is the number of days from the end of the
    month (ie. -1 for the last day of the month)

    @return:
            the tuple (error_code, value) with value an integer corresponding
            to `day_str' (in the range 1 to 31 or -1 to -31). value is 0 in
            case of error and the error_code is set.

    """
    try:
        d = int(day_str)
    except ValueError:
        return (CAL_DAYOOR, 0)
    if d > 31 or d < -31 or d == 0:
        return (CAL_DAYOOR, 0)
    return (CAL_NOERROR, d)


def parse_inc(day_opt):
    """Parse an option string.

    Return the parsed integer or 0 in case of syntax error.

    """
    try:
        val = int(day_opt)
    except ValueError:
        s = day_opt.lower()
        if s[:5] == "first":
            return 1
        elif s[:4] == "last":
            return -1
        elif s[:6] == "second":
            return 2
        elif s[:5] == "third":
            return 3
        elif s[:6] == "fourth":
            return 4
        elif s[:5] == "fifth":
            return 5
        else:
            return 0
    else:
        return val


def parse_range(str, parse, set_range, user_data):
    """Parse a range.

       Examples of range:
              1-10: value must be between 1 and 10 (included)
               3  : value must be 3
             15--1: value must be between 15 and the last possible value (for a
                    month, it's the last day)
               *  : always match
       The provided parse function is used to parse both side of the range and
       must return a tupe (error_code, integer) with integer set to 0 in case
       of error.
       The provided set_range function is called to set a value. The first
       parameter is the provided user_data
       In case of parsing error (syntax error), the tuple (error_code,
       idx_in_str) is returned with idx_in_str the index of the error in str.
       Return code (first item in the tuple):
           CAL_NOERROR --> No error
           CAL_BADMONTHNAME --> Invalid month name
           CAL_BADDAYNAME --> Invalid day name
           CAL_MONTHOOR --> Month number out of range
           CAL_DAYOOR --> Day number out of range (1-31)
           CAL_EMPTYFIELD --> The last part of the range is missing (Mon-)

    """
    lst = str.split('-', 1)
    i = len(lst)
    if i == 1 or not lst[0]:
        # Single value
        err, min = parse(lst[0] if i == 1 else '-' + lst[1])
        if min == 0:
            return (err, str.index(lst[0] if i == 1 else '-' + lst[1]))
        set_range(user_data, min, min)
        return (CAL_NOERROR, 0)
    err, min = parse(lst[0].strip())
    if min == 0:
        return (err, str.index(lst[0]))
    last = lst[1].strip()
    if not last:
        return (CAL_EMPTYFIELD, str.index('-'))
    err, max = parse(last)
    if max == 0:
        return (err, str.index(last))
    set_range(user_data, min, max)
    return (CAL_NOERROR, 0)


def parse_list(str, parse, set_range, user_data):
    """Split a list of values (2,4,5,6-10) and call parse_range for each of
       them.

       In case of parsing error (syntax error), the tuple (error_code,
       idx_in_str) is returned with idx_in_str the index of the error in str.
       Return code (first item in the tuple):
            CAL_NOERROR --> No error
            CAL_EMPTYFIELD --> No value to parse or missing value
            CAL_BADMONTHNAME --> Invalid month name
            CAL_BADDAYNAME --> Invalid day name
            CAL_MONTHOOR --> Month number out of range
            CAL_DAYOOR --> Day number out of range (1-31)

    """
    for tok in str.split(","):
        if not tok:
            return (CAL_EMPTYFIELD, 0)
        error, idx = parse_range(tok, parse, set_range, user_data)
        if error != CAL_NOERROR:
            return (error, str.index(tok) + idx)
    return (CAL_NOERROR, 0)


class cal_t:

    def __init__(self, year):
        self.month = 0
        self.year = [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]
        self.cal_year = year
        # Curses stuff to highlight the corresponding days in print_cal()
        self.highlight_on = colors.HIGH
        self.highlight_off = colors.NORMAL
        self.highlight_len = colors.CHAR_COLOR_LEN

    def add_day(self, month, day):
        self.month |= 1 << (month - 1)
        self.year[month - 1] |= 1 << (day - 1)

    def remove_day(self, month, day):
        self.year[month - 1] &= ~(1 << (day - 1))
        if self.year[month - 1] == 0:
            self.month &= ~(1 << (month - 1))

    def add_week(self, wday, year):
        """Given a weekday bit field (wday), set this pattern to every week of
        the selected month of the year.

        """
        # Duplicate the week structure to create a month structure
        days = 0
        for m in range(5):
            days |= wday << (m * 7)
        for m in range(12):
            # If the month is selected
            if is_bit_set(self.month, m + 1):
                # What is the weekday name of the first day of the month?
                dow = day_of_week(1, m + 1, year)
                # Copy the month pattern
                self.year[m] |= days << (7 - dow)
                self.year[m] |= wday >> dow

    def add_one_week(self, wday, year, week):
        """Add a week pattern to the selected month.

        The week parameter tells which week to set (1, 2, 3, 4, 5 for the
        first, second, third, fourth and fifth week of the month or -1,
        -2, ... for the last, the one before the last, ... week of the month)

        """
        days = wday | (wday << 7)
        for m in range(12):
            if not is_bit_set(self.month, m + 1):
                continue
            # First, second, third, ...
            if week >= 0:
                dow = day_of_week(1, m + 1, year)
                d = (days >> dow) & 0x7F
                self.year[m] |= d << (7 * (week - 1))
            # Last, one before last, ...
            else:
                last = get_last_day(year, m + 1)
                dow = day_of_week(last, m + 1, year)
                d = (days >> (dow + 1)) & 0x7F
                i = last + 7 * week
                if i < 0:
                    self.year[m] |= d >> -i
                else:
                    self.year[m] |= d << i

    def set_day_increment(self, year, month, day, increment):
        """Set a day plus an increment.

        increment is the number of days to add or remove (ie. +1 to add
        one day, -3 to remove 3 days)

        """
        d = datetime.date(year, month, day)
        delta = datetime.timedelta(increment)
        new_date = d + delta
        if new_date.year == year:
            self.add_day(new_date.month, new_date.day)

    def move_day(self, year, month, day, increment):
        """Move a day.

        `increment' is the number of days to move the provided day
        (it may be negative)

        """
        self.remove_day(month, day)
        self.set_day_increment(year, month, day, increment)

    def move_week_day(self, year, wday, increment):
        """Move a week day."""
        for m in range(1, 13):
            if is_bit_set(self.month, m):
                last_day = get_last_day(year, m)
                d = 1
                while d <= last_day:
                    if day_of_week(d, m, year) == wday:
                        if is_bit_set(self.year[m - 1], d):
                            self.move_day(year, m, d, increment)
                        d += 7
                    else:
                        d += 1

    def parse_and_move_week_day(self, year, day_str):
        """Parse a week day and move it.

        Return the tuple (error_code, index_in_day_str).

        """
        i = 0
        lstr = len(day_str)
        while i < lstr and day_str[i].isspace():
            i += 1
        if i == lstr:
            return (CAL_BADDAYNAME, 0)
        wday = get_day_number(day_str[i:])
        if wday == 0:
            return (CAL_BADDAYNAME, i)
        wday -= 1
        try:
            j = day_str.index('+')
        except ValueError:
            try:
                j = day_str.index('-')
            except ValueError:
                return (CAL_EMPTYFIELD, i)
        try:
            incr = int(day_str[j:])
        except ValueError:
            return (CAL_EMPTYFIELD, j)

        self.move_week_day(year, wday, incr)
        return (CAL_NOERROR, 0)

    def parse_option_day_move(self, year, str):
        """Parse the option string used to move days if they fall on some
        week days.

        In case of parsing error (syntax error), the returned tuple is
        (error_code, index_in_str) with index_in_str the index of the error in
        str.
        The syntax of str is as follow:
            <wday1>+<nb>,<wday2>-<nb>,...
         with
            wday1 and wday2: weekday names to move
            +<nb> and -<nb>: number of day to move the weekday

         For instance:
            Sat+2,Sun+1 --> Will move all the Saturdays and Sundays to Mondays
            Sat-1,Sun+1 --> Will move all the Saturdays to Fridays and the
                            Sundays to Mondays
         Warning:
            Sat+1,Sun-2 --> Will move all the Saturdays and Sundays to Fridays!
                            The string is parse from left to right, so Sat+1
                            will move all the Saturdays to Sundays and then the
                            Sun-2 will move all the Sundays (included the
                            previously moved Saturdays) to Fridays

         Return:
            CAL_NOERROR --> No error
            CAL_EMPTYFIELD --> No value to parse or missing value
            CAL_BADDAYNAME --> Invalid day name

        """
        for tok in str.split(','):
            error, idx = self.parse_and_move_week_day(year, tok)
            if error != CAL_NOERROR:
                return (error, str.index(tok))
        return (CAL_NOERROR, 0)

    def calcomp_easter(self, date_str, year):
        """Compute Easter for the provided year.

        The string date_str is an increment from Easter (ie. +1 for
        the day following Easter or -42 for 42 days before Easter)

        """
        month, day = get_easter(year)
        try:
            incr = int(date_str)
        except ValueError:
            incr = 0
        self.set_day_increment(year, month, day, incr)

    def calcomp_paskha(self, date_str, year):
        """Compute Paskha for the provided year.

        The string date_str is an increment from Paskha (ie. +1 for
        the day following Paskha or -42 for 42 days before Paskha)

        """
        try:
            incr = int(date_str)
        except ValueError:
            incr = 0
        yday = get_paskha(year)
        d = datetime.date(year, 1, 1)
        delta = datetime.timedelta(yday - 1 + incr)
        new_date = d + delta
        if new_date.year == year:
            self.add_day(new_date.month, new_date.day)

    def parse_option(self, day_opt, wday, year):
        """Parse the option string.

        Return the tuple (error_code, idx_in_day_opt) with error_code:
            CAL_NOERROR --> No error
            CAL_BADOPTION --> The option string is invalid (1, -1, First,
                              Last, ...)

        """
        val = parse_inc(day_opt)
        if val == 0:
            return (CAL_BADOPTION, 0)
        self.add_one_week(wday, year, val)
        return (CAL_NOERROR, 0)

    def parse_open_day(self, day_opt, year):
        """Parse the option string.

        Return the tuple (error_code, idx_in_day_opt) with error_code:
            CAL_NOERROR --> No error
            CAL_BADOPTION --> The option string is invalid (1, -1, First,
                              Last, ...)

        """
        n = 0
        lstr = len(day_opt)
        while n < lstr and day_opt[n].isspace():
            n += 1
        if n == lstr:
            d = {"day": 0}
            add_range_week(d, 2, 6)
            wday = d["day"]
            self.add_week(wday, year)
            return (CAL_NOERROR, 0)
        val = parse_inc(day_opt[n:])
        if val == 0:
            return (CAL_BADOPTION, n)
        for m in range(12):
            if is_bit_set(self.month, m + 1):
                ret = get_open_day(m + 1, year, val)
                if ret > 0 and ret < 32:
                    self.add_day(m + 1, ret)
        return (CAL_NOERROR, 0)

    def calinvert(self):
        """Invert the calendar."""
        for m in range(12):
            if is_bit_set(self.month, m + 1):
                self.year[m] = ~(self.year[m])
            else:
                self.month |= 1 << m
                self.year[m] = ~0

    def calmatch(self, day, month, year):
        """Check that the provided date matches the calendar.

        Return:
            CAL_NOERROR --> The date matches the calendar
            CAL_NOMATCH --> The date does not match the calendar

        """
        if  self.cal_year != year or \
            not is_bit_set(self.month, month) or \
            not is_bit_set(self.year[month - 1], day):
            return CAL_NOMATCH
        return CAL_NOERROR

    def calcomp_month(self, month_str):
        """Parse the month part.

        Example of month string:
                        *: Every month
                       2: February
                     1-6: January to June (inclusive)
            feb-December: February to December
                 1-4,dec: list of values (January to April and December)

        In case of parsing error (syntax error), the tuple (error_code,
        idx_in_str) is returned with idx_in_str the index of the error in
        month_str.
        Return code (first item in the tuple):
           CAL_NOERROR --> No error
           CAL_EMPTYFIELD --> No value to parse or missing value
           CAL_BADMONTHNAME --> Invalid month name
           CAL_BADDAYNAME --> Invalid day name
           CAL_MONTHOOR --> Month number out of range
           CAL_DAYOOR --> Day number out of range (1-31)

        """
        if month_str.strip() == '*':
            self.month = ~0
            return (CAL_NOERROR, 0)
        return parse_list(month_str, parse_month, add_range_month, self)

    def calcomp_day(self, day_str, day_opt, year):
        """Parse the day part and the optional modifier.

                     day_str: Day string to parse
                     day_opt: Option string to parse
                        year: Year to use to compile the calendar
        If the day string start by a number, it relates to the day number in
        the month. In this case, the option modifier is not used.
        If not, it relates to the day in the week.

        Examples of day string:
                         *: Every day of the week
                       Mon: Monday
                         1: First day of the month
                      2-12: Second to 12th day of the month
                   Tue-Fri: Tuesday to Friday (included)
                 1-4,21-25: list of values (1 to 4 and 21 to 25)
                        -1: Last day of the month
                     20--2: From the 20th to the last but one day of the month

        Example of option string:
                Last or -1: Last week in the month
          First or 1 or +1: First week in the month
         Second or 2 or +2: Second week in the month

        Example of day and option string:
                 mon/first: First monday of the month
              mon-wed/last: Monday to Friday of the last week of the month
         Monday,sat/Second: Monday and Saturday of the second week of the month

        In case of parsing error (syntax error), the tuple (error_code,
        idx_in_str) is returned with idx_in_str the index of the error in
        day_str or day_opt (if error_code == CAL_BADOPTION).
        Return code (first item in the tuple):
           CAL_NOERROR --> No error
           CAL_EMPTYFIELD --> No value to parse or missing value
           CAL_BADMONTHNAME --> Invalid month name
           CAL_BADDAYNAME --> Invalid day name
           CAL_MONTHOOR --> Month number out of range
           CAL_DAYOOR --> Day number out of range (1-31)
           CAL_BADOPTION --> The option string is invalid (1, -1, First,
                             Last, ...)

        """
        n = 0
        dlen = len(day_str)
        while n < dlen and day_str[n].isspace():
            n += 1
        # Day number in the month: 1-31, -1, ...
        if n != dlen and (day_str[n].isdigit() or (day_str[n] == '-' and
                                                   n + 1 != dlen and
                                                   day_str[n + 1].isdigit())):
            error, idx = parse_list(day_str[n:], parse_mday,
                                    add_range_all_month, self)
            if error != CAL_NOERROR:
                return (error, n + idx)
            # Day options (Sat+2,Sun+1, ...)
            n = 0
            olen = len(day_opt)
            while n < olen and day_opt[n].isspace():
                n += 1
            if n == olen:
                return (CAL_NOERROR, 0)
            error, idx = self.parse_option_day_move(year, day_opt[n:])
            return (error, n + idx)
        # Open
        if day_str[n:n + 4].lower() == "open":
            return self.parse_open_day(day_opt, year)
        # Day in the week: Mon-Sun or *
        if day_str[n] == '*':
            d = {"day": ~0}
        else:
            d = {"day": 0}
            error, idx = parse_list(day_str[n:], parse_wday, add_range_week, d)
            if error != CAL_NOERROR:
                return (error, n + idx)
        day = d["day"]
        # Day options (first, second, last, 1, +1, 2, +2, ...)
        n = 0
        olen = len(day_opt)
        while n < olen and day_opt[n].isspace():
            n += 1
        if n == olen:
            self.add_week(day, year)
            return (CAL_NOERROR, 0)
        error, idx = self.parse_option(day_opt[n:], day, year)
        return (error, n + idx)

    def setfirstweekday(self, day):
        calendar.setfirstweekday(day)

    def print_cal(self, month=None):
        """Print the calendar."""
        DAY_LEN = 3
        GUT_LEN = 4
        loc = locale_utils.get_locale()
        self.setfirstweekday(loc.first_week_day)
        day_labels = list()
        for i in range(loc.first_week_day, 7 + loc.first_week_day):
            day_labels.append(
                        loc.days['format']['abbreviated'][i % 7][:DAY_LEN])
        str_day = ' '.join(day_labels).encode('utf-8')
        if month is not None:
            t = "%s %d" % (loc.months['format']['wide'][month].encode('utf-8'),
                           self.cal_year)
            print "%s" % t.center((DAY_LEN + 1) * 7)
            print str_day
            for w in calendar.monthcalendar(self.cal_year, month):
                for d in w:
                    if not d:
                        sys.stdout.write(" " * (DAY_LEN + 1))
                        continue
                    if self.calmatch(d, month, self.cal_year) != CAL_NOMATCH:
                        sys.stdout.write(" " * (1 - self.highlight_len))
                        sys.stdout.write(self.highlight_on)
                        sys.stdout.write("%2.d" % d)
                        sys.stdout.write(self.highlight_off)
                        sys.stdout.write(" " * (1 - self.highlight_len))
                    else:
                        sys.stdout.write("%3.d " % d)
                print ""
        else:
            c = calendar.Calendar(loc.first_week_day)
            ml = 1
            mr = 2
            print "%s" % str(self.cal_year).center((DAY_LEN + 1) * 14 +
                                                   GUT_LEN)
            for row in c.yeardayscalendar(self.cal_year, 2):
                str_ml = loc.months['format']['wide'][ml].encode('utf-8')
                str_mr = loc.months['format']['wide'][mr].encode('utf-8')
                print "%s%s%s" % (str_ml.center((DAY_LEN + 1) * 7),
                                  " " * GUT_LEN,
                                  str_mr.center((DAY_LEN + 1) * 7))
                print str_day + ' ' * GUT_LEN + ' ' + str_day
                for w in range(5):
                    if len(row[0]) > w:
                        for d in row[0][w]:
                            if not d:
                                sys.stdout.write(" " * (DAY_LEN + 1))
                                continue
                            if (self.calmatch(d, ml, self.cal_year) !=
                                CAL_NOMATCH):
                                sys.stdout.write(" " *
                                                 (1 - self.highlight_len))
                                sys.stdout.write(self.highlight_on)
                                sys.stdout.write("%2.d" % d)
                                sys.stdout.write(self.highlight_off)
                                sys.stdout.write(" " *
                                                 (1 - self.highlight_len))
                            else:
                                sys.stdout.write("%3.d " % d)
                    else:
                        sys.stdout.write(' ' * ((DAY_LEN + 1) * 7))
                    sys.stdout.write(' ' * GUT_LEN)
                    if len(row[1]) > w:
                        for d in row[1][w]:
                            if not d:
                                sys.stdout.write(" " * (DAY_LEN + 1))
                                continue
                            if (self.calmatch(d, mr, self.cal_year) !=
                                    CAL_NOMATCH):
                                sys.stdout.write(" " *
                                                 (1 - self.highlight_len))
                                sys.stdout.write(self.highlight_on)
                                sys.stdout.write("%2.d" % d)
                                sys.stdout.write(self.highlight_off)
                                sys.stdout.write(" " *
                                                 (1 - self.highlight_len))
                            else:
                                sys.stdout.write("%3.d " % d)
                    else:
                        sys.stdout.write(' ' * ((DAY_LEN + 1) * 7))
                    print " "
                print " "
                ml += 2
                mr += 2


def calcomp(calendar, year):
    """Compile a calendar string into a form that is suitable for subsequent
    calendar checks.

    Arguments:
    calendar -- string to parse
    year -- year (used to compute Easter). Subsequent calendar checks must be
            in this year

    Return a triplet (error_code, idx_in_str, new_cal)

    Example of calendar string:
              Jan/mon (or 1/mon): Every Monday in January
                    1-6/Sun/Last: Last Sunday on the month January to September
                       Easter+39: 39 days after Easter
                        easter-2: 2 days before Easter
                        jan/open: All the open days (Monday to Friday) in
                                  January
                  feb/open/first: First open day in February
                     1-3/Open/-1: Last open day in month from January to March
                      not easter: Every day except Easter

    In case of parsing error (syntax error), the tuple (error_code, idx_in_str)
    is returned with idx_in_str the index of the error in calendar.
    Return code (first item in the tuple):
       CAL_NOERROR --> No error
       CAL_EMPTYFIELD --> No value to parse or missing value
       CAL_BADMONTHNAME --> Invalid month name
       CAL_BADDAYNAME --> Invalid day name
       CAL_MONTHOOR --> Month number out of range
       CAL_DAYOOR --> Day number out of range (1-31)
       CAL_BADOPTION --> The option string is invalid (1, -1, First, Last, ...)

    """
    cal = cal_t(year)
    n = 0
    clen = len(calendar)
    while n < clen and calendar[n].isspace():
        n += 1
    # Invert
    if n != clen:
        if calendar[n] == '!':
            invert_cal = True
            n += 1
        else:
            if calendar[n:n + 3].lower() == "not":
                invert_cal = True
                n += 3
            else:
                invert_cal = False
    while n < len and calendar[n].isspace():
        n += 1
    tok = calendar[n:].split('/')
    if not tok[0]:
        if invert_cal:
            cal.calinvert()
        return (CAL_NOERROR, 0, cal)
    # Easter
    if tok[0][:6].lower() == "easter":
        cal.calcomp_easter(tok[0][6:], year)
        if invert_cal:
            cal.calinvert()
        return (CAL_NOERROR, 0, cal)
    # Paskha
    if tok[0][:6].lower() == "paskha":
        cal.calcomp_paskha(tok[0][6:], year)
        if invert_cal:
            cal.calinvert()
        return (CAL_NOERROR, 0, cal)
    error, idx = cal.calcomp_month(tok[0])
    if error != CAL_NOERROR:
        return (error, idx + calendar.index(tok[0]), cal)
    if len(tok) == 1 or not tok[1]:
        add_range_all_month(cal, 1, 31)
        if invert_cal:
            cal.calinvert()
        return (CAL_NOERROR, 0, cal)
    day_str = tok[1]
    opt_str = tok[2] if len(tok) > 2 else ''
    error, idx = cal.calcomp_day(day_str, opt_str, year)
    if error != CAL_NOERROR:
        if error == CAL_BADOPTION:
            i = idx + calendar.index(opt_str)
        else:
            i = idx + calendar.index(day_str)
        return (error, i, cal)
    if invert_cal:
        cal.calinvert()
    return (CAL_NOERROR, 0, cal)


def str2cal(calendar, year):
    """Compile a multiline calendar string into a form that is suitable for
    subsequent calendar checks.  The provided string may contains comments (on
    a line by themselves and starting by the `#' character)

    Arguments:
    calendar -- string to parse
    year -- year (used to compute Easter). Subsequent calendar checks must be
            in this year

    Return a triplet (error_code, idx_in_str, new_cal)

    Example of calendar string:
              Jan/mon (or 1/mon): Every Monday in January
                    1-6/Sun/Last: Last Sunday on the month January to June
                       Easter+39: 39 days after Easter
                        easter-2: 2 days before Easter
                        jan/open: All the open days (Monday to Friday) in
                                  January
                  feb/open/first: First open day in February
                     1-3/Open/-1: Last open day in month from January to March
                      not easter: Every day except Easter
               12/25/Sat+2,Sun+1: Christmas.  If on a Saturday or Sunday, move
                                  to the following Monday

    In case of parsing error (syntax error), the tuple (error_code, idx_in_str)
    is returned with idx_in_str the index of the error in calendar.
    Return code (first item in the tuple):
       CAL_NOERROR --> No error
       CAL_EMPTYFIELD --> No value to parse or missing value
       CAL_BADMONTHNAME --> Invalid month name
       CAL_BADDAYNAME --> Invalid day name
       CAL_MONTHOOR --> Month number out of range
       CAL_DAYOOR --> Day number out of range (1-31)
       CAL_BADOPTION --> The option string is invalid (1, -1, First, Last, ...)

    """
    cals_add = list()
    cals_del = list()
    for line in calendar.splitlines():
        s = line.strip()
        if not s or s[0] == '#':
            continue
        error, idx, cal = calcomp(s, year)
        if error != CAL_NOERROR:
            return (error, idx + calendar.index(s), cal)
        if s[0] == '!' or s[:3].lower() == 'not':
            cals_del.append(cal)
        else:
            cals_add.append(cal)
    cal = cal_t(year)
    for c in cals_add:
        cal.month |= c.month
        for m in range(12):
            cal.year[m] |= c.year[m]
    for c in cals_del:
        cal.month &= c.month
        for m in range(12):
            cal.year[m] &= c.year[m]
    return (CAL_NOERROR, 0, cal)
