/* Schedwi
   Copyright (C) 2007-2013 Herve Quatremain

   This file is part of Schedwi.

   Schedwi is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 3 of the License, or
   (at your option) any later version.

   Schedwi is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

/* sql_status.c -- job status management functions */

#include <schedwi.h>

#if STDC_HEADERS
#include <stdlib.h>
#include <string.h>
#else
#if HAVE_STDLIB_H
#include <stdlib.h>
#endif
#if HAVE_STRING_H
#include <string.h>
#endif
#endif

#if HAVE_ASSERT_H
#include <assert.h>
#endif

#include <lib_functions.h>
#include <sql_common.h>
#include <sql_status.h>

#define SQLSTATUS_WORKLOAD_LIST "SELECT DISTINCT workload_date FROM job_status ORDER BY workload_date"
#define SQLSTATUS_GET "SELECT status,start_time,retry_num,error_msg FROM job_status WHERE workload_date=%d AND job_id=%ld"
#define SQLSTATUS_UPDATE "UPDATE job_status SET status=%d,start_time=%u,retry_num=%d,error_msg=%s WHERE workload_date=%d AND job_id=%s"
#define SQLSTATUS_SET "INSERT INTO job_status (workload_date,job_id,status,start_time,retry_num,error_msg) VALUES (%d,%s,%d,%u,%d,%s)"
#define SQLSTATUS_UPDATE_START_TIME "UPDATE job_status SET start_time=%u WHERE workload_date=%d AND job_id=%ld"
#define SQLSTATUS_DEL "DELETE FROM job_status WHERE workload_date=%d"
#define SQLSTATUS_DEL_TABLES "DELETE FROM %s_s WHERE workload_date=%d"
#define SQLSTATUS_COPY "INSERT INTO %s_s SELECT *,%d FROM %s"
#define SQLSTATUS_GET_OLD "SELECT workload_date FROM %s_s WHERE workload_date<%d"

/*
 * List of tables to copy.  These tables will be copied in tables with the
 * same name but suffixed by `_s'
 */
static char * status_tables[] = {	"calendars",
					"job_main",
					"job_start_limit",
					"job_max_duration",
					"job_retries",
					"job_retries_interval",
					"job_username",
					"job_file_out",
					"job_file_err",
					"job_control_group",
					"job_loadenv",
					"job_success_return_code",
					"job_command",
					"job_arguments",
					"environments",
					"environment_var",
					"job_environment",
					"host_environment",
					"job_host",
					"constraint_file",
					"jobset_background",
					"job_icon_default",
					"job_icon_completed",
					"job_icon_failed",
					"job_icon_running",
					"job_icon_waiting",
					"jobset_icon_default",
					"jobset_icon_completed",
					"jobset_icon_failed",
					"jobset_icon_running",
					"jobset_icon_waiting",
					"link",
					NULL
				};


/*
 * Get all the available workload from the job_status database table
 *
 * Return:
 *     0 --> No error.  rows contains the retrieved workload date and must be
 *           freed by the caller by
 *              lwc_delLL (rows, (void (*)(const void *)) sql_free_row);
 *    -1 --> Memory allocation error (if error_func() is not NULL, it is called
 *           with user_data_error_func as its first parameter and the error
 *           message as the second parameter)
 *    -2 --> SQL error (if error_func() is not NULL, it is called with
 *           user_data_error_func as its first parameter and the error message
 *           as the second parameter)
 */
int
sql_status_workload_list (lwc_LL **rows,
			void (*error_func)(void *, const char *, int),
			void *user_data_error_func)
{
	char *err_msg = NULL;
	int ret;

#if HAVE_ASSERT_H
	assert (rows != NULL);
#endif

	ret = sql_select (	NULL, NULL, &err_msg, NULL, rows,
				SQLSTATUS_WORKLOAD_LIST,
				SQL_END);

	if (ret != 0) {
		if (error_func != NULL) {
			error_func (user_data_error_func, err_msg, ret);
		}
		if (err_msg != NULL) {
			free (err_msg);
		}
		return ret;
	}
	return 0;
}


/*
 * Get the status of the provided job/jobset
 *
 * Return:
 *     0 --> No error.  status contains the retrieved status and start_time
 *           the associated start time.
 *    -1 --> Memory allocation error (if error_func() is not NULL, it is called
 *           with user_data_error_func as its first parameter and the error
 *           message as the second parameter)
 *    -2 --> SQL error (if error_func() is not NULL, it is called with
 *           user_data_error_func as its first parameter and the error message
 *           as the second parameter)
 */
int
sql_status_get (int workload_date,
		unsigned long long int job_id,
		int *status,
		long int *start_time,
		int *retry_num,
		char **message,
		void (*error_func)(void *, const char *, int),
		void *user_data_error_func)
{
	char *err_msg = NULL;
	lwc_LL *rows;
	int ret;
	char *s;
	row_item_t *row;

#if HAVE_ASSERT_H
	assert (status != NULL);
#endif

	ret = sql_select (	NULL, NULL, &err_msg, NULL, &rows,
				SQLSTATUS_GET,
				SQL_INT, (long int)workload_date,
				SQL_INT, (long int)job_id,
				SQL_END);

	if (ret != 0) {
		if (error_func != NULL) {
			error_func (user_data_error_func, err_msg, ret);
		}
		if (err_msg != NULL) {
			free (err_msg);
		}
		return ret;
	}

	row = (row_item_t *) lwc_delStartLL (rows);

	/* Message */
	if (message != NULL) {
		if (	   row == NULL
			|| row[3].value_string == NULL
			|| row[3].value_string[0] == '\0')
		{
			*message = NULL;
		}
		else {
			s = (char *) malloc (row[3].len + 1);
			if (s == NULL) {
				sql_free_row (row);
				lwc_delLL (rows,
					(void (*)(const void *))sql_free_row);
				if (error_func != NULL) {
					error_func (user_data_error_func,
						_("Memory allocation error"),
						0);
				}
				return -1;
			}
			strcpy (s, row[3].value_string);
			*message = s;
		}
	}

	/* Status */
	*status = (row == NULL) ? 0 : (int)sql_row_item2ll (&(row[0]));

	/* Start time */
	if (start_time != NULL) {
		*start_time = (row == NULL) ? 0
				: (long int)sql_row_item2ll (&(row[1]));
	}

	/* Current retry number */
	if (retry_num != NULL) {
		*retry_num = (row == NULL) ? 0
				: (int)sql_row_item2ll (&(row[2]));
	}

	sql_free_row (row);
	lwc_delLL (rows, (void (*)(const void *))sql_free_row);
	return 0;
}


/*
 * Set the status of the provided job/jobset
 *
 * Return:
 *     0 --> No error
 *    -1 --> Memory allocation error (if error_func() is not NULL, it is called
 *           with user_data_error_func as its first parameter and the error
 *           message as the second parameter)
 *    -2 --> SQL error (if error_func() is not NULL, it is called with
 *           user_data_error_func as its first parameter and the error message
 *           as the second parameter)
 */
int
sql_status_set (int workload_date,
		const char *job_id,
		int status,
		unsigned int start_time,
		int retry_num,
		const char *error_message,
		void (*error_func)(void *, const char *, int),
		void *user_data_error_func)
{
	char *err_msg = NULL;
	int ret;
	unsigned long long int nb = 0;

#if HAVE_ASSERT_H
	assert (job_id != NULL);
#endif

	ret = sql_non_select (  NULL, NULL, &err_msg, NULL, &nb, NULL,
				SQLSTATUS_UPDATE,
				SQL_INT, (long int)status,
				SQL_INT, (long int)start_time,
				SQL_INT, (long int)retry_num,
				SQL_STRING, (error_message != NULL)
							? error_message: "",
				SQL_INT, (long int)workload_date,
				SQL_STRING, job_id,
				SQL_END);
	if (ret != 0) {
		if (error_func != NULL) {
			error_func (user_data_error_func, err_msg, ret);
		}
		if (err_msg != NULL) {
			free (err_msg);
		}
		return ret;
	}

	/* Update successful.  There was already a row for this job */
	if (nb != 0) {
		return 0;
	}

	/* There is not yet a status row for this job.  Add one */
	err_msg = NULL;
	ret = sql_non_select (  NULL, NULL, &err_msg, NULL, NULL, NULL,
				SQLSTATUS_SET,
				SQL_INT, (long int)workload_date,
				SQL_STRING, job_id,
				SQL_INT, (long int)status,
				SQL_INT, (long int)start_time,
				SQL_INT, (long int)retry_num,
				SQL_STRING, (error_message != NULL)
							? error_message: "",
				SQL_END);
	if (ret != 0) {
		if (error_func != NULL) {
			error_func (user_data_error_func, err_msg, ret);
		}
		if (err_msg != NULL) {
			free (err_msg);
		}
		return ret;
	}

	return 0;
}


/*
 * Change the start time of a running job/jobset
 *
 * Return:
 *     0 --> No error
 *    -1 --> Memory allocation error (if error_func() is not NULL, it is called
 *           with user_data_error_func as its first parameter and the error
 *           message as the second parameter)
 *    -2 --> SQL error (if error_func() is not NULL, it is called with
 *           user_data_error_func as its first parameter and the error message
 *           as the second parameter)
 */
int
sql_status_adjust_start_time (int workload_date,
			unsigned long long int job_id,
			unsigned int start_time,
			void (*error_func)(void *, const char *, int),
			void *user_data_error_func)
{
	char *err_msg = NULL;
	int ret;

	ret = sql_non_select (  NULL, NULL, &err_msg, NULL, NULL, NULL,
				SQLSTATUS_UPDATE_START_TIME,
				SQL_INT, (long int)start_time,
				SQL_INT, (long int)workload_date,
				SQL_INT, (long int)job_id,
				SQL_END);
	if (ret != 0) {
		if (error_func != NULL) {
			error_func (user_data_error_func, err_msg, ret);
		}
		if (err_msg != NULL) {
			free (err_msg);
		}
		return ret;
	}

	return 0;
}


/*
 * Delete a workload
 *
 * Return:
 *     0 --> No error
 *    -1 --> Memory allocation error (if error_func() is not NULL, it is called
 *           with user_data_error_func as its first parameter and the error
 *           message as the second parameter)
 *    -2 --> SQL error (if error_func() is not NULL, it is called with
 *           user_data_error_func as its first parameter and the error message
 *           as the second parameter)
 */
int
sql_status_delete (int workload_date,
		void (*error_func)(void *, const char *, int),
		void *user_data_error_func)
{
	char *err_msg = NULL;
	unsigned int err, i;
	int ret;
	char *buff;
	unsigned int buff_len;

	buff = NULL;
	buff_len = 0;

	/* First, delete the entry in the job_status table */
	ret = sql_non_select (  &buff, &buff_len, &err_msg, NULL, NULL, NULL,
				SQLSTATUS_DEL,
				SQL_INT, (long int)workload_date,
				SQL_END);
	if (ret != 0) {
		if (error_func != NULL) {
			error_func (user_data_error_func, err_msg, ret);
		}
		if (err_msg != NULL) {
			free (err_msg);
		}
		if (buff != NULL) {
			free (buff);
		}
		return ret;
	}

	/* And then delete the entries for all the other tables */
	for (i = 0; status_tables[i] != NULL; i++);
	err = 0;
	do {
		i--;
		ret = sql_non_select (  &buff, &buff_len, &err_msg, NULL, NULL,
					NULL,
					SQLSTATUS_DEL_TABLES,
					SQL_STRING_NON_ESCAPE,
							status_tables[i],
					SQL_INT, (long int)workload_date,
					SQL_END);
		if (ret != 0) {
			err = ret;
			if (error_func != NULL) {
				error_func (	user_data_error_func,
						err_msg, ret);
			}
			if (err_msg != NULL) {
				free (err_msg);
			}
		}
	} while (i > 0);

	if (buff != NULL) {
		free (buff);
	}

	if (err != 0) {
		return err;
	}
	return 0;
}


/*
 * Check for old workloads
 * For workloads older than the provided workload, the callback() function is
 * called with the following parameters:
 *    - The provided user_data
 *    - Table name (without the `_s')
 *    - The workload date
 *
 * Return:
 *     2 --> Old workloads found
 *     0 --> No error.
 *    -1 --> Memory allocation error (if error_func() is not NULL, it is called
 *           with user_data_error_func as its first parameter and the error
 *           message as the second parameter)
 *    -2 --> SQL error (if error_func() is not NULL, it is called with
 *           user_data_error_func as its first parameter and the error message
 *           as the second parameter)
 */
int
sql_status_get_old (	int workload_date,
			int (*callback)(void *, const char *, int),
			void *user_data,
			void (*error_func)(void *, const char *, int),
			void *user_data_error_func)
{
	char *err_msg = NULL;
	lwc_LL *rows;
	row_item_t *row;
	unsigned int i;
	int ret, global_ret;
	char *buff;
	unsigned int buff_len;

#if HAVE_ASSERT_H
	assert (callback != NULL);
#endif

	buff = NULL;
	buff_len = 0;
	global_ret = 0;
	for (i = 0; status_tables[i] != NULL; i++) {
		ret = sql_select (&buff, &buff_len, &err_msg, NULL, &rows,
				SQLSTATUS_GET_OLD,
				SQL_STRING_NON_ESCAPE, status_tables[i],
				SQL_INT, (long int)workload_date,
				SQL_END);
		if (ret != 0) {
			if (error_func != NULL) {
				error_func (	user_data_error_func, err_msg,
						ret);
			}
			if (err_msg != NULL) {
				free (err_msg);
			}
			if (buff != NULL) {
				free (buff);
			}
			return ret;
		}

		while ((row = (row_item_t *) lwc_delStartLL (rows)) != NULL) {
			/*
			 * row[0] --> workload_date
			 */
			callback (	user_data, status_tables[i],
					(int)sql_row_item2ll (&(row[0])));
			sql_free_row (row);
			global_ret = 2;
		}
		lwc_delLL (rows, (void (*)(const void *))sql_free_row);
	}

	if (buff != NULL) {
		free (buff);
	}

	return global_ret;
}


/*
 * Create a new workload by copying tables
 *
 * Return:
 *     0 --> No error
 *    -1 --> Memory allocation error (if error_func() is not NULL, it is called
 *           with user_data_error_func as its first parameter and the error
 *           message as the second parameter)
 *    -2 --> SQL error (if error_func() is not NULL, it is called with
 *           user_data_error_func as its first parameter and the error message
 *           as the second parameter)
 */
int
sql_status_copy_to_workload (int workload_date,
		void (*error_func)(void *, const char *, int),
		void *user_data_error_func)
{
	char *err_msg = NULL;
	unsigned int i;
	int ret;
	char *buff;
	unsigned int buff_len;

	buff = NULL;
	buff_len = 0;

	/*
	 * First, delete the workload from all the tables if it already
	 * exists. It should not, but to be on the safe side...
	 */
	sql_status_delete (workload_date, NULL, NULL);

	/* Copy all the tables */
	for (i = 0; status_tables[i] != NULL; i++) {
		ret = sql_non_select (  &buff, &buff_len, &err_msg, NULL, NULL,
					NULL,
					SQLSTATUS_COPY,
					SQL_STRING_NON_ESCAPE,
							status_tables[i],
					SQL_INT, (long int)workload_date,
					SQL_STRING_NON_ESCAPE,
							status_tables[i],
					SQL_END);
		if (ret != 0) {
			if (error_func != NULL) {
				error_func (	user_data_error_func,
						err_msg, ret);
			}
			if (err_msg != NULL) {
				free (err_msg);
			}
			if (buff != NULL) {
				free (buff);
			}

			/* Try to remove the successfully copied tables */
			sql_status_delete (workload_date, NULL, NULL);
			return ret;
		}
	}
	if (buff != NULL) {
		free (buff);
	}

	return 0;
}

/*-----------------============== End Of File ==============-----------------*/
