/* Schedwi
   Copyright (C) 2007 Herve Quatremain

   This file is part of Schedwi.

   Schedwi is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 3 of the License, or
   (at your option) any later version.

   Schedwi is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

/*
 * calendar.c -- Manage calendars
 *
 * The calcomp function is used to compile a calendar string to an internal
 * object.
 * The calmatch function check the provided date with compiled calendar
 *
 * The calendar string is inspired by several tools such as calendar(1) or
 * crontab(5).
 * Its format is one of the following:
 *
 *    + month/month_day/option
 *      with month
 *         - An asterisk (*), which stands for all month
 *         - A month name.  The first three characters (case insensitive) are
 *           scanned so the following month names are all valid: Jan, february,
 *           DeCemBer, ocTob
 *         - A month number between 1 and 12
 *         - A range of month.  Ranges are two month name or number separated
 *           with a hyphen.  The specified range is inclusive. For example,
 *           6-10 specifies June, July, August, September and October.  The
 *           range Oct-Fev (or 10-2) specifies October, November, December,
 *           January and February.
 *           Other examples: JAN-MAR (or 1-3 or Jan-3 or 1-march),
 *           April-August, sep-sep
 *         - A list.  A list is a set of month (or ranges) separated by commas.
 *           Examples:
 *               Jan,sep-nov
 *               1,2,3,5-8,december
 *               Jan-4,Nov
 *      with month_day
 *         - An asterisk (*), which stands for all the days of the month
 *         - A day number between 1 and 31
 *         - A negative day number from the end if the month.  -1 stands for
 *           the last day of the month (30, 31, 28 or 29 depending of the
 *           month), -2 for the day before the last day of the month, ...
 *         - A range of days.  Ranges are two day numbers separated
 *           with a hyphen.  The specified range is inclusive. For example,
 *           6-10 specifies the 6th, 7th, 8th, 9th and 10th.  The range
 *           15--1 specifies all the days between the 15th and the last day
 *           of the month included. -1-7 specifies all the days between the
 *           last day of the month and the 7th (therefore from the 1st to the
 *           7th, plus the last day of the month)
 *         - A list.  A list is a set of days (or ranges) separated by commas.
 *           For example: 1-15,17,20--1
 *      with option (optional) used to specify that if a day fall on a
 *      specific week day, it must be moved.  Option is the list of week day
 *      names and increment (or decrement).  Items in the list are separated
 *      by commas.  An item is composed of a week day name followed by + (or
 *      -) and a number of days.
 *      Examples:
 *           Sat+2         All days that fall on Saturday are moved to the next
 *                         Monday (Saturday + 2 days = Monday)
 *           Sat-1         All days that fall on Saturday are moved to the
 *                         previous Friday
 *           Sat+2,Sun+1   All the days that fall the weekend are moved to
 *                         the next Monday
 *      Warning, the option string is parsed from left to right.  For instance,
 *           Sat+1,Sun-2   means that all Saturdays and Sundays will be moved
 *                         to the previous Friday (first, Saturdays are moved
 *                         to Sundays and then Sundays are moved to Fridays)
 *      The correct order for this list is then
 *           Sun-2,Sat+1   All days that fall on Sundays are moved to Fridays
 *                         and all days the fall on Saturdays are moved to the
 *                         next Monday
 *
 *    + month/week_day/option
 *      with month the same as the previous format
 *      with week_day
 *         - A day name.  The first three characters (case insensitive) are
 *           scanned so the following day names are all valid: mon, Friday,
 *           Saturdays, sUnDAy
 *         - A range of days.  Ranges are two day names separated
 *           with a hyphen.  The specified range is inclusive.  For example,
 *           thu-saturday specifies Thusday, Friday and Saturday.  Sun-tue
 *           specifies Sunday, Monday and Tuesday.  Saturday-Tuesday
 *           specifies Saturday, Sunday, Monday and Tuesday.
 *         - A list.  A list is a set of day names (or ranges) separated by
 *           commas.  Example: MON,wed,Fri-sun
 *      with option used to specifies the week in the month (first, second,
 *      last, ...).  If option is not set, all the days of the `week_day'
 *      field applies to all the weeks of the month.  Valid values for
 *      option are: First, Second, Third, Fourth, Fifth, Last, 1, 2, 3, 4, 5,
 *      -1 (for last week of the month), -2, -3, -4, -5.  The option field is
 *      case insensitive.
 *
 *    + month/Open/option
 *      with month the same as the previous formats.
 *      Open is a special keyword which stands for the days between
 *      Monday and Friday (included)
 *      Option used to specifies the day in the month (first, second,
 *      last, ...).  If option is not set, all the open days of the month
 *      are selected.  Valid values for option are: First, Second, Third,
 *      Fourth, Fifth, Last, 1, 2, 3, 4, 5, 6, 7, 8, ..., -1 (for last open
 *      day of the month), -2, -3, -4, -5, ....  The option field is
 *      case insensitive.  For instance 1 (or first) means the first open
 *      day of the month, -1 (or last) stands for the last open day of the
 *      month.
 *
 *    + Easter or Easter+n or Easter-n
 *      for Easter plus or less a number of days. For example, Easter-2 stands
 *      for 2 days before Easter.
 *
 *    + Paskha or Paskha+n or Paskha-n
 *      for Paskha (Orthodox Easter) plus or less a number of days. For
 *      example, Paskha+21 stands for 21 days after Paskha.
 *
 *  Not or ! at the beginning of the calendar string means `Every day except
 *  the ones defined by the following calendar string'
 *
 *  Full examples:
 *      1/Thur
 *            Every Thursday of January
 *      Jan-March/open
 *            Every open days (Monday to Friday) in January, February and March
 *      1-12/Saturday,Sunday
 *            Every week-end
 *      7/Fri/First
 *            First Friday in July
 *      05/01
 *            1st May
 *      Jan/1/Sat+2,Sun+1
 *            1st January.  If the 1st January occurs a week-end, moves it to
 *            the following Monday (Sat+2 = Monday and Sun+1 = Monday)
 *      Easter+49
 *            49 days after Easter (Pentecost-Whitsunday)
 *      11/Sun/-1
 *            Last Sunday of November
 *      12/open/last
 *            Last open day of the year
 *      Jan/OPEN/1
 *            First open day of the year
 *      2/2,3
 *            2nd and 3rd of February
 *      not 1/sat,sun
 *            Every day except the Saturdays and Sundays in January
 *      ! 1-6/open/last
 *            Every day except the last open days in January to June
 */

#include <schedwi.h>

#include <time.h>
#include <strings.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <assert.h>

#include <lib_functions.h>
#include <calendar.h>

#ifndef HAVE_STRNCASECMP
#define strncasecmp(s1,s2,n) schedwi_strncasecmp(s1,s2,n)
#endif

/*
 * Check if the provided year is a leap year
 */
static int
is_leap_year (int year)
{
	return ((year % 4 == 0) && (year % 100)) || (year % 400 == 0);
}


/*
 * Return the last day number of the provided month for the provided year
 * (or -1 if the month or the year is out of range)
 */
int
get_last_day (int year, int month)
{
	if (month < 1 || month > 12 || year < 0) {
		return -1;
	}

	switch (month) {
		case 1:
		case 3:
		case 5:
		case 7:
		case 8:
		case 10:
		case 12:
			return 31;
		case 2:
			if (is_leap_year (year)) {
				return 29;
			}
			return 28;
		default:
			return 30;
	}
}


/*
 * Compute and return the week day (0=Sunday, 1=Monday, ..., 6=Saturday)
 * (return -1 if the day, the month or the year is out of range)
 */
int
day_of_week (int day, int month, int year)
{
	int y, m;

	if (month < 1 || month > 12 || year < 0 || day < 1 || day > 31) {
		return -1;
	}

	if (month < 3) {
		y = year - 1;
		m = 0;
	}
	else {
		y = year;
		m = 2;
	}

	return (	  (23 * month / 9) + day + 4 + year + (y / 4)
			- (y / 100) + (y / 400) - m) % 7;
}


/*
 * Return the day number in the provided month that is not a week-end day
 */
static int
get_open_day (int month, int year, int inc)
{
	int wday, nb_week, mday;

	assert (month > 0 && month < 13);

	if (inc > 0) {
		inc--;
		mday = 1;
		wday = day_of_week (mday, month, year);
		if (wday == 0) {
			wday = 1;
			mday = 2;
		}
		else
		if (wday == 6) {
			wday = 1;
			mday = 3;
		}

		nb_week = inc / 5 + (wday + (inc % 5) > 5);
		return mday + inc + nb_week * 2;
	}
	else {
		inc++;
		mday = get_last_day (year, month);
		wday = day_of_week (mday, month, year);
		if (wday == 0) {
			wday = 5;
			mday -= 2;
		}
		else
		if (wday == 6) {
			wday = 5;
			mday--;
		}
		nb_week = -inc / 5 + (wday + (inc % 5) < 1);
		return mday + inc - nb_week * 2;
	}
}


/*
 * Set easter_month and easter_day to the month number (1-12) and day number
 * (1-31) of Easter for the provided year
 */
static void
get_easter (int year, int *easter_month, int *easter_day)
{
	int a, b, c ,d, e, f, g ,h ,i ,k ,l ,m ,p;

	a = year % 19;
	b = year / 100;
	c = year % 100;
	d = b >> 2;
	e = b & 3;
	f = (b + 8) / 25;
	g = (b - f + 1) / 3;
	h = (19 * a + b -d -g + 15) % 30;
	i = c >> 2;
	k = c & 3;
	l = (32 + (e << 1) + (i << 1) - h -k) % 7;
	m = (a + 11 * h + 22 * l) / 451;
	*easter_month = (h + l - 7 * m + 114) / 31;
	p = (h + l - 7 * m + 114) % 31;
	*easter_day = p + 1;
}


/*
 * Return the number of days in the year for Paskha (Orthodox Easter)
 */
static int
get_paskha (int year)
{
	int a,b ,c, d, e, cumdays;

	a=year % 19;
	b=year %  4;
	c=year %  7;
	d = (19 * a + 15) % 30;
	e = (2 * b + 4 * c + 6 * d + 6) % 7;
	cumdays = 31 + 28;
	if (is_leap_year (year)) {
		cumdays++;
	}
	return cumdays + 22 + d + e + 13;
}


/*
 * Bits field computation
 */
static int
is_bit_set (unsigned long int bit_field, int val)
{
	assert (val > 0);

	return bit_field & (1 << (val - 1));
}

static void
add_day (cal_t *cal, int month, int day)
{
	assert (cal != NULL &&  month > 0 && month < 13 && day > 0 && day < 32);

	cal->month |= 1 << (month - 1);
	cal->year[month - 1] |= 1 << (day - 1);
}

static void
remove_day (cal_t *cal, int month, int day)
{
	assert (cal != NULL &&  month > 0 && month < 13 && day > 0 && day < 32);

	cal->year[month - 1] &= ~(1 << (day - 1));
	if (cal->year[month - 1] == 0) {
		cal->month &= ~(1 << (month - 1));
	}
}

static void
add_range (unsigned long int *bit_field, int min, int max, int length)
{
	assert (bit_field != NULL);

	if (min <= 0) {
		min = 1;
	}
	if (max <= 0) {
		max = length;
	}

	if (min > max) {
		*bit_field |= ~(~0 << max);
		*bit_field |= (~(~0 << (length - min + 1))) << (min -1);
	}
	else {
		*bit_field |= (~(~0 << (max - min + 1))) << (min -1);
	}
}

static void
add_range_week (void *user_data, int min, int max)
{
	assert (user_data != NULL);

	add_range (user_data, min, max, 7);
}

static void
add_range_month (void *user_data, int min, int max)
{
	assert (user_data != NULL);

	add_range (user_data, min, max, 31);
}

/*
 * Convert a negative value to a day number in the month (1-31).
 * For example, -1 is the last day of the month (31 for August for
 * instance), -2 is the day before the last day of the month, ...
 */
static int
convert_mday (int val, int year, int month)
{
	int last_day, ret;;

	assert (month > 0 && month < 13);

	/* If the value is positive, nothing has to be done */
	if (val > 0) {
		return val;
	}
	last_day = get_last_day (year, month);
	ret = last_day + val + 1;
	if (ret <= 0) {
		return last_day + 1;
	}
	return ret;
}

static void
add_range_all_month (void *user_data, int min, int max)
{
	int m, i, j;
	cal_t *cal = user_data;

	assert (user_data != NULL);

	for (m = 0; m < 12; m++) {
		if (is_bit_set (cal->month, m + 1)) {
			i = convert_mday (min, cal->cal_year, m + 1);
			j = convert_mday (max, cal->cal_year, m + 1);
			add_range (&(cal->year[m]), i, j, 31);
		}
	}
}


/*
 * Given a weekday bit field (wday), set this pattern to every week of
 * the selected month of the year
 */
static void
add_week (cal_t *cal, unsigned long int wday, int year)
{
	int m, dow;
	unsigned long int days;

	assert (cal != NULL);

	/* Duplicate the week structure to create a month structure */
	days = 0;
	for (m = 0; m < 5; m++) {
		days |= wday << (m * 7);
	}

	for (m = 0; m < 12; m++) {
		/* If the month is selected */
		if (is_bit_set (cal->month, m + 1)) {
			/*
			 * What is the weekday name of the first day
			 * of the month?
			 */
			dow = day_of_week (1, m + 1, year);

			/* Copy the month pattern */
			cal->year[m] |= days << (7 - dow);
			cal->year[m] |= wday >> dow;
		}
	}
}

/*
 * Add a week pattern to the selected month. The week parameter tells which
 * week to set (1, 2, 3, 4, 5 for the first, second, third, fourth and fifth
 * week of the month or -1, -2, ... for the last, the one before the last, ...
 * week of the month)
 */
static void
add_one_week (cal_t *cal, unsigned long int wday, int year, int week)
{
	int m, i, dow, last;
	unsigned long int days, d;

	assert (cal != NULL);

	days = wday | (wday << 7);

	for (m = 0; m < 12; m++) {
		if (is_bit_set (cal->month, m + 1) == 0) {
			continue;
		}
		/* First, second, third, ... */
		if (week >= 0) {
			dow = day_of_week (1, m + 1, year);
			d = (days >> dow) & 0x7F;
			cal->year[m] |= d << (7 * (week - 1));
		}
		/* Last, one before last, ... */
		else {
			last = get_last_day (year, m + 1);
			dow = day_of_week (last, m + 1, year);
			d = (days >> (dow + 1)) & 0x7F;
			i = last + 7 * week;
			if (i < 0) {
				cal->year[m] |= d >> -i;
			}
			else {
				cal->year[m] |= d << i;
			}
		}
	}
}


/*
 * Parse a range
 *
 * Examples of range:
 *       1-10: value must be between 1 and 10 (included)
 *        3  : value must be 3
 *      15--1: value must be between 15 and the last possible value (for a
 *             month, it's the last day)
 *        *  : always match
 * The provided parse function is used to parse both side of the range and must
 * return an integer (and 0 in case of error)
 * The provided set_range function is called to set a value. The first
 * parameter is the provided user_data
 * In case of parsing error (syntax error), *error_idx_in_str is set with
 * the index of the error in str
 *
 * Return:
 *    CAL_NOERROR --> No error
 *    CAL_BADMONTHNAME --> Invalid month name
 *    CAL_BADDAYNAME --> Invalid day name
 *    CAL_MONTHOOR --> Month number out of range
 *    CAL_DAYOOR --> Day number out of range (1-31)
 *    CAL_EMPTYFIELD --> The last part of the range is missing (Mon-)
 */
static cal_errcode_t
parse_range (	char *str, int *error_idx_in_str,
		int (*parse)(const char *, cal_errcode_t*),
		void (*set_range)(void *, int, int),
		void *user_data)
{
	char * last;
	int min, max, idx;
	cal_errcode_t error;

	assert (str != NULL && parse != NULL && set_range != NULL);

	idx = 0;
	while (isspace (*str)) {
		idx++;
		str++;
	}

	last = strchr (str, '-');
	/* Single value */
	if (	   last == NULL
		|| (str == last && (last = strchr (last + 1, '-')) == NULL))
	{
		min = parse (str, &error);
		if (min == 0) {
			if (error_idx_in_str != NULL) {
				*error_idx_in_str = idx;
			}
			return error;
		}
		set_range (user_data, min, min);
		return CAL_NOERROR;
	}

	*last = '\0';

	min = parse (str, &error);
	if (min == 0) {
		if (error_idx_in_str != NULL) {
			*error_idx_in_str = idx;
		}
		return error;
	}

	last += 1;
	idx += last - str;
	while (isspace (*last)) {
		idx++;
		last++;
	}

	/* Missing last value */
	if (*last == '\0') {
		if (error_idx_in_str != NULL) {
			*error_idx_in_str = idx - 1;
		}
		return CAL_EMPTYFIELD;
	}

	/* 1-10 or Jan-Nov or 2-Sep or Mon-Fri */
	max = parse (last, &error);
	if (max == 0 ) {
		if (error_idx_in_str != NULL) {
			*error_idx_in_str = idx;
		}
		return error;
	}
	set_range (user_data, min, max);
	return CAL_NOERROR;
}


/*
 * Split a list of values (2,4,5,6-10) and call parse_range for each of them.
 * In case of parsing error (syntax error), *error_idx_in_str is set with
 * the index of the error in str
 *
 * Return:
 *    CAL_NOERROR --> No error
 *    CAL_MALLOC --> Memory allocation error
 *    CAL_EMPTYFIELD --> No value to parse or missing value
 *    CAL_BADMONTHNAME --> Invalid month name
 *    CAL_BADDAYNAME --> Invalid day name
 *    CAL_MONTHOOR --> Month number out of range
 *    CAL_DAYOOR --> Day number out of range (1-31)
 */
static cal_errcode_t
parse_list (	const char *str, int *error_idx_in_str,
		int (*parse)(const char *, cal_errcode_t *),
		void (*set_range)(void *, int, int),
		void *user_data)
{
	char *s, *svg, *tok;
	cal_errcode_t ret;

	assert (str != NULL && parse != NULL && set_range != NULL);

	s = (char *) malloc (strlen (str) + 1);
	if (s == NULL) {
		return CAL_MALLOC;
	}

	strcpy (s, str);

	tok = strtok_r (s, ",", &svg);
	if (tok == NULL) {
		free (s);
		if (error_idx_in_str != NULL) {
			*error_idx_in_str = 0;
		}
		return CAL_EMPTYFIELD;
	}
	while (tok != NULL) {
		ret = parse_range (	tok, error_idx_in_str,
					parse, set_range, user_data);
		if (ret != CAL_NOERROR) {
			if (error_idx_in_str != NULL) {
				*error_idx_in_str += tok - s;
			}
			free (s);
			return ret;
		}
		tok = strtok_r (NULL, ",", &svg);
	}
	free (s);
	return CAL_NOERROR;
}


/*
 * Convert a month name to a number (1 to 12)
 *
 * Return:
 *   The number of months since January, in the range 1 to 12 or
 *   0 if month_name is invalid
 */
static int
get_month_number (const char *month_name)
{
	assert (month_name != NULL);

	if (strncasecmp (month_name, "Jan", 3) == 0) {
		return 1;
	}
	if (strncasecmp (month_name, "Feb", 3) == 0) {
		return 2;
	}
	if (strncasecmp (month_name, "Mar", 3) == 0) {
		return 3;
	}
	if (strncasecmp (month_name, "Apr", 3) == 0) {
		return 4;
	}
	if (strncasecmp (month_name, "May", 3) == 0) {
		return 5;
	}
	if (strncasecmp (month_name, "Jun", 3) == 0) {
		return 6;
	}
	if (strncasecmp (month_name, "Jul", 3) == 0) {
		return 7;
	}
	if (strncasecmp (month_name, "Aug", 3) == 0) {
		return 8;
	}
	if (strncasecmp (month_name, "Sep", 3) == 0) {
		return 9;
	}
	if (strncasecmp (month_name, "Oct", 3) == 0) {
		return 10;
	}
	if (strncasecmp (month_name, "Nov", 3) == 0) {
		return 11;
	}
	if (strncasecmp (month_name, "Dec", 3) == 0) {
		return 12;
	}
	return 0;
}


/*
 * Parse a month (1-12 or Jan-Dec)
 *
 * Return:
 *   The number of months since January, in the range 1 to 12 or
 *   0 if month_str is invalid (*error is set with the correct error code)
 */
static int
parse_month (const char *month_str, cal_errcode_t *error)
{
	int i, m, ret;

	assert (month_str != NULL);

	for (i = 0; isspace (month_str[i]); i++);
	if (isdigit (month_str[i])) {
		m = atoi (month_str + i);
		if (m < 1 || m > 12) {
			if (error != NULL) {
				*error = CAL_MONTHOOR;
			}
			return 0;
		}
		return m;
	}
	ret = get_month_number (month_str + i);
	if (ret == 0 && error != NULL) {
		*error = CAL_BADMONTHNAME;
	}
	return ret;
}


/*
 * Convert a day name to a number (1-7)
 *
 * Return:
 *   The number of days since Sunday, in the range 1 to 7 or
 *   0 if day_name is invalid
 */
static int
get_day_number (const char *day_name)
{
	assert (day_name != NULL);

	if (strncasecmp (day_name, "Sun", 3) == 0) {
		return 1;
	}
	if (strncasecmp (day_name, "Mon", 3) == 0) {
		return 2;
	}
	if (strncasecmp (day_name, "Tue", 3) == 0) {
		return 3;
	}
	if (strncasecmp (day_name, "Wed", 3) == 0) {
		return 4;
	}
	if (strncasecmp (day_name, "Thu", 3) == 0) {
		return 5;
	}
	if (strncasecmp (day_name, "Fri", 3) == 0) {
		return 6;
	}
	if (strncasecmp (day_name, "Sat", 3) == 0) {
		return 7;
	}
	return 0;
}


/*
 * Parse a weekday name
 *
 * Return:
 *   The number of days since Sunday, in the range 1 to 7 or
 *   0 if day_str is invalid (*error is set with the correct error code)
 */
static int
parse_wday (const char *day_str, cal_errcode_t *error)
{
	int i, ret;

	assert (day_str != NULL);

	for (i = 0; isspace (day_str[i]); i++);
	ret = get_day_number (day_str + i);
	if (ret == 0 && error != NULL) {
		*error = CAL_BADDAYNAME;
	}
	return ret;
}


/*
 * Parse a day in the month (-1 to -31 or 1 to 31: negative values tell
 * how many days from the end of the month (ie. -1 for the last day of
 * the month))
 *
 * Return:
 *   The day of the month, in the range 1 to 31 or -1 to -31 or
 *   0 if day_str is invalid (*error is set with the correct error code)
 */
static int
parse_mday (const char *day_str, cal_errcode_t *error)
{
	int i, d;

	assert (day_str != NULL);

	for (i = 0; isspace (day_str[i]); i++);
	d = atoi (day_str + i);
	if (d > 31 || d < -31) {
		if (error != NULL) {
			*error = CAL_DAYOOR;
		}
		return 0;
	}
	return d;
}


/*
 * Set a day plus an increment.
 * increment is the number of days to add or remove (ie. +1 to add
 * one day, -3 to remove 3 days)
 */
static void
set_day_increment (cal_t *cal, int year, int month, int day, int increment)
{
	struct tm stm;
	time_t tt;

	assert (cal != NULL);

	stm.tm_year = year - 1900;
	stm.tm_mon  = month - 1;
	stm.tm_mday = day;
	stm.tm_hour = 12;
	stm.tm_min  = 0;
	stm.tm_sec  = 0;
	tt = mktime (&stm) + increment * 86400;
	localtime_r (&tt, &stm);
	if (year - 1900 == stm.tm_year) {
		add_day (cal, stm.tm_mon + 1, stm.tm_mday);
	}
}


/*
 * Move a day.
 * The string date_str is the number of days to move the provided day
 * (it may be negative)
 */
static void
move_day (cal_t *cal, int year, int month, int day, int increment)
{
	assert (cal != NULL);

	remove_day (cal, month, day);
	set_day_increment (cal, year, month, day, increment);
}


/*
 * Move a week day
 */
static void
move_week_day (cal_t *cal, int year, int wday, int increment)
{
	int m, d, last_day;

	for (m = 1; m <= 12; m++) {
		if (is_bit_set (cal->month, m)) {
			last_day = get_last_day (year, m);
			for (d = 1; d <= last_day; d++) {
				if (day_of_week (d, m, year) == wday) {
					if (is_bit_set (cal->year[m - 1],
							d) != 0)
					{
						move_day (	cal,
								year, m, d,
								increment);
					}
					d += 6;
				}
			}
		}
	}
}


/*
 * Parse a week day and move it
 */
static cal_errcode_t
parse_and_move_week_day (	cal_t *cal, int year, const char *day_str,
				int *error_idx_in_daystr)
{
	int i, wday;

	assert (cal != NULL && day_str != NULL);

	for (i = 0; isspace (day_str[i]); i++);
	wday = get_day_number (day_str + i);
	if (wday == 0) {
		if (error_idx_in_daystr != NULL) {
			*error_idx_in_daystr = i;
		}
		return CAL_BADDAYNAME;
	}
	wday--;

	while (day_str[i] != '\0' && day_str[i] != '+' && day_str[i] != '-') {
		i++;
	}
	if (day_str[i] == '\0') {
		if (error_idx_in_daystr != NULL) {
			*error_idx_in_daystr = i;
		}
		return CAL_EMPTYFIELD;
	}

	move_week_day (cal, year, wday, atoi (day_str + i));
	return CAL_NOERROR;
}


/*
 * Parse the option string used to move days if they fall on some
 * week days.
 * In case of parsing error (syntax error), *error_idx_in_str is set with
 * the index of the error in str
 * The syntax (in str) is as follow:
 *    <wday1>+<nb>,<wday2>-<nb>,...
 * with
 *    wday1 and wday2: weekday names to move
 *    +<nb> and -<nb>: number of day to move the weekday
 *
 * For instance:
 *    Sat+2,Sun+1 --> Will move all the Saturdays and Sundays to Mondays
 *    Sat-1,Sun+1 --> Will move all the Saturdays to Fridays and the Sundays
 *                    to Mondays
 * Warning:
 *    Sat+1,Sun-2 --> Will move all the Saturdays and Sundays to Fridays!
 *                    The string is parse from left to right, so Sat+1 will
 *                    move all the Saturdays to Sundays and then the Sun-2 will
 *                    move all the Sundays (included the previously moved
 *                    Saturdays) to Fridays
 *
 * Return:
 *    CAL_NOERROR --> No error
 *    CAL_MALLOC --> Memory allocation error
 *    CAL_EMPTYFIELD --> No value to parse or missing value
 *    CAL_BADDAYNAME --> Invalid day name
 */
static cal_errcode_t
parse_option_day_move (	cal_t *cal, int year, const char *str,
			int *error_idx_in_str)
{
	char *s, *svg, *tok;
	cal_errcode_t ret;

	assert (cal != NULL && str != NULL);

	s = (char *) malloc (strlen (str) + 1);
	if (s == NULL) {
		return CAL_MALLOC;
	}

	strcpy (s, str);

	tok = strtok_r (s, ",", &svg);
	if (tok == NULL) {
		free (s);
		if (error_idx_in_str != NULL) {
			*error_idx_in_str = 0;
		}
		return CAL_EMPTYFIELD;
	}
	while (tok != NULL) {
		ret = parse_and_move_week_day (	cal, year, tok,
						error_idx_in_str);
		if (ret != CAL_NOERROR) {
			if (error_idx_in_str != NULL) {
				*error_idx_in_str += tok - s;
			}
			free (s);
			return ret;
		}
		tok = strtok_r (NULL, ",", &svg);
	}
	free (s);
	return CAL_NOERROR;
}


/*
 * Compute Easter for the provided year
 * The string date_str is increment from Easter (ie. +1 for
 * the day following Easter or -42 for 42 days before Easter)
 */
static void
calcomp_easter (cal_t *cal, const char *date_str, int year)
{
	int month, day;

	assert (cal != NULL && date_str != NULL);

	get_easter (year, &month, &day);
	set_day_increment (cal, year, month, day, atoi (date_str));
}


/*
 * Compute Paskha for the provided year
 * The string date_str is increment from Paskha (ie. +1 for
 * the day following Paskha or -42 for 42 days before Paskha)
 */
static void
calcomp_paskha (cal_t *cal, const char *date_str, int year)
{
	int yday;
	time_t paskha_t;
	struct tm paskha_tm;

	assert (cal != NULL && date_str != NULL);

	yday = get_paskha (year);

	paskha_tm.tm_year = year - 1900;
	paskha_tm.tm_mon  = 0;
	paskha_tm.tm_mday = 1;
	paskha_tm.tm_hour = 12;
	paskha_tm.tm_min  = 0;
	paskha_tm.tm_sec  = 0;
	paskha_t = mktime (&paskha_tm) + (yday - 1 + atoi (date_str)) * 86400;
	localtime_r (&paskha_t, &paskha_tm);
	if (year - 1900 == paskha_tm.tm_year) {
		add_day (cal, paskha_tm.tm_mon + 1, paskha_tm.tm_mday);
	}
}


/*
 * Parse the month part
 *                   cal: cal structure
 *    	       month_str: String to parse
 * 	error_idx_in_str: In case of syntax error, index of the error in
 * 	                  month_str
 *
 * Example of month string:
 *                 *: Every month
 * 	           2: February
 * 	         1-6: January to June (inclusive)
 * 	feb-December: February to December
 * 	     1-4,dec: list of values (January to April and December)
 *
 * Return:
 *    CAL_NOERROR --> No error
 *    CAL_MALLOC --> Memory allocation error
 *    CAL_EMPTYFIELD --> No value to parse or missing value
 *    CAL_BADMONTHNAME --> Invalid month name
 *    CAL_BADDAYNAME --> Invalid day name
 *    CAL_MONTHOOR --> Month number out of range
 *    CAL_DAYOOR --> Day number out of range (1-31)
 */
static cal_errcode_t
calcomp_month (cal_t *cal, const char *month_str, int *error_idx_in_str)
{
	assert (cal != NULL && month_str != NULL);

	if (*month_str == '*') {
		cal->month = ~0;
		return CAL_NOERROR;
	}

	return parse_list (	month_str, error_idx_in_str,
				parse_month, add_range_month, &(cal->month));
}


/*
 * Parse an option string
 *
 * Return:
 *    The parsed integer or
 *    0 in case of syntax error
 */
static int
parse_inc (const char *day_opt)
{
	int val;

	assert (day_opt != NULL);

	val = atoi (day_opt);
	if (val == 0) {
		if (strncasecmp (day_opt, "First", 5) == 0) {
			val = 1;
		}
		else
		if (strncasecmp (day_opt, "Last", 4) == 0) {
			val = -1;
		}
		else
		if (strncasecmp (day_opt, "Second", 6) == 0) {
			val = 2;
		}
		else
		if (strncasecmp (day_opt, "Third", 5) == 0) {
			val = 3;
		}
		else
		if (strncasecmp (day_opt, "Fourth", 6) == 0) {
			val = 4;
		}
		else
		if (strncasecmp (day_opt, "Fifth", 5) == 0) {
			val = 5;
		}
	}
	return val;
}


/*
 * Parse the option string
 *
 * Return:
 *    CAL_NOERROR --> No error
 *    CAL_BADOPTION --> The option string is invalid (1, -1, First, Last, ...)
 *                      error_idx_in_dayopt is set with the index of the error
 *                      in day_opt
 */
static cal_errcode_t
parse_option (	cal_t *cal, const char *day_opt,
		unsigned long int wday, int year, int *error_idx_in_dayopt)
{
	int val;

	assert (cal != NULL && day_opt != NULL);

	val = parse_inc (day_opt);
	if (val == 0) {
		if (error_idx_in_dayopt != NULL) {
			*error_idx_in_dayopt = 0;
		}
		return CAL_BADOPTION;
	}
	add_one_week (cal, wday, year, val);
	return CAL_NOERROR;
}


/*
 * Parse the option string
 *
 * Return:
 *    CAL_NOERROR --> No error
 *    CAL_BADOPTION --> The option string is invalid (1, -1, First, Last, ...)
 *                      error_idx_in_dayopt is set with the index of the error
 *                      in day_opt
 */
static cal_errcode_t
parse_open_day (cal_t *cal, const char *day_opt, int year,
		int *error_idx_in_dayopt)
{
	int val, m, n, ret;
	unsigned long int wday;

	assert (cal != NULL);

	for (n = 0; day_opt != NULL && isspace (day_opt[n]); n++);
	if (day_opt == NULL || day_opt[n] == '\0') {
		wday = 0;
		add_range_week (&wday, 2, 6);
		add_week (cal, wday, year);
		return CAL_NOERROR;
	}

	val = parse_inc (day_opt + n);
	if (val == 0) {
		if (error_idx_in_dayopt != NULL) {
			*error_idx_in_dayopt = n;
		}
		return CAL_BADOPTION;
	}
	for (m = 0; m < 12; m++) {
		if (is_bit_set (cal->month, m + 1)) {
			ret = get_open_day (m + 1, year, val);
			if (ret > 0 && ret < 32) {
				add_day (cal, m + 1, ret);
			}
		}
	}
	return CAL_NOERROR;
}


/*
 * Parse the day part and the optional modifier.
 *                   cal: cal structure
 * 	         day_str: Day string to parse
 * 	         day_opt: Option string to parse
 * 	            year: Year to use to compile the calendar
 *   error_idx_in_daystr: In case of syntax error, index of the error in
 * 	                  day_str
 *
 * If the day string start by a number, it relates to the day number in the
 * month. In this case, the option modifier is not used.
 * If not, it relates to the day in the week.
 *
 * Example of day string:
 *                  *: Every day of the week
 * 	          Mon: Monday
 * 	            1: First day of the month
 * 	         2-12: Second to 12th day of the month
 * 	      Tue-Fri: Tuesday to Friday (included)
 *  	    1-4,21-25: list of values (1 to 4 and 21 to 25)
 *  	           -1: Last day of the month
 *  	        20--2: From the 20th to the last but one day of the month
 *
 * Example of option string:
 *         Last or -1: Last week in the month
 *   First or 1 or +1: First week in the month
 *  Second or 2 or +2: Second week in the month
 *
 * Example of day and option string:
 *          mon/first: First monday of the month
 *       mon-wed/last: Monday to Friday of the last week of the month
 *  Monday,sat/Second: Monday and Saturday of the second week of the month
 *
 * Return:
 *    CAL_NOERROR --> No error
 *    CAL_MALLOC --> Memory allocation error
 *    CAL_EMPTYFIELD --> No value to parse or missing value
 *    CAL_BADMONTHNAME --> Invalid month name
 *    CAL_BADDAYNAME --> Invalid day name
 *    CAL_MONTHOOR --> Month number out of range
 *    CAL_DAYOOR --> Day number out of range (1-31)
 *    CAL_BADOPTION --> The option string is invalid (1, -1, First, Last, ...)
 */
static cal_errcode_t
calcomp_day (	cal_t *cal, const char *day_str, const char *day_opt, int year,
		int *error_idx_in_daystr)
{
	int n;
	unsigned long int day;
	cal_errcode_t ret;

	assert (cal != NULL && day_str != NULL);

	for (n = 0; isspace (day_str[n]); n++);

	/* Day number in the month: 1-31 */
	if (atoi (day_str + n) != 0) {
		ret = parse_list (	day_str + n, error_idx_in_daystr,
					parse_mday, add_range_all_month, cal);
		if (ret != CAL_NOERROR) {
			if (error_idx_in_daystr != NULL) {
				*error_idx_in_daystr += n;
			}
			return ret;
		}

		/* Day options (Sat+2,Sun+1, ...) */
		for (n = 0; day_opt != NULL && isspace (day_opt[n]); n++);
		if (day_opt == NULL || day_opt[n] == '\0') {
			return CAL_NOERROR;
		}
		ret = parse_option_day_move (	cal, year, day_opt + n,
						error_idx_in_daystr);
		if (ret != CAL_NOERROR && error_idx_in_daystr != NULL) {
			*error_idx_in_daystr += day_opt - day_str + n;
		}
		return ret;
	}

	/* Open */
	if (strncasecmp (day_str + n, "open", 4) == 0) {
		ret = parse_open_day (cal, day_opt, year, error_idx_in_daystr);
		if (ret != CAL_NOERROR && error_idx_in_daystr != NULL) {
			*error_idx_in_daystr += day_opt - day_str;
		}
		return ret;
	}

	/* Day in the week: Mon-Sun or * */
	if (day_str[n] == '*') {
		day = ~0;
	}
	else {
		day = 0;
		ret = parse_list (	day_str + n, error_idx_in_daystr,
					parse_wday, add_range_week, &day);
		if (ret != CAL_NOERROR) {
			if (error_idx_in_daystr != NULL) {
				*error_idx_in_daystr += n;
			}
			return ret;
		}
	}

	/* Day options (first, second, last, 1, +1, 2, +2, ...) */
	for (n = 0; day_opt != NULL && isspace (day_opt[n]); n++);
	if (day_opt == NULL || day_opt[n] == '\0') {
		add_week (cal, day, year);
		return CAL_NOERROR;
	}
	ret = parse_option (cal, day_opt + n, day, year, error_idx_in_daystr);
	if (ret != CAL_NOERROR && error_idx_in_daystr != NULL) {
		*error_idx_in_daystr += day_opt - day_str + n;

	}
	return ret;
}


/*
 * Invert the calendar
 */
static void
calinvert (cal_t *cal)
{
	int m;

	assert (cal != NULL);

	for (m = 0; m < 12; m++) {
		if (is_bit_set (cal->month, m + 1)) {
			cal->year[m] = ~(cal->year[m]);
		}
		else {
			cal->month |= 1 << m;
			cal->year[m] = ~0;
		}
	}
}


/*
 * Compile a calendar string into a form that is suitable for subsequent
 * calendar checks.
 *                cal: a pointer to a calendar storage area
 *           calendar: string to parse
 *               year: year (used to compute Easter). Subsequent calendar
 *                     checks must be in this year
 *   error_idx_in_str: In case of syntax error, index of the error in
 * 	               calendar
 *
 * Example of calendar string:
 *           Jan/mon (or 1/mon): Every Monday in January
 *                 1-6/Sun/Last: Last Sunday on the month January to September
 *                    Easter+39: 39 days after Easter
 *                     easter-2: 2 days before Easter
 *                     jan/open: All the open days (Monday to Friday) in
 *                               January
 *               feb/open/first: First open day in February
 *                  1-3/Open/-1: Last open day in month from January to March
 *                   not easter: Every day except Easter
 *
 * Return:
 *    CAL_NOERROR --> No error
 *    CAL_MALLOC --> Memory allocation error
 *    CAL_EMPTYFIELD --> No value to parse or missing value
 *    CAL_BADMONTHNAME --> Invalid month name
 *    CAL_BADDAYNAME --> Invalid day name
 *    CAL_MONTHOOR --> Month number out of range
 *    CAL_DAYOOR --> Day number out of range (1-31)
 *    CAL_BADOPTION --> The option string is invalid (1, -1, First, Last, ...)
 */
cal_errcode_t
calcomp (cal_t *cal, const char *calendar, int year, int *error_idx_in_str)
{
	char *s, *svg, *month_str, *day_str, *opt_str, invert_cal;
	int n;
	cal_errcode_t ret;

	if (cal == NULL || calendar == NULL) {
		return CAL_NOERROR;
	}

	for (n = 0; isspace (calendar[n]); n++);

	/* Invert */
	if (calendar[n] == '!') {
		invert_cal = 1;
		n++;
	}
	else {
		if (strncasecmp (calendar + n, "not", 3) == 0) {
			invert_cal = 1;
			n += 3;
		}
		else {
			invert_cal = 0;
		}
	}

	while (isspace (calendar[n]) != 0) {
		n++;
	}

	/* Copy the string to parse */
	s = (char *) malloc (strlen (calendar + n) + 1);
	if (s == NULL) {
		return CAL_MALLOC;
	}
	strcpy (s, calendar + n);

	/* Initialize the calendar structure */
#if HAVE_MEMSET
	memset (cal, 0, sizeof (cal_t));
#else
	schedwi_memset (cal, 0, sizeof (cal_t));
#endif
	cal->cal_year = year;

	month_str = strtok_r (s, "/", &svg);
	if (month_str == NULL) {
		free (s);
		if (invert_cal != 0) {
			calinvert (cal);
		}
		return CAL_NOERROR;
	}

	/* Easter */
	if (strncasecmp (month_str, "Easter", 6) == 0) {
		calcomp_easter (cal, month_str + 6, year);
		free (s);
		if (invert_cal != 0) {
			calinvert (cal);
		}
		return CAL_NOERROR;
	}

	/* Paskha */
	if (strncasecmp (month_str, "Paskha", 6) == 0) {
		calcomp_paskha (cal, month_str + 6, year);
		free (s);
		if (invert_cal != 0) {
			calinvert (cal);
		}
		return CAL_NOERROR;
	}

	ret = calcomp_month (cal, month_str, error_idx_in_str);
	if (ret != CAL_NOERROR) {
		if (error_idx_in_str != NULL) {
			*error_idx_in_str += month_str - s + n;
		}
		free (s);
		return ret;
	}

	day_str = strtok_r (NULL, "/", &svg);
	if (day_str == NULL) {
		free (s);
		add_range_all_month (cal, 1, 31);
		if (invert_cal != 0) {
			calinvert (cal);
		}
		return CAL_NOERROR;
	}

	opt_str = strtok_r (NULL, "/", &svg);
	ret = calcomp_day (cal, day_str, opt_str, year, error_idx_in_str);
	if (ret != CAL_NOERROR) {
		if (error_idx_in_str != NULL) {
			*error_idx_in_str += day_str - s + n;
		}
		free (s);
		return ret;
	}
	free (s);
	if (invert_cal != 0) {
		calinvert (cal);
	}
	return ret;
}


/*
 * Check that the provided date matches the calendar
 *
 * Return:
 *    CAL_NOERROR --> The date matches the calendar
 *    CAL_NOMATCH --> The date does not match the calendar
 */
cal_errcode_t
calmatch (const cal_t *cal, int day, int month, int year)
{
	assert (cal != NULL);

	if (	   cal->cal_year != year
		|| is_bit_set (cal->month, month) == 0
		|| is_bit_set (cal->year[month - 1], day) == 0)
	{
		return CAL_NOMATCH;
	}
	return CAL_NOERROR;
}


/*
 * Copy the src string to dst and replace all the LF (\n) by end-of-lines (0)
 *
 * Return:
 *   The number of lines (ie. number of \n + 1)
 */
static unsigned int
copy_and_replace_LF (char *dest, const char *src)
{
	size_t i;
	unsigned int num_lines;

	num_lines = 0;
	for (i = 0; src[i] != '\0'; i++) {
		if (src[i] == '\n') {
			num_lines++;
			dest[i] = '\0';
		}
		else {
			dest[i] = src[i];
		}
	}
	dest[i] = '\0';
	return num_lines + 1;
}


/*
 * Compile a multiline calendar string into a form that is suitable for
 * subsequent calendar checks.  The provided string may contains comments (on
 * a line by themselves and starting by the `#' character)
 *                cal: a pointer to a calendar storage area
 *           calendar: string to parse
 *               year: year (used to compute Easter). Subsequent calendar
 *                     checks must be in this year
 *   error_idx_in_str: In case of syntax error, index of the error in
 * 	               calendar
 *
 * Example of calendar string:
 *           Jan/mon (or 1/mon): Every Monday in January
 *                 1-6/Sun/Last: Last Sunday on the month January to September
 *                    Easter+39: 39 days after Easter
 *                     easter-2: 2 days before Easter
 *                     jan/open: All the open days (Monday to Friday) in
 *                               January
 *               feb/open/first: First open day in February
 *                  1-3/Open/-1: Last open day in month from January to March
 *                   not easter: Every day except Easter
 *            12/25/Sat+2,Sun+1: Christmas.  If on a Saturday or Sunday, move
 *                               to the following Monday
 *
 * Return:
 *    CAL_NOERROR --> No error
 *    CAL_MALLOC --> Memory allocation error
 *    CAL_EMPTYFIELD --> No value to parse or missing value
 *    CAL_BADMONTHNAME --> Invalid month name
 *    CAL_BADDAYNAME --> Invalid day name
 *    CAL_MONTHOOR --> Month number out of range
 *    CAL_DAYOOR --> Day number out of range (1-31)
 *    CAL_BADOPTION --> The option string is invalid (1, -1, First, Last, ...)
 */
cal_errcode_t
str2cal (cal_t *cal, const char *calendar, int year, int *error_idx_in_str)
{
	int begin, idx_add, idx_remove, idx_err, m;
	char *work_str;
	size_t len;
	unsigned int num_lines;
	cal_t *array_cal, cal_tmp;
	cal_errcode_t ret;

	if (cal == NULL || calendar == NULL) {
		return CAL_NOERROR;
	}

	/* Copy the string to parse */
	len = strlen (calendar);
	work_str = (char *) malloc (len + 1);
	if (work_str == NULL) {
		return CAL_MALLOC;
	}
	num_lines = copy_and_replace_LF (work_str, calendar);

	/* Allocate the array of cal_t objects */
	array_cal = (cal_t *) malloc (sizeof (cal_t) * num_lines);
	if (array_cal == NULL) {
		free (work_str);
		return CAL_MALLOC;
	}
#if HAVE_MEMSET
	memset (array_cal, 0, sizeof (cal_t) * num_lines);
#else
	schedwi_memset (array_cal, 0, sizeof (cal_t) * num_lines);
#endif
	idx_add = 0;
	idx_remove = num_lines - 1;

	/* Parse each line */
	begin = 0;
	while (begin < len) {

		/* Skip the spaces */
		while (isspace (work_str[begin]) != 0) {
			begin++;
		}
		if (work_str[begin] == '\0') {
			begin++;
			continue;
		}
		if (work_str[begin] == '#') {
			/* Comment */
			while (work_str[begin] != '\0') {
				begin++;
			}
			begin++;
			continue;
		}

		/* Compile the line */
		ret = calcomp (&cal_tmp, work_str + begin, year, &idx_err);
		if (ret != CAL_NOERROR) {
			free (work_str);
			free (array_cal);
			if (error_idx_in_str != NULL) {
				*error_idx_in_str = begin + idx_err;
			}
			return ret;
		}

		/* Invert */
		if (	   work_str[begin] == '!'
			|| strncasecmp (work_str + begin, "not", 3) == 0)
		{
			memcpy (array_cal + idx_remove--,
				&cal_tmp, sizeof (cal_t));
		}
		else {
			memcpy (array_cal + idx_add++,
				&cal_tmp, sizeof (cal_t));
		}

		/* Go to the next line */
		while (work_str[begin] != '\0') {
			begin++;
		}
		begin++;
	}
	free (work_str);

	/* Initialize the calendar structure */
#if HAVE_MEMSET
	memset (cal, 0, sizeof (cal_t));
#else
	schedwi_memset (cal, 0, sizeof (cal_t));
#endif
	cal->cal_year = year;

	/* Add all the cal_t object to the returned calendar */
	for (begin = 0; begin <= idx_remove; begin++) {
		cal->month |= array_cal[begin].month;
		for (m = 0; m < 12; m++) {
			cal->year[m] |= array_cal[begin].year[m];
		}
	}
	for (;begin < num_lines; begin++) {
		cal->month &= array_cal[begin].month;
		for (m = 0; m < 12; m++) {
			cal->year[m] &= array_cal[begin].year[m];
		}
	}
	free (array_cal);

	return CAL_NOERROR;
}

/*------------------------======= End Of File =======------------------------*/
