# Schedwi
# Copyright (C) 2013 Herve Quatremain
#
# This file is part of Schedwi.
#
# Schedwi is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
#
# Schedwi is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

"""Workload selection calendar."""

import calendar
from datetime import datetime

from muntjac.api import (VerticalLayout, InlineDateField, GridLayout, Label,
                        Alignment, PopupView, NativeButton)
from muntjac.ui import popup_view
from muntjac.ui.button import IClickListener
from muntjac.data.property import IValueChangeListener
from muntjac.addon.refresher.refresher import Refresher
from muntjac.addon.refresher.refresher import RefreshListener

import locale_utils
import simple_queries_wl
import status_utils


class SelectWorkload(PopupView):
    def __init__(self, sql_session):
        self._month_view = MonthView(self, sql_session)
        super(SelectWorkload, self).__init__(self._month_view)

    def get_selected_workload(self):
        return self._month_view.get_selected_workload()

    def get_selected_workload_str(self):
        return self._month_view.get_selected_workload_str()

    def get_refresher(self):
        return self._month_view.get_refresher()


class MonthView(popup_view.IContent, IValueChangeListener, RefreshListener):
    def __init__(self, select_workload_obj, sql_session,
                 month=None, year=None):
        super(MonthView, self).__init__()
        self._c = select_workload_obj
        self._sql_session = sql_session
        d = datetime.today()
        self._year = d.year if year is None else year
        self._month = d.month if month is None else month
        self._locale = locale_utils.get_locale()
        self._firstweekday = self._locale.first_week_day  # 0: Monday, ...
        self._selected_workload = None
        self._status = None

        self._root = VerticalLayout()
        self._root.setSizeUndefined()
        self._root.setSpacing(False)
        self._root.setMargin(True)

        self._datetime = InlineDateField()
        self._datetime.setLocale(self._locale)
        self._datetime.setResolution(InlineDateField.RESOLUTION_MONTH)
        d = datetime(self._year, self._month, 15)
        self._datetime.setValue(d)
        self._datetime.setImmediate(True)
        self._datetime.addListener(self, IValueChangeListener)
        self._root.addComponent(self._datetime)

        self._day_grid = GridLayout(7, 7)
        self._day_grid.setWidth('100%')
        # First row is day names
        for i in range(7):
            j = (i + self._firstweekday) % 7
            l = Label(
                self._locale.days['format']['abbreviated'][j].encode('utf-8'))
            l.addStyleName("calweekdayname")
            self._day_grid.addComponent(l, i, 0)
        self._root.addComponent(self._day_grid)
        self.refresh()
        self._refresher = Refresher()
        self._refresher.setRefreshInterval(60000)   # Every minute
        self._refresher.addListener(self, RefreshListener)

    def get_refresher(self):
        return self._refresher

    def draw_month(self, month=None, year=None):
        if month is not None:
            self._month = month
        if year is not None:
            self._year = year
        c = calendar.Calendar(self._firstweekday)
        calendar.setfirstweekday(self._firstweekday)
        i = 0
        for d in c.itermonthdates(self._year, self._month):
            col = i % 7
            row = i / 7 + 1
            self._day_grid.removeComponent(col, row)
            if d.month == self._month:
                try:
                    js = self._status[self._year][self._month][d.day]
                except:
                    b = Label(str(d.day))
                    b.setStyleName('daybutton')
                    b.setSizeUndefined()
                else:
                    b = NativeButton(str(d.day))
                    b.setStyleName('daybutton')
                    b.addStyleName(status_utils.status2colorname(js["status"]))
                    b.addListener(DayClick(self, js), IClickListener)
                self._day_grid.addComponent(b, col, row)
                self._day_grid.setComponentAlignment(b, Alignment.MIDDLE_RIGHT)
            i += 1
        while i < 42:
            self._day_grid.removeComponent(i % 7, i / 7 + 1)
            i += 1

    def refresh(self, source=None):
        try:
            lst = simple_queries_wl.sql_get_workload_list_text(
                                                            self._sql_session)
        except:
            self._selected_workload = None
            self._status = None
        else:
            status = dict()
            first = None
            for js in lst:
                if first is None:
                    first = js
                y = js["datetime"].year
                m = js["datetime"].month
                d = js["datetime"].day
                status[y] = status.get(y, {})
                status[y][m] = status[y].get(m, {})
                status[y][m][d] = js
            if self._selected_workload is None:
                self._selected_workload = first
            else:
                try:
                    dt = self._selected_workload["datetime"]
                    js = status[dt.year][dt.month][dt.day]
                except:
                    self._selected_workload = first
                else:
                    self._selected_workload = js
            self._status = status
        self.draw_month()

    def hide_popup(self):
        self._c.setPopupVisible(False)

    def set_selected_workload(self, value):
        if self._selected_workload is None:
            self._selected_workload = value
        else:
            dt1 = self._selected_workload["datetime"]
            dt2 = value["datetime"]
            if (dt1.year != dt2.year or
                dt1.month != dt2.month or
                dt1.day != dt2.day):
                self._selected_workload = value

    def get_selected_workload(self):
        if self._selected_workload is None:
            return None
        return self._selected_workload["date"]

    def get_selected_workload_str(self):
        if self._selected_workload is not None:
            return self._selected_workload["date_str"]
        else:
            return _("No workloads")

    def getMinimizedValueAsHTML(self):
        return self.get_selected_workload_str()

    def getPopupComponent(self):
        return self._root

    def valueChange(self, event):
        self.draw_month(event.getProperty().getValue().month,
                        event.getProperty().getValue().year)


class DayClick(IClickListener):
    def __init__(self, month_view_obj, job_status_dict):
        super(DayClick, self).__init__()
        self._c = month_view_obj
        self._js = job_status_dict

    def buttonClick(self, event):
        self._c.set_selected_workload(self._js)
        self._c.hide_popup()
