# Schedwi
# Copyright (C) 2012, 2013 Herve Quatremain
#
# This file is part of Schedwi.
#
# Schedwi is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
#
# Schedwi is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

"""Jobset navigation tree UI."""

from muntjac.api import Panel, Tree, HorizontalLayout, NativeButton, Alignment
from muntjac.data.util.hierarchical_container import HierarchicalContainer
from muntjac.data.property import IValueChangeListener
from muntjac.ui.tree import (IExpandListener, ICollapseListener,
                             IItemStyleGenerator)
from muntjac.ui.button import IClickListener
from muntjac.ui.abstract_select import ItemDescriptionGenerator

from simple_queries_job import (sql_get_children, sql_count_children,
                                sql_count_children2, sql_get_job)
from web.cutcopypastemenu import (CutCopyPasteMenuJobsetTree,
                                  CutCopyPasteMenuListener)
from web.autorefresh import AutoRefresh
from web.findwidget import Find
import status_utils


class JobsetTree(Panel, CutCopyPasteMenuListener,
                 IValueChangeListener, IExpandListener, ICollapseListener):

    """Jobset tree UI."""

    _ROOT_JOBSET_ID = 1

    def __init__(self, application, sql_session, workload=None):
        """Create the UI.

        @param application:
                    the L{web.main.Main} object.
        @type application:
                    L{muntjac.api.Application}
        @param sql_session:
                    SQLAlchemy session.
        @param workload:
                    workload to consider.
        """
        super(JobsetTree, self).__init__()
        self._application = application
        self._sql_session = sql_session
        self._workload = workload

        self._listeners = list()
        self._selected = self._ROOT_JOBSET_ID

        self.setCaption(_("Jobsets:"))
        self._tree = Tree()
        self._tree.setSizeFull()
        self._tree.setImmediate(True)
        self._tree.setNullSelectionAllowed(False)
        self._tree.setItemDescriptionGenerator(Tooltips(sql_session, workload))
        self._tree.addStyleName("jobsettree")
        if workload is not None:
            self._tree.setItemStyleGenerator(TreeStyleGenerator(self))
        self.addComponent(self._tree)

        # Container
        self._tree.setContainerDataSource(self._new_container())
        self._tree.setItemCaptionPropertyId('name')

        # Callbacks
        self._tree.addListener(self, IValueChangeListener)
        self._tree.addListener(self, IExpandListener)
        self._tree.addListener(self, ICollapseListener)
        self._tree.addActionHandler(CutCopyPasteMenuJobsetTree(self))

        # Bread crumb
        self._bread_crumb = BreadCrumb(self)
        self.add_listener(self._bread_crumb)

        # Expand the first tree level
        self._tree.expandItem(self._ROOT_JOBSET_ID)
        self._tree.select(self._ROOT_JOBSET_ID)

    def get_breadcrumb(self):
        """Return the breadcrumb L{muntjac.api.HorizontalLayout}."""
        return self._bread_crumb

    def do_select(self, jobset_id):
        """Mark as selected the provided jobset.

        @param jobset_id:   the database ID of the jobset to select.
        """
        if jobset_id is not None:
            self._tree.select(jobset_id)

    def _new_container(self):
        """Create and return a new container data source.

        @return:    the new empty container data source.
        @rtype:
            L{muntjac.data.util.hierarchical_container.HierarchicalContainer}
        """
        c = HierarchicalContainer()
        c.addContainerProperty('name', str, None)
        # Root jobset
        item = c.addItem(self._ROOT_JOBSET_ID)
        item.getItemProperty('name').setValue('/')
        return c

    def add_listener(self, obj):
        """Add a new listener.  Those listeners are called when a jobset is
        selected or unselected.

        @param obj:     the listener object.
        @type obj:      L{JobsetTreeListener}
        """
        self._listeners.append(obj)

    def fire_listeners(self, jobset_ids):
        """Call the listerners previously added with L{add_listener}.

        @param jobset_ids:
                    the list of all the jobset IDs from the root jobset up to
                    the selected one.
        """
        map(lambda obj: obj.refresh(jobset_ids), self._listeners)

    def get_datasource(self):
        """Return the container data source used by the jobset tree.

        @return:    the container data source.
        @rtype:
            L{muntjac.data.util.hierarchical_container.HierarchicalContainer}
        """
        return self._tree.getContainerDataSource()

    def _build_jobset_hierarchy(self, jobset_id):
        """Build and return the jobset ID list for the provided jobset ID.

        @param jobset_id:
                    the ID of the jobset for which the list must be built.
        @return:    the list of the jobset IDs from the root jobset up to
                    the provided jobset.
        """
        c = self.get_datasource()
        ids = [jobset_id]
        while True:
            jobset_id = c.getParent(jobset_id)
            if jobset_id is not None:
                ids.insert(0, jobset_id)
            else:
                break
        return ids

    def valueChange(self, event):
        """Callback when a jobset is selected or unselected."""
        AutoRefresh.reset()
        selected_jobset_id = event.getProperty().getValue()
        if selected_jobset_id is not None:
            # If a jobset is selected, build it's jobset hierarchy and call
            # all the listeners (previously added with self.add_listener())
            h = self._build_jobset_hierarchy(selected_jobset_id)
            self.fire_listeners(h)
            self._selected = selected_jobset_id
        else:
            self.fire_listeners(None)
            self._selected = None

    def nodeExpand(self, event):
        """Callback when a jobset is expanded.  Jobset nodes are only
        retrieved from the database when the user expand the tree.
        """
        AutoRefresh.reset()
        jobset_id = event.getItemId()
        c = self.get_datasource()
        session = self._sql_session.open_session()
        for child in sql_get_children(session, jobset_id,
                                      True, self._workload):
            child_id = child.id
            item = c.addItem(child_id)
            item.getItemProperty('name').setValue(child.name.encode('utf-8'))
            c.setParent(child_id, jobset_id)
            # Only show the expand arrow if the jobset has children
            if sql_count_children(session, child_id, True, self._workload):
                c.setChildrenAllowed(child_id, True)
            else:
                c.setChildrenAllowed(child_id, False)
            if (self._selected == child_id and
                self._tree.getValue() != self._selected):
                self._tree.select(child_id)
        self._sql_session.close_session(session)

    def nodeCollapse(self, event):
        """Callback when a jobset is collapsed."""
        AutoRefresh.reset()
        jobset_id = event.getItemId()
        c = self.get_datasource()
        # Remove all the existing children first
        if c.hasChildren(jobset_id):
            for child_id in c.getChildren(jobset_id):
                self._tree.collapseItem(child_id)
                c.removeItemRecursively(child_id)

    def select_jobset(self, jobset_ids):
        """Select the provided jobset.

        @param jobset_ids:
                    the jobset hierarchy.  This is a list of all the parent
                    jobsets up to the root jobset.  The first item of this
                    list is the root jobset and the last is the jobset to
                    select.
        @type jobset_ids: list
        """
        AutoRefresh.reset()
        for i in jobset_ids[:-1]:
            if not self._tree.isExpanded(i):
                self._tree.expandItem(i)
        self._tree.select(jobset_ids[-1])

    def refresh(self):
        """Refresh (reload) the view.  This method is called:
             - regularly by a L{muntjac.addon.refresher.refresher.Refresher}
             - when job/jobset are moved, copied, deleted...
        """
        # Save the ID of the selected item and the expanded jobsets in the tree
        selected = self._selected
        c = self.get_datasource()
        expandedIds = list()
        for i in c.getItemIds():
            if self._tree.isExpanded(i):
                expandedIds.append(i)

        # Create a new container
        self._tree.collapseItemsRecursively(self._ROOT_JOBSET_ID)
        c = self._new_container()
        self._tree.setContainerDataSource(c)

        # Re-expand the previously expanded items
        while True:
            expand_done = False
            for i in expandedIds:
                if i in c.getItemIds() and not self._tree.isExpanded(i):
                    self._tree.expandItem(i)
                    if self._tree.hasChildren(i):
                        expand_done = True
            if expand_done is False:
                break

        # Re-select the previously selected item
        self._selected = selected
        if selected in c.getItemIds():
            self._tree.select(selected)
        else:
            # If the previously selected item does not exist anymore in
            # the database, select the root jobset
            try:
                sql_get_job(self._sql_session, selected, self._workload)
            except:
                self._tree.select(self._ROOT_JOBSET_ID)

    def get_parent_id(self):
        """From the L{web.cutcopypastemenu.CutCopyPasteMenuListener} class.

        @return:    the database ID of the parent jobset of the selected item.
        """
        if self._selected is None:
            return None
        ids = self._build_jobset_hierarchy(self._selected)
        return ids[-1]

    def get_jobset_hierarchy(self, item_id):
        """From the L{web.cutcopypastemenu.CutCopyPasteMenuListener} class.

        @return:    the hierarchy of database ID of the parent jobsets.
        """
        ids = self._build_jobset_hierarchy(item_id)
        return ids[:-1]

    def get_name(self, item_id):
        """From the L{web.cutcopypastemenu.CutCopyPasteMenuListener} class.

        @return:    the name of the provided item ID.
        """
        c = self.get_datasource()
        return c.getItem(item_id).getItemProperty('name').getValue()

    def repaint(self):
        """From the L{web.cutcopypastemenu.CutCopyPasteMenuListener} class.
        Refresh the view.
        """
        self.refresh()

    def repaint_tools(self):
        """From the L{web.cutcopypastemenu.CutCopyPasteMenuListener} class.
        Repaint the toolbar.
        """
        self.refresh()

    def is_jobset(self, item_id):
        """From the L{web.cutcopypastemenu.CutCopyPasteMenuListener} class.

        @return:    True if the item is a jobset else False.
        """
        return True


class JobsetTreeListener(object):

    """Interface for listening for a new jobset selection."""

    def refresh(self, jobset_hierarchy):
        """Called when a jobset has been selected.

        @param jobset_hierarchy:
                    the jobset hierarchy.  This is a list of all the parent
                    jobsets up to the root jobset.  The first item of this
                    list is the root jobset and the last is the currently
                    selected jobset.  May be None if nothing is selected.
        @type jobset_hierarchy: list
        """
        raise NotImplementedError


class BreadCrumb(HorizontalLayout, JobsetTreeListener):

    """Bread crumb (path of the selected jobset) UI."""

    def __init__(self, jobset_tree_obj):
        """Create the object.

        @param jobset_tree_obj:
                    the associated L{JobsetTree} object.
        @type jobset_tree_obj:
                    L{JobsetTree}
        """
        super(BreadCrumb, self).__init__()
        self._c = jobset_tree_obj
        self._hierarchy_buttons = dict()
        self.setWidth('100%')
        self._bc = HorizontalLayout()
        self._bc.setSpacing(True)
        self.addComponent(self._bc)

        self._cb = Find(self._c, self._c._application,
                        self._c._sql_session, self._c._workload)
        self._cb.setMargin(False, True, False, False)
        self.addComponent(self._cb)
        self.setComponentAlignment(self._cb, Alignment.MIDDLE_RIGHT)
        self.addStyleName("breadcrumb")

    def refresh(self, jobset_hierarchy):
        """Called when a jobset has been selected in the tree. Rebuild the
        bread crumb.

        @param jobset_hierarchy:
                    the jobset hierarchy.  This is a list of all the parent
                    jobsets up to the root jobset.  The first item of this
                    list is the root jobset and the last is the currently
                    selected jobset.  May be None if nothing is selected.
        @type jobset_hierarchy: list
        """
        bc = self._bc
        if jobset_hierarchy is None:
            return
        if jobset_hierarchy[-1] not in self._hierarchy_buttons:
            bc.removeAllComponents()
            self._hierarchy_buttons = dict()
            l = len(jobset_hierarchy)
            for i in range(l):
                name = self._c.get_name(jobset_hierarchy[i])
                b = NativeButton(name)
                b.addStyleName('breadcrumbbt')
                b.addStyleName("notdisplayed")
                b.addListener(BreadCrumbJobsetHandler(self._c,
                                                      jobset_hierarchy[i]),
                              IClickListener)
                bc.addComponent(b)
                self._hierarchy_buttons[jobset_hierarchy[i]] = b
        else:
            to_remove = list()
            session = self._c._sql_session.open_session()
            for key, b in self._hierarchy_buttons.iteritems():
                b.removeStyleName("displayed")
                b.addStyleName("notdisplayed")
                if key not in jobset_hierarchy:
                    try:
                        sql_get_job(session, key, self._c._workload)
                    except:
                        to_remove.append(key)
            self._c._sql_session.close_session(session)
            for jobset_id in to_remove:
                bc.removeComponent(self._hierarchy_buttons[jobset_id])
                del self._hierarchy_buttons[jobset_id]
        self._hierarchy_buttons[jobset_hierarchy[-1]].addStyleName("displayed")
        self._hierarchy_buttons[jobset_hierarchy[-1]].removeStyleName(
                                                            "notdisplayed")


class BreadCrumbJobsetHandler(IClickListener):

    """Callback for when a jobset in the bread crumb is clicked."""

    def __init__(self, jobset_tree_obj, jobset_id):
        """Initialize the callback.

        @param jobset_tree_obj:
                    the associated L{JobsetTree} object.
        @type jobset_tree_obj:    L{JobsetTree}
        @param jobset_id:
                    the database ID of the jobset.
        """
        super(BreadCrumbJobsetHandler, self).__init__()
        self._c = jobset_tree_obj
        self._jobset_id = jobset_id

    def buttonClick(self, event):
        AutoRefresh.reset()
        self._c.do_select(self._jobset_id)


class Tooltips(ItemDescriptionGenerator):

    """Jobset tree item tooltips."""

    def __init__(self, sql_session, workload=None):
        """Initialize the callback.

        @param sql_session:
                    SQLAlchemy session.
        @param workload:
                    workload to consider.
        """
        self._sql_session = sql_session
        self._workload = workload

    def generateDescription(self, source, itemId, propertyId):
        try:
            nb_jobs, nb_jobsets = sql_count_children2(self._sql_session,
                                                      itemId, self._workload)
        except:
            nb_jobs = nb_jobsets = 0
        if nb_jobs == 1:
            s = _("1 job")
        else:
            s = _("%d jobs") % nb_jobs
        if nb_jobsets == 1:
            s += '<br/>' + _("1 jobset")
        else:
            s += '<br/>' + _("%d jobsets") % nb_jobsets
        return s


class TreeStyleGenerator(IItemStyleGenerator):

    """IItemStyleGenerator object to style the tree which the jobset status."""

    def __init__(self, c):
        """Initialize the object.

        @param c:
                    The L{JobsetTree} object.
        """
        self._c = c

    def getStyle(self, itemId):
        s = status_utils.get_status(self._c._sql_session, itemId,
                                    self._c._workload)
        if s:
            return status_utils.status2colorname(s.status)
