# Schedwi
# Copyright (C) 2012, 2013 Herve Quatremain
#
# This file is part of Schedwi.
#
# Schedwi is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
#
# Schedwi is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

"""Environment group selection widget inside the job window."""

from sqlalchemy import desc

from muntjac.api import (VerticalLayout, HorizontalLayout, CheckBox, Button,
                         ListSelect, Alignment, Window, Label, Panel)
from muntjac.ui.button import IClickListener
from muntjac.data.property import IValueChangeListener
from muntjac.ui.themes import BaseTheme
from muntjac.terminal.theme_resource import ThemeResource
from muntjac.data.util.indexed_container import IndexedContainer
from muntjac.ui.window import Notification

from tables.job_environment import job_environment
from tables.job_environment_s import job_environment_s
from tables.host_environment import host_environment
from tables.host_environment_s import host_environment_s
from tables.environment_var import environment_var
from tables.environment_var_s import environment_var_s
from tables.job_main import job_main
from tables.job_main_s import job_main_s
import environments_utils
import path
import simple_queries_job
from web.selectenvwidget import SelectEnvGroup
from web.autorefresh import AutoRefresh


class EnvWidget(VerticalLayout):

    """Environment group selection widget."""

    def __init__(self, jobset_ids, application_window, host_widget,
                 sql_session, workload=None, job_id=None):
        """Build the widget.

        @param jobset_ids:
                    IDs (list) of the parent jobsets.
        @param application_window:
                    application window which is used to display the "select
                    environment group" popup.
        @param host_widget:
                    the L{web.jobwidgets.Host} object (used to retrieve the
                    host id associated with the job/jobset
        @param sql_session:
                    SQLAlchemy session.
        @param workload:
                    workload to consider.
        @param job_id:
                    ID of the edited job/jobset.
        """
        super(EnvWidget, self).__init__()
        self._jobset_ids = jobset_ids
        self._application_window = application_window
        self._host_widget = host_widget
        self._sql_session = sql_session
        self._workload = workload
        self._job_id = job_id

        # Layout
        self.setSizeFull()
        self.setSpacing(True)
        self.setMargin(False)

        self._cb = CheckBox(_("Enable environment variables"))
        self._cb.setDescription(_("Environment variables can be defined \
                                 for this job/jobset"))
        self._cb.setImmediate(True)
        self.addComponent(self._cb)

        h = HorizontalLayout()
        h.setSizeFull()
        h.setSpacing(False)
        h.setMargin(False)
        self.addComponent(h)
        self.setExpandRatio(h, 1.0)

        self._list = ListSelect(_("Environment groups:"))
        self._list.setSizeFull()
        self._list.setNullSelectionAllowed(False)
        self._list.addListener(EnvGroupSelected(self), IValueChangeListener)
        self._list.setImmediate(True)
        h.addComponent(self._list)
        h.setExpandRatio(self._list, 1.0)

        var_bt = VerticalLayout()
        var_bt.setSpacing(True)
        var_bt.setMargin(False)
        var_bt.setWidth('20px')
        h.addComponent(var_bt)
        h.setComponentAlignment(var_bt, Alignment.MIDDLE_CENTER)

        b = Button()
        b.setStyleName(BaseTheme.BUTTON_LINK)
        b.setIcon(ThemeResource('icons/add14.png'))
        b.setDescription(_("Add an environment group"))
        b.addListener(EnvAdd(self), IClickListener)
        var_bt.addComponent(b)
        self._btadd = b

        b = Button()
        b.setStyleName(BaseTheme.BUTTON_LINK)
        b.setIcon(ThemeResource('icons/delete14.png'))
        b.setDescription(_("Remove the selected environment group"))
        b.addListener(EnvDel(self), IClickListener)
        var_bt.addComponent(b)
        self._btdel = b

        b = Button()
        b.setStyleName(BaseTheme.BUTTON_LINK)
        b.setIcon(ThemeResource('icons/go-top14.png'))
        b.setDescription(_("Move to the top"))
        b.addListener(EnvTop(self), IClickListener)
        var_bt.addComponent(b)
        self._bttop = b

        b = Button()
        b.setStyleName(BaseTheme.BUTTON_LINK)
        b.setIcon(ThemeResource('icons/go-up14.png'))
        b.setDescription(_("Raise the environment group"))
        b.addListener(EnvUp(self), IClickListener)
        var_bt.addComponent(b)
        self._btup = b

        b = Button()
        b.setStyleName(BaseTheme.BUTTON_LINK)
        b.setIcon(ThemeResource('icons/go-down14.png'))
        b.setDescription(_("Lower the environment group"))
        b.addListener(EnvDown(self), IClickListener)
        var_bt.addComponent(b)
        self._btdown = b

        b = Button()
        b.setStyleName(BaseTheme.BUTTON_LINK)
        b.setIcon(ThemeResource('icons/go-bottom14.png'))
        b.setDescription(_("Move to the bottom"))
        b.addListener(EnvBottom(self), IClickListener)
        var_bt.addComponent(b)
        self._btbottom = b

        b = Button(_("Show variables"))
        b.setDescription(_("Display all the environment variables set for \
                          this job/jobset"))
        b.addListener(EnvShowVariables(self), IClickListener)
        self.addComponent(b)
        self._cb.addListener(EnvEnable(h, b), IClickListener)

        # Content
        c = IndexedContainer()
        c.addContainerProperty('name', str, None)
        if job_id is None:
            self._cb.setValue(True)
        else:
            session = sql_session.open_session()
            h = simple_queries_job.sql_get_job(session, job_id, workload)
            self._cb.setValue(bool(h.has_env))
            self._cb.fireClick()
            if workload is None:
                query = session.query(job_environment)
                query = query.filter_by(job_id=job_id)
                query = query.order_by(job_environment.position)
            else:
                query = session.query(job_environment_s)
                query = query.filter_by(job_id=job_id)
                query = query.filter(job_environment_s.workload_date ==
                                     workload)
                query = query.order_by(job_environment_s.position)
            for je in query.all():
                e = environments_utils.get_env(session, je.env_id, workload)
                item = c.addItem(je.env_id)
                item.getItemProperty('name').setValue(e.name.encode('utf-8'))
            sql_session.close_session(session)
        self._list.setContainerDataSource(c)
        self._list.setItemCaptionPropertyId('name')
        self.set_button_state()

    def select_env_group(self, item_id):
        """Mark as selected in the list the specified environment group.

        @param item_id:
                    the environment group ID to select.
        """
        self._list.select(item_id)

    def get_selected(self):
        """Return the currently selected environment group ID.

        @return:  the database/container ID of the selected environment group.
        """
        return self._list.getValue()

    def get_datasource(self):
        """Return the container data source.

        @return:        the container data source associated with the
                        environement group list.
        """
        return self._list.getContainerDataSource()

    def remove_env_group(self, item_id=None):
        """Remove an environment group from the list.

        @param item_id:
                    the environment group ID to remove. If None (the
                    default), the currently selected item is removed.
        """
        if item_id is not None:
            self._list.removeItem(item_id)
        else:
            self._list.removeItem(self.get_selected())

    def set_button_state(self):
        """Enable or disable (greyed out) action buttons whether an
        environment group is selected or not.
        """
        selected_item_id = self.get_selected()
        if selected_item_id is None:
            self._btdel.setEnabled(False)
            self._bttop.setEnabled(False)
            self._btup.setEnabled(False)
            self._btdown.setEnabled(False)
            self._btbottom.setEnabled(False)
        else:
            self._btdel.setEnabled(True)
            c = self.get_datasource()
            if c.isFirstId(selected_item_id):
                self._bttop.setEnabled(False)
                self._btup.setEnabled(False)
            else:
                self._bttop.setEnabled(True)
                self._btup.setEnabled(True)
            if c.isLastId(selected_item_id):
                self._btdown.setEnabled(False)
                self._btbottom.setEnabled(False)
            else:
                self._btdown.setEnabled(True)
                self._btbottom.setEnabled(True)

    def get_env_ids(self):
        """Return the ordered list of the environment IDs in the container.

        @return:    the ID list.
        """
        c = self.get_datasource()
        lst = list()
        for i in range(c.size()):
            lst.append(c.getIdByIndex(i))
        return lst

    def updateJob(self, job):
        """Update the provided job object with the environment group details

        @param job:
                    the job object to update.
        @return:    None on success or an error message.
        """
        job.has_env = int(self._cb.getValue())
        ids = self.get_env_ids()
        lst = list()
        for i in range(len(ids)):
            env_id = ids[i]
            if self._workload is None:
                lst.append(job_environment(env_id, i + 1))
            else:
                lst.append(job_environment_s(env_id, i + 1,
                                             self._workload))
        job.job_environment = lst
        return None


class EnvEnable(IClickListener):

    """Callback for the Enable environment checkbox"""

    def __init__(self, widget1, widget2):
        """Initialize the callback.

        @param widget1:
                    the widget to disable (greyed out) if the checkbox
                    is not checked.
        @param widget2:
                    the second widget to also disable.
        """
        super(EnvEnable, self).__init__()
        self._w1 = widget1
        self._w2 = widget2

    def buttonClick(self, event):
        AutoRefresh.reset()
        if event.getButton().booleanValue():
            self._w1.setEnabled(True)
            self._w2.setEnabled(True)
        else:
            self._w1.setEnabled(False)
            self._w2.setEnabled(False)


class EnvGroupSelected(IValueChangeListener):

    """Callback for when an environment group is selected in the list."""

    def __init__(self, env_widget_obj):
        """Initialize the callback.

        @param env_widget_obj:
                    the associated L{EnvWidget} object.
        """
        super(EnvGroupSelected, self).__init__()
        self._c = env_widget_obj

    def valueChange(self, event):
        AutoRefresh.reset()
        # Enable/disable action buttons
        self._c.set_button_state()


class EnvAdd(IClickListener):

    """Callback for the Add new environment button."""

    def __init__(self, env_widget_obj):
        """Initialize the callback.

        @param env_widget_obj:
                    the associated L{EnvWidget} object.
        """
        super(EnvAdd, self).__init__()
        self._c = env_widget_obj

    def buttonClick(self, event):
        AutoRefresh.reset()
        AddEnvGroup(self._c)


class EnvDel(IClickListener):

    """Callback for the Delete button."""

    def __init__(self, env_widget_obj):
        """Initialize the callback.

        @param env_widget_obj:
                    the associated L{EnvWidget} object.
        """
        super(EnvDel, self).__init__()
        self._c = env_widget_obj

    def buttonClick(self, event):
        AutoRefresh.reset()
        env_group_id = self._c.get_selected()
        if env_group_id is not None:
            c = self._c.get_datasource()
            to_select = c.prevItemId(env_group_id)
            if to_select is None:
                to_select = c.nextItemId(env_group_id)
            self._c.remove_env_group()
            if to_select is not None:
                self._c.select_env_group(to_select)


class EnvTop(IClickListener):

    """Callback for the Move to top button."""

    def __init__(self, env_widget_obj):
        """Initialize the callback.

        @param env_widget_obj:
                    the associated L{EnvWidget} object.
        """
        super(EnvTop, self).__init__()
        self._c = env_widget_obj

    def buttonClick(self, event):
        AutoRefresh.reset()
        env_group_id = self._c.get_selected()
        if env_group_id is not None:
            c = self._c.get_datasource()
            if not c.isFirstId(env_group_id):
                item = c.getItem(env_group_id)
                name = item.getItemProperty('name').getValue()
                c.removeItem(env_group_id)
                new_item = c.addItemAt(0, env_group_id)
                new_item.getItemProperty('name').setValue(name)
                self._c.select_env_group(env_group_id)
                self._c.set_button_state()


class EnvUp(IClickListener):

    """Callback for the Move up button."""

    def __init__(self, env_widget_obj):
        """Initialize the callback.

        @param env_widget_obj:
                    the associated L{EnvWidget} object.
        """
        super(EnvUp, self).__init__()
        self._c = env_widget_obj

    def buttonClick(self, event):
        AutoRefresh.reset()
        env_group_id = self._c.get_selected()
        if env_group_id is not None:
            c = self._c.get_datasource()
            prev_item_id = c.prevItemId(env_group_id)
            if prev_item_id is not None:
                item = c.getItem(env_group_id)
                name = item.getItemProperty('name').getValue()
                c.removeItem(env_group_id)
                idx = c.indexOfId(prev_item_id)
                new_item = c.addItemAt(idx, env_group_id)
                new_item.getItemProperty('name').setValue(name)
                self._c.select_env_group(env_group_id)
                self._c.set_button_state()


class EnvDown(IClickListener):

    """Callback for the Move down button."""

    def __init__(self, env_widget_obj):
        """Initialize the callback.

        @param env_widget_obj:
                    the associated L{EnvWidget} object.
        """
        super(EnvDown, self).__init__()
        self._c = env_widget_obj

    def buttonClick(self, event):
        AutoRefresh.reset()
        env_group_id = self._c.get_selected()
        if env_group_id is not None:
            c = self._c.get_datasource()
            next_item_id = c.nextItemId(env_group_id)
            if next_item_id is not None:
                item = c.getItem(env_group_id)
                name = item.getItemProperty('name').getValue()
                c.removeItem(env_group_id)
                idx = c.indexOfId(next_item_id)
                new_item = c.addItemAt(idx + 1, env_group_id)
                new_item.getItemProperty('name').setValue(name)
                self._c.select_env_group(env_group_id)
                self._c.set_button_state()


class EnvBottom(IClickListener):

    """Callback for the Move to bottom button."""

    def __init__(self, env_widget_obj):
        """Initialize the callback.

        @param env_widget_obj:
                    the associated L{EnvWidget} object.
        """
        super(EnvBottom, self).__init__()
        self._c = env_widget_obj

    def buttonClick(self, event):
        AutoRefresh.reset()
        env_group_id = self._c.get_selected()
        if env_group_id is not None:
            c = self._c.get_datasource()
            if not c.isLastId(env_group_id):
                item = c.getItem(env_group_id)
                name = item.getItemProperty('name').getValue()
                c.removeItem(env_group_id)
                new_item = c.addItem(env_group_id)
                new_item.getItemProperty('name').setValue(name)
                self._c.select_env_group(env_group_id)
                self._c.set_button_state()


class AddEnvGroup(IClickListener):

    """Add an environment group window."""

    _bt_captions = [_('Cancel'), _('OK')]

    def __init__(self, env_widget_obj):
        """Create and display the Add window.

        @param env_widget_obj:
                    the associated L{EnvWidget} object.
        """
        super(AddEnvGroup, self).__init__()

        self._c = env_widget_obj

        self._w = Window('Add Environment group')
        self._w.setWidth("300px")

        # VerticalLayout as content by default
        v = self._w.getContent()
        v.setSizeFull()
        v.setMargin(True)
        v.setSpacing(True)

        t = SelectEnvGroup(env_widget_obj._sql_session,
                           _('Environment group:'),
                           _('Select an environment group'),
                           workload=env_widget_obj._workload)
        t.setWidth('100%')
        self._name = t
        v.addComponent(t)
        v.setExpandRatio(t, 1.0)

        # Button box
        h_bt = HorizontalLayout()
        h_bt.setMargin(False)
        h_bt.setSpacing(True)

        for caption in self._bt_captions:
            b = Button(_(caption))
            b.addListener(self, IClickListener)
            h_bt.addComponent(b)

        v.addComponent(h_bt)
        v.setComponentAlignment(h_bt, Alignment.BOTTOM_RIGHT)

        self._c._application_window.addWindow(self._w)
        self._w.center()

    def buttonClick(self, event):
        AutoRefresh.reset()
        # First button is Cancel
        if event.getButton().getCaption() != _(self._bt_captions[0]):
            # Retrieve the selected item
            item_id, name = self._name.getSelected()
            c = self._c.get_datasource()
            if c.containsId(item_id):
                self._c._application_window.showNotification(
                            _("Environment group already in job/jobset"),
                            _("<br/>The selected environment group is already \
                             associated with the job/jobset."),
                            Notification.TYPE_ERROR_MESSAGE)
                return
            item = c.addItem(item_id)
            # try clause to get around a strange exception in
            # indexed_container.py, in firePropertyValueChange:
            # self._singlePropertyValueChangeListeners.get(source._propertyId)
            # AttributeError: 'NoneType' object has no attribute 'get'
            try:
                item.getItemProperty('name').setValue(name)
            except:
                pass
            self._c.select_env_group(item_id)
        # Close the window
        self._c._application_window.removeWindow(self._w)


class EnvShowVariables(IClickListener):

    """Callback for the Show variables button."""

    def __init__(self, env_widget_obj):
        """Initialize the callback.

        @param env_widget_obj:
                    the associated L{EnvWidget} object.
        """
        super(EnvShowVariables, self).__init__()
        self._c = env_widget_obj

    def buttonClick(self, event):
        AutoRefresh.reset()
        ShowVariablesPopup(self._c)


class ShowVariablesPopup(IClickListener):

    """Display the environment variables associated with a job/jobset."""

    def __init__(self, env_widget_obj, x=None, y=None):
        """Create the popup.

        @param env_widget_obj:
                    the associated L{EnvWidget} object.
        @param x:
                the X coordinate of the mouse which is used to position
                the popup.  If not set, the popup is centered.
        @param y:
                the Y coordinate of the mouse which is used to position
                the popup.  If not set, the popup is centered.
        """
        super(ShowVariablesPopup, self).__init__()

        self._c = env_widget_obj

        self._w = Window(_("Environment Variables"))
        self._w.setWidth('470px')
        self._w.setHeight('470px')

        # VerticalLayout as content by default
        v = self._w.getContent()
        v.setSizeFull()
        v.setMargin(True)
        v.setSpacing(True)

        p = Panel(_('Environment variables:'))
        p.setSizeFull()
        p.setStyleName('envwidgetenvpane')
        v.addComponent(p)
        v.setExpandRatio(p, 1.0)

        r = self.build_xhtml()

        l = Label(r,
                  contentMode=Label.CONTENT_XHTML)
        p.addComponent(l)

        # Close button
        ok = Button(_('Close'))
        ok.addListener(self, IClickListener)
        v.addComponent(ok)
        v.setComponentAlignment(ok, Alignment.BOTTOM_RIGHT)

        self._c._application_window.addWindow(self._w)
        if x and y:
            self._w.setPositionX(x)
            self._w.setPositionY(y)
        else:
            self._w.center()

    def buttonClick(self, event):
        AutoRefresh.reset()
        self._c._application_window.removeWindow(self._w)

    def build_xhtml(self):
        """Return a XHTML string listing all the environment variables
        associated with the currently edit job/jobset.

        @return:    the XHTML string.
        """
        session = self._c._sql_session.open_session()

        # From the jobset hierarchy, get the jobsets for which environment
        # groups are disabled (has_env == 0)
        jobset_ids = list(self._c._jobset_ids)
        if self._c._workload is None:
            query = session.query(job_main)
            query = query.filter(job_main.id.in_(jobset_ids))
            query = query.filter(job_main.has_env == 0)
        else:
            query = session.query(job_main_s)
            query = query.filter(job_main_s.id.in_(jobset_ids))
            query = query.filter(job_main_s.has_env == 0)
            query = query.filter(job_main_s.workload_date == self._c._workload)
        for h in query.all():
            if h.id in jobset_ids:
                # Remove from the jobset hierarchy all the jobsets (and their
                # parents that shouldn't load environment groups
                jobset_ids = jobset_ids[jobset_ids.index(h.id) + 1:]

        # Retrieve the environment groups associated with the parent jobsets
        envgroup_by_job = dict()
        for job_id in jobset_ids:
            envgroup_by_job[job_id] = list()
        if jobset_ids:
            if self._c._workload is None:
                query = session.query(job_environment)
                query = query.filter(job_environment.job_id.in_(jobset_ids))
                query = query.order_by(job_environment.position)
            else:
                query = session.query(job_environment_s)
                query = query.filter(job_environment_s.job_id.in_(jobset_ids))
                query = query.filter(job_environment_s.workload_date ==
                                     self._c._workload)
                query = query.order_by(job_environment_s.position)
            for je in query.all():
                envgroup_by_job[je.job_id].append({'env_id': je.env_id,
                                                   'position': je.position})
        # For the currently edited job/jobset, get its environment groups
        # from the form as they may not be available from the database
        if self._c._job_id is None:
            current_job_id = 'New'
        else:
            current_job_id = self._c._job_id
        envgroup_by_job[current_job_id] = list()
        ids = self._c.get_env_ids()
        for i in range(len(ids)):
            envgroup_by_job[current_job_id].append({'env_id': ids[i],
                                                    'position': i + 1})

        # Retrieve the environment variables for each jobset in the
        # hierarchy.  Starts from the leaf job/jobset (the one currently
        # edited) and go up to the root jobset.  In the way, the duplicated
        # variable names are detected so they can be displayed with a
        # line-through style
        jobset_ids.append(current_job_id)
        already_defined = list()
        env_vars = dict()
        for i in range(len(jobset_ids) - 1, -1, -1):
            job_id = jobset_ids[i]
            lst = list()
            for j in range(len(envgroup_by_job[job_id]) - 1, -1, -1):
                envgroup = envgroup_by_job[job_id][j]
                e = environments_utils.get_env(session, envgroup['env_id'],
                                               self._c._workload)
                if self._c._workload is None:
                    query = session.query(environment_var)
                    query = query.filter(environment_var.env_id ==
                                        envgroup['env_id'])
                    query = query.order_by(desc(environment_var.position))
                else:
                    query = session.query(environment_var_s)
                    query = query.filter(environment_var_s.env_id ==
                                         envgroup['env_id'])
                    query = query.filter(environment_var_s.workload_date ==
                                         self._c._workload)
                    query = query.order_by(desc(environment_var_s.position))
                for ev in query.all():
                    lst.append({
                        'name': e.name,
                        'env_key': ev.env_key,
                        'env_value': ev.env_value,
                        'alreadydefined': (True
                                          if ev.env_key in already_defined
                                          else False)})
                    already_defined.append(ev.env_key)
            env_vars[job_id] = lst

        # Get the host associated with this job/jobset to retrieve the
        # environment variables defined at its level
        host_id = self._c._host_widget.get_selected()
        if self._c._workload is None:
            query = session.query(host_environment)
            query = query.filter(host_environment.host_id == host_id)
            query = query.order_by(desc(host_environment.position))
        else:
            query = session.query(host_environment_s)
            query = query.filter(host_environment_s.host_id == host_id)
            query = query.filter(host_environment_s.workload_date ==
                                 self._c._workload)
            query = query.order_by(desc(host_environment_s.position))
        lst = list()
        for he in query.all():
            e = environments_utils.get_env(session, he.env_id,
                                           self._c._workload)
            if self._c._workload is None:
                query = session.query(environment_var)
                query = query.filter(environment_var.env_id == he.env_id)
                query = query.order_by(desc(environment_var.position))
            else:
                query = session.query(environment_var_s)
                query = query.filter(environment_var_s.env_id == he.env_id)
                query = query.filter(environment_var_s.workload_date ==
                                     self._c._workload)
                query = query.order_by(desc(environment_var_s.position))
            for ev in query.all():
                lst.append({
                    'name': e.name,
                    'env_key': ev.env_key,
                    'env_value': ev.env_value,
                    'alreadydefined': (True
                                      if ev.env_key in already_defined
                                      else False)})
                already_defined.append(ev.env_key)

        # Build the result XHTML string
        result_str = ''
        if lst:
            previous_name = None
            result_str += ('<p class="envwidgetjobname">' +
                           _('From host ') +
                           self._c._host_widget.get_selected_name() + '</p>')
            for j in range(len(lst) - 1, -1, -1):
                if previous_name != lst[j]['name']:
                    if previous_name is not None:
                        result_str += '</ol>'
                    previous_name = lst[j]['name']
                    result_str += ('<p class="envwidgetenvname">' +
                                   previous_name.encode('utf-8') +
                                   '</p>' +
                                   '<ol class="envwidgetenvvars">')
                if lst[j]['alreadydefined']:
                    result_str += '<li class="envwidgetenvdefined">'
                else:
                    result_str += '<li>'
                result_str += (lst[j]['env_key'].encode('utf-8') +
                               '=' +
                               lst[j]['env_value'].encode('utf-8') +
                               '</li>')
            else:
                result_str += '</ol>'

        full_jobset_hierarchy = list(self._c._jobset_ids)
        full_jobset_hierarchy.append(current_job_id)
        for i in range(len(jobset_ids)):
            job_id = jobset_ids[i]
            previous_name = None
            if job_id in env_vars and env_vars[job_id]:
                if job_id == 'New':
                    job_name = _('this new job/jobset')
                else:
                    idx = full_jobset_hierarchy.index(job_id)
                    p = path.Path(session, id=full_jobset_hierarchy[:idx + 1],
                                  workload=self._c._workload)
                    job_name = p.path.encode('utf-8')
                result_str += ('<p class="envwidgetjobname">' +
                               _('From ') + job_name + '</p>')
                for j in range(len(env_vars[job_id]) - 1, -1, -1):
                    if previous_name != env_vars[job_id][j]['name']:
                        if previous_name is not None:
                            result_str += '</ol>'
                        previous_name = env_vars[job_id][j]['name']
                        result_str += ('<p class="envwidgetenvname">' +
                                       previous_name.encode('utf-8') +
                                       '</p>' +
                                       '<ol class="envwidgetenvvars">')
                    if env_vars[job_id][j]['alreadydefined']:
                        result_str += '<li class="envwidgetenvdefined">'
                    else:
                        result_str += '<li>'
                    result_str += (env_vars[job_id][j]['env_key'].encode(
                                                                'utf-8') +
                                   '=' +
                                   env_vars[job_id][j]['env_value'].encode(
                                                                'utf-8') +
                                   '</li>')
                else:
                    result_str += '</ol>'
        self._c._sql_session.close_session(session)
        return result_str
