# Schedwi
# Copyright (C) 2013 Herve Quatremain
#
# This file is part of Schedwi.
#
# Schedwi is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
#
# Schedwi is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

"""Job/jobset popup menu."""

from muntjac.event.action import Action, IHandler
from muntjac.terminal.theme_resource import ThemeResource
from muntjac.ui.window import Notification
from muntjac.event.shortcut_listener import ShortcutListener
from muntjac.event.shortcut_action import KeyCode

from web.boxes import DeleteBox, QuestionBox, BoxCloseListener
from web.jobwindow import JobWindow
from web.autorefresh import AutoRefresh
from path import id2path
from simple_queries_job import sql_get_job, sql_get_children
import status_utils
from cmd_job.cp import cp_recursive, relink
from cmd_job.rm import rm_recursive
from cmd_wl.start import start_id
from cmd_wl.stop import stop_id


_SELECTED_ID = None
_IS_CUT = False


class DeleteConfirmed(BoxCloseListener):

    """Confirmation popup callback."""

    def __init__(self, obj, item_id):
        """Initialize the object.

        @param obj:
                    L{CutCopyPasteMenuListener} object to retrieve the
                    required details.
        @param item_id:
                    container item ID to remove.
        """
        self._c = obj
        self._id = item_id

    def boxClose(self, event):
        """Deletion confirmed."""
        global _SELECTED_ID
        AutoRefresh.reset()
        if _SELECTED_ID == self._id:
            _SELECTED_ID = None
        session = self._c._sql_session.open_session()
        p = id2path(session, self._id, self._c._workload)
        rm_recursive(session, p, True, True)
        self._c._sql_session.close_session(session)
        self._c.repaint()


def _delete_job(obj, item_id):
    """Delete the provided item.

    @param obj:
                    L{CutCopyPasteMenuListener} object to retrieve the
                    required details.
    @param item_id:
                    container item ID to remove.
    """
    d = DeleteBox(obj, _("Delete"),
                  _('Are you sure you want to permanently delete "%s"?') % \
                                obj.get_name(item_id),
  _('If you delete a jobset, all its content will also be permanently lost.') \
                          if obj.is_jobset(item_id) else \
                  _('If you delete a job, it will be permanently lost.'))
    d.addListener(DeleteConfirmed(obj, item_id))


class StartStopConfirmed(BoxCloseListener):

    """Confirmation popup callback."""

    def __init__(self, obj, item_id, do_start):
        """Initialize the object.

        @param obj:
                    L{CutCopyPasteMenuListener} object to retrieve the
                    required details.
        @param item_id:
                    Database ID of the job or jobset to start or stop.
        @param do_start:
                    If True, the operation is to start a job or jobset.
                    If False, the specified job or jobset must be stopped.
        """
        self._c = obj
        self._id = item_id
        self._do_start = do_start

    def boxClose(self, event):
        """Operation confirmed."""
        session = self._c._sql_session.open_session()
        if self._do_start:
            start_id(session, self._id, self._c._workload)
        else:
            stop_id(session, self._id, self._c._workload)
        self._c._sql_session.close_session(session)
        n = Notification(
            _("This may take up to a minute<br/>to be taken into account"),
            Notification.TYPE_WARNING_MESSAGE)
        n.setDelayMsec(3000)
        self._c.getWindow().showNotification(n)


def _start_stop_job(obj, item_id, do_start):
    """Start or stop the provided job or jobset.

    @param obj:
                    L{CutCopyPasteMenuListener} object to retrieve the
                    required details.
    @param item_id:
                    Database ID of the job or jobset to start or stop.
    @param do_start:
                    If True, the operation is to start a job or jobset.
                    If False, the specified job or jobset must be stopped.
    """
    s = status_utils.get_status(obj._sql_session, item_id, obj._workload)
    if do_start:
        if s.status == status_utils.RUNNING:
            n = Notification(
                        _('"%s" is already running') % obj.get_name(item_id),
                        Notification.TYPE_WARNING_MESSAGE)
            n.setDelayMsec(3000)
            obj.getWindow().showNotification(n)
        else:
            d = QuestionBox(obj, _("Start"),
                            _('Are you sure you want to start "%s" now?') % \
                                obj.get_name(item_id))
    else:
        if s.status != status_utils.RUNNING:
            n = Notification(
                        _('"%s" is already stopped') % obj.get_name(item_id),
                        Notification.TYPE_WARNING_MESSAGE)
            n.setDelayMsec(3000)
            obj.getWindow().showNotification(n)
        else:
            d = QuestionBox(obj, _("Stop"),
                            _('Are you sure you want to stop "%s" now?') % \
                                obj.get_name(item_id),
                _("The job will be killed.") if not obj.is_jobset(item_id) \
                else _("All the children will be killed recursively."))
    d.addListener(StartStopConfirmed(obj, item_id, do_start))


class GenericContextMenu(IHandler):

    # Possible actions
    _ACTION_EDIT = Action(_('Edit...'), ThemeResource('icons/edit.png'))
    _ACTION_CUT = Action(_('Cut'), ThemeResource('icons/cut.png'))
    _ACTION_COPY = Action(_('Copy'), ThemeResource('icons/copy.png'))
    _ACTION_DELETE = Action(_('Delete'), ThemeResource('icons/delete.png'))
    _ACTION_ADD = Action(_('Add...'), ThemeResource('icons/add.png'))
    _ACTION_START = Action(_('Force Start'), ThemeResource('icons/start.png'))
    _ACTION_STOP = Action(_('Stop'), ThemeResource('icons/stop.png'))

    def __init__(self):
        super(GenericContextMenu, self).__init__()


class CutCopyPasteMenu(GenericContextMenu):

    # Actions for the context menu
    _ACTION_PASTE = Action(_('Paste Into Jobset'),
                           ThemeResource('icons/paste.png'))
    _ACTION_NEWJOB = Action(_('Create New Job...'),
                            ThemeResource('icons/job-new.png'))
    _ACTION_NEWJOBSET = Action(_('Create New Jobset...'),
                               ThemeResource('icons/jobset-new.png'))

    def __init__(self, obj, sql_session, workload=None):
        """Initialize the object.

        @param obj:
                    L{CutCopyPasteMenuListener} object to retrieve the
                    details required by the menu.
        @param sql_session:
                    SQLAlchemy session.
        @param workload:
                    workload to consider.
        """
        super(CutCopyPasteMenu, self).__init__()
        self._sql_session = sql_session
        self._workload = workload
        self._c = obj

    def handleAction(self, a, sender, target):
        global _SELECTED_ID, _IS_CUT

        AutoRefresh.reset()
        if not target:
            target = self._c.get_parent_id()

        # Edit
        if a.getCaption() == self._ACTION_EDIT.getCaption():
            self.do_edit(target)

        # Cut
        elif a.getCaption() == self._ACTION_CUT.getCaption():
            self.do_cut(target)

        # Copy
        elif a.getCaption() == self._ACTION_COPY.getCaption():
            self.do_copy(target)

        # Paste
        elif a.getCaption() == self._ACTION_PASTE.getCaption():
            self.do_paste(target)

        # Delete
        elif a.getCaption() == self._ACTION_DELETE.getCaption():
            self.do_delete(target)

        # New Job
        elif a.getCaption() == self._ACTION_NEWJOB.getCaption():
            self.do_new_job(target)

        # New JobSet
        elif a.getCaption() == self._ACTION_NEWJOBSET.getCaption():
            self.do_new_jobset(target)

        # Force start
        elif a.getCaption() == self._ACTION_START.getCaption():
            self.do_start(target)

        # stop
        elif a.getCaption() == self._ACTION_STOP.getCaption():
            self.do_stop(target)

    def _get_name(self, target):
        """Get a new name for the job/jobset to copy or move.

        @param target:
                    the database ID of the parent jobset.
        @return:    the new name or None in case of error.
        """
        global _SELECTED_ID, _IS_CUT

        if not _SELECTED_ID:
            return None
        p = id2path(self._sql_session, target, self._workload)
        if _SELECTED_ID in p.id:
            self._c.getWindow().showNotification(
                _("Cannot move a jobset into itself") if _IS_CUT else \
                _("Cannot copy a jobset into itself"),
                _("<br/>The destination jobset is inside the source jobset."),
                Notification.TYPE_ERROR_MESSAGE)
            return None

        # Try to find a unique name for the pasted job
        # First, get the selected job/jobset name
        session = self._sql_session.open_session()
        try:
            job = sql_get_job(session, _SELECTED_ID, self._workload)
        except:
            self._sql_session.close_session(session)
            self._c.getWindow().showNotification(
                                _("Cannot get the source job details"),
                                _("<br/>Maybe someone else just removed it."),
                                Notification.TYPE_ERROR_MESSAGE)
            return None
        name = job.name

        # Then, get the list of the job/jobset names in the destination jobset
        dest_names = list()
        for job in sql_get_children(session, target, False, self._workload):
            dest_names.append(job.name)
        self._sql_session.close_session(session)

        i = 1
        n = name
        while n in dest_names:
            n = name + '.%d' % i
            i += 1
        return n

    def _move(self, name, target):
        """Move/rename a job/jobset.

        @param name:
                    new name.
        @param target:
                    the database ID of the parent jobset.
        """
        global _SELECTED_ID

        session = self._sql_session.open_session()
        # Source job/jobset
        try:
            src_job = sql_get_job(session, _SELECTED_ID, self._workload)
        except:
            self._sql_session.close_session(session)
            self._c.getWindow().showNotification(
                                _("Cannot get the source job details"),
                                _("<br/>Maybe someone else just removed it."),
                                Notification.TYPE_ERROR_MESSAGE)
        else:
            src_job.parent = target
            src_job.name = name
            self._sql_session.close_session(session)

    def _copy(self, name, target):
        """Copy a job/jobset.

        @param name:
                    new name.
        @param target:
                    the database ID of the parent jobset.
        """
        global _SELECTED_ID

        session = self._sql_session.open_session()
        # Source job/jobset
        try:
            src_job = sql_get_job(session, _SELECTED_ID, self._workload)
        except:
            self._sql_session.close_session(session)
            self._c.getWindow().showNotification(
                                _("Cannot get the source job details"),
                                _("<br/>Maybe someone else just removed it."),
                                Notification.TYPE_ERROR_MESSAGE)
            return
        # Destination jobset
        try:
            dest_jobset = sql_get_job(session, target, self._workload)
        except:
            self._sql_session.close_session(session)
            self._c.getWindow().showNotification(
                                _("Cannot get the destination jobset details"),
                                _("<br/>Maybe someone else just removed it."),
                                Notification.TYPE_ERROR_MESSAGE)
            return
        self._sql_session.close_session(session)

        # Do the copy and then recreate links
        job_assoc = {}
        job_links = {}
        cp_recursive(self._sql_session, src_job, dest_jobset,
                     job_assoc, job_links, name)
        relink(self._sql_session, job_assoc, job_links)

    def is_cut_or_copy(self):
        """Tell if an item has been marked for copy/cut.

        @return:
            True is an item has been marked for copy or cut. False otherwise.
        """
        global _SELECTED_ID
        if _SELECTED_ID is None:
            return False
        # Check that the item is still in the database
        try:
            sql_get_job(self._sql_session, _SELECTED_ID, self._workload)
        except:
            _SELECTED_ID = None
            return False
        return True

    def do_edit(self, target, x=None, y=None):
        """Edit the job or jobset.

        @param target:
                database ID of the job or jobset to edit.
        @param x:
                the X coordinate of the mouse which is used to position
                the popup.  If None, the popup is centered.
        @param y:
                the Y coordinate of the mouse which is used to position
                the popup.  If None, the popup is centered.
        """
        JobWindow(self._c, self._c.get_jobset_hierarchy(target),
                  self._c.getWindow(), self._sql_session, self._workload,
                  target, x=x, y=y)

    def do_start(self, target):
        """Force start the job or jobset.

        @param target:
                database ID of the job or jobset to start.
        """
        _start_stop_job(self._c, target, True)

    def do_stop(self, target):
        """Stop the job or jobset.

        @param target:
                database ID of the job or jobset to stop.
        """
        _start_stop_job(self._c, target, False)

    def do_cut(self, target):
        """Cut the job or jobset.

        @param target:
                database ID of the job or jobset to cut.
        """
        global _SELECTED_ID, _IS_CUT
        _SELECTED_ID = target
        _IS_CUT = True
        self._c.repaint_tools()

    def do_copy(self, target):
        """Copy the job or jobset.

        @param target:
                database ID of the job or jobset to copy.
        """
        global _SELECTED_ID, _IS_CUT
        _SELECTED_ID = target
        _IS_CUT = False
        self._c.repaint_tools()

    def do_paste(self, target):
        """Paste a job or jobset in the provided jobset.

        @param target:
                database ID of the parent jobset in which to paste
                the previously copied or cut job or jobset.
        """
        global _SELECTED_ID, _IS_CUT
        name = self._get_name(target)
        if name:
            if _IS_CUT:
                self._move(name, target)
            else:
                self._copy(name, target)
            self._c.repaint()

    def do_delete(self, target):
        """Delete a job or jobset.

        @param target:
                database ID of the job or jobset to delete.
        """
        _delete_job(self._c, target)

    def do_new_jobset(self, target):
        """Create a new jobset.

        @param target:
                database ID of the parent jobset in which to create
                the new jobset.
        """
        JobWindow(self._c, self._c.get_jobset_hierarchy(target),
                  self._c.getWindow(), self._sql_session, self._workload,
                  None, False)

    def do_new_job(self, target):
        """Create a new job.

        @param target:
                database ID of the parent jobset in which to create
                the new job.
        """
        JobWindow(self._c, self._c.get_jobset_hierarchy(target),
                  self._c.getWindow(), self._sql_session, self._workload,
                  None, True)


class CutCopyPasteMenuJobsetTree(CutCopyPasteMenu):

    _ROOT_JOBSET_ID = 1

    def __init__(self, jobsetTree):
        super(CutCopyPasteMenuJobsetTree, self).__init__(jobsetTree,
                                                       jobsetTree._sql_session,
                                                       jobsetTree._workload)

    def getActions(self, target, sender):
        if target is not None:
            if self._workload is not None:
                return [self._ACTION_EDIT,
                        self._ACTION_START, self._ACTION_STOP]
            if target == self._ROOT_JOBSET_ID:
                # Root Jobset
                return [self._ACTION_EDIT, self._ACTION_PASTE]
            else:
                return [self._ACTION_EDIT, self._ACTION_CUT,
                        self._ACTION_COPY, self._ACTION_PASTE,
                        self._ACTION_DELETE]


class CutCopyPasteMenuJobTable(CutCopyPasteMenu):
    def __init__(self, jobTable):
        super(CutCopyPasteMenuJobTable, self).__init__(jobTable,
                                                       jobTable._sql_session,
                                                       jobTable._workload)

    def getActions(self, target, sender):
        if not target:
            # Background
            if self._workload is None:
                return [self._ACTION_NEWJOBSET, self._ACTION_NEWJOB,
                        self._ACTION_PASTE]
            else:
                return None

        if self._workload is not None:
            return [self._ACTION_EDIT, self._ACTION_START, self._ACTION_STOP]
        if self._c.is_jobset(target):
            # Jobset
            return [self._ACTION_EDIT, self._ACTION_CUT,
                    self._ACTION_COPY, self._ACTION_PASTE,
                    self._ACTION_DELETE]
        else:
            # Job
            return [self._ACTION_EDIT, self._ACTION_CUT,
                    self._ACTION_COPY, self._ACTION_DELETE]


class DeleteShortcut(ShortcutListener):

    def __init__(self, obj):
        """Initialize the object.

        @param obj:     L{web.jobtable.JobTable} object.
        """
        super(DeleteShortcut, self).__init__(None, KeyCode.DELETE, None)
        self._c = obj

    def handleAction(self, sender, target):
        AutoRefresh.reset()
        if self._c.get_selected():
            _delete_job(self._c, self._c.get_selected())


class CutCopyPasteMenuListener(object):

    """Interface for providing details required for popup menus."""

    def get_parent_id(self):
        """Return the database ID of the parent jobset."""
        raise NotImplementedError

    def get_jobset_hierarchy(self, item_id):
        """Return the hierarchy of database ID of the parent jobsets."""
        raise NotImplementedError

    def get_name(self, item_id):
        """Return the name of the provided item ID."""
        raise NotImplementedError

    def repaint(self):
        """Refresh the view."""
        raise NotImplementedError

    def repaint_tools(self):
        """Repaint the cut/copy/paste button."""
        raise NotImplementedError

    def is_jobset(self, item_id):
        """Tell if the provided object is a jobset or not.

        @return:    True if the item is a jobset else False.
        """
        raise NotImplementedError
