# Schedwi
# Copyright (C) 2012 Herve Quatremain
#
# This file is part of Schedwi.
#
# Schedwi is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
#
# Schedwi is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

"""Command argument UI."""

import copy

from muntjac.api import (ListSelect, Button, Label, Window, TextField,
                         VerticalLayout, HorizontalLayout)
from muntjac.event.shortcut_action import KeyCode
from muntjac.ui.button import IClickListener
from muntjac.data.property import IValueChangeListener
from muntjac.data.util.indexed_container import IndexedContainer
from muntjac.ui.themes import BaseTheme
from muntjac.terminal.theme_resource import ThemeResource

import sql_hierarchy
from tables.job_arguments import job_arguments
from tables.job_arguments_s import job_arguments_s
from web.jobwidgets import JobWidget
from web.autorefresh import AutoRefresh


class CommandArgs(JobWidget):

    """Command argument widget."""

    def __init__(self, jobset_ids, application_window,
                 sql_session, workload=None, job_id=None):
        """Build the widget.

        @param jobset_ids:
                    IDs (list) of the parent jobsets.
        @param application_window:
                    application window which is used to display the "edit/add
                    argument" popup.
        @param sql_session:
                    SQLAlchemy session (it can be an opened session)
        @param workload:
                    workload to consider.
        @param job_id:
                    ID of the edited job/jobset.
        """
        super(CommandArgs, self).__init__(jobset_ids, sql_session,
                                          workload, job_id)
        self._application_window = application_window

        # Retrieve the current and the default value from the database
        ids = list(jobset_ids)
        if job_id is not None:
            ids.append(job_id)
        from_obj, obj = sql_hierarchy.get_list_by_id(sql_session, ids,
                       job_arguments if workload is None else job_arguments_s,
                       job_arguments.position if workload is None \
                                              else job_arguments_s.position,
                        workload)
        c = self._new_container(obj)
        if job_id is None or not from_obj:
            self._container = None
            self._default_container = c
        else:
            self._container = c
            foo, obj = sql_hierarchy.get_list_by_id(sql_session,
                       jobset_ids,
                       job_arguments if workload is None else job_arguments_s,
                       job_arguments.position if workload is None \
                                              else job_arguments_s.position,
                        workload)
            self._default_container = self._new_container(obj)

        self._savedValue = None

        self._vb = VerticalLayout()
        self._vb.setSpacing(True)
        self._vb.setMargin(False)
        self._vb.setSizeFull()
        self._vb.addComponent(Label(_('Command arguments:')))

        hb = HorizontalLayout()
        hb.setSpacing(False)
        hb.setMargin(False)
        hb.setSizeFull()

        bt_box = VerticalLayout()
        bt_box.setSpacing(True)
        bt_box.setMargin(False)
        bt_box.setWidth('20px')

        # Edit an argument
        b = Button()
        b.setStyleName(BaseTheme.BUTTON_LINK)
        b.setIcon(ThemeResource('icons/edit14.png'))
        b.setDescription(_("Edit the selected argument"))
        b.addListener(EditClicked(self), IClickListener)
        bt_box.addComponent(b)
        self._btedit = b

        # Add a new argument
        b = Button()
        b.setStyleName(BaseTheme.BUTTON_LINK)
        b.setIcon(ThemeResource('icons/add14.png'))
        b.setDescription(_("Add a new argument"))
        b.addListener(NewClicked(self), IClickListener)
        bt_box.addComponent(b)
        self._btadd = b

        # Remove an argument
        b = Button()
        b.setStyleName(BaseTheme.BUTTON_LINK)
        b.setIcon(ThemeResource('icons/delete14.png'))
        b.setDescription(_("Remove the selected argument"))
        b.addListener(DelClicked(self), IClickListener)
        bt_box.addComponent(b)
        self._btdel = b

        # Move to top
        b = Button()
        b.setStyleName(BaseTheme.BUTTON_LINK)
        b.setIcon(ThemeResource('icons/go-top14.png'))
        b.setDescription(_("Move to the top"))
        b.addListener(TopClicked(self), IClickListener)
        bt_box.addComponent(b)
        self._bttop = b

        # Move up the selected argument
        b = Button()
        b.setStyleName(BaseTheme.BUTTON_LINK)
        b.setIcon(ThemeResource('icons/go-up14.png'))
        b.setDescription(_("Raise the argument"))
        b.addListener(UpClicked(self), IClickListener)
        bt_box.addComponent(b)
        self._btup = b

        # Move down the selected argument
        b = Button()
        b.setStyleName(BaseTheme.BUTTON_LINK)
        b.setIcon(ThemeResource('icons/go-down14.png'))
        b.setDescription(_("Lower the argument"))
        b.addListener(DownClicked(self), IClickListener)
        bt_box.addComponent(b)
        self._btdown = b

        # Move to bottom
        b = Button()
        b.setStyleName(BaseTheme.BUTTON_LINK)
        b.setIcon(ThemeResource('icons/go-bottom14.png'))
        b.setDescription(_("Move to the bottom"))
        b.addListener(BottomClicked(self), IClickListener)
        bt_box.addComponent(b)
        self._btbottom = b

        hb.addComponent(bt_box)

        self._w = ListSelect()
        self._w.setSizeFull()
        self._w.setNullSelectionAllowed(False)
        self._w.setImmediate(True)
        self._w.setDescription(_("Command arguments"))
        self._w.addListener(Selected(self), IValueChangeListener)
        hb.addComponent(self._w)
        hb.setExpandRatio(self._w, 1.0)

        self._vb.addComponent(hb)
        self._vb.setExpandRatio(hb, 1.0)

        self._addInheritedWidget(self._vb)

        if self._container is not None:
            self._w.setContainerDataSource(self._container)
            self.set_state_button()
        else:
            self._savedValue = copy.deepcopy(self._default_container)
            self._setDefaultValue()
            self._inherited.setValue(True)
            self._doGreyOut()
        self._w.setItemCaptionPropertyId('argument')

    def setFocus(self):
        self._w.focus()

    def _new_container(self, lst):
        """Build a new container.

        @param lst:
                    list of job_arguments database objects.
        @return:    the new container.
        """
        c = IndexedContainer()
        c.addContainerProperty('argument', str, None)
        if lst:
            for arg in lst:
                # A position at -1 means no argument
                if arg.position == -1:
                    break
                item = c.addItem(c.generateId())
                item.getItemProperty('argument').setValue(
                                                arg.argument.encode('utf-8'))
        return c

    def getFieldWidget(self):
        return self._vb

    def _setDefaultValue(self):
        self._w.setContainerDataSource(self._default_container)

    def _saveValue(self):
        self._savedValue = self.get_current_datasource()

    def _restoreValue(self):
        self._w.setContainerDataSource(self._savedValue)
        self.set_state_button()

    def _enable_buttons(self, state):
        """Enable/disable (greyed-out) the buttons.

        @param state:
                    True to enable the buttons or False to disable them.
        """
        self._btedit.setEnabled(state)
        self._btdel.setEnabled(state)
        self._btup.setEnabled(state)
        self._btdown.setEnabled(state)
        self._bttop.setEnabled(state)
        self._btbottom.setEnabled(state)

    def set_state_button(self):
        """Set the state (enable/disable) of the buttons based on
        whether an item is selected and the position (top/button) of
        this item.
        """
        selected_item_id = self._w.getValue()
        if selected_item_id is None:
            self._enable_buttons(False)
        else:
            self._btedit.setEnabled(True)
            self._btdel.setEnabled(True)
            if selected_item_id == \
                    self.get_current_datasource().firstItemId():
                self._btup.setEnabled(False)
                self._bttop.setEnabled(False)
            else:
                self._btup.setEnabled(True)
                self._bttop.setEnabled(True)

            if selected_item_id == \
                    self.get_current_datasource().lastItemId():
                self._btdown.setEnabled(False)
                self._btbottom.setEnabled(False)
            else:
                self._btdown.setEnabled(True)
                self._btbottom.setEnabled(True)

    def get_current_datasource(self):
        """Return the ContainerDataSource in used."""
        return self._w.getContainerDataSource()

    def get_value_from_item_id(self, item_id):
        """Return the argument from the ContainerDataSource item ID."""
        c = self.get_current_datasource()
        return c.getItem(item_id).getItemProperty('argument').getValue()

    def remove_item(self, item_id):
        """Remove the provided argument from the list."""
        if item_id is not None:
            c = self.get_current_datasource()
            c.removeItem(item_id)

    def updateJob(self, job):
        """Update the provided database job object with the command
        arguments.

        @param job:
                the L{tables.job_main.job_main} or
                L{tables.job_main_s.job_main_s} database object to update.
        @return: None
        """
        if self.isInherited():
            job.job_arguments = []
        else:
            c = self.get_current_datasource()
            l = c.size()
            if l == 0:
                # If no argument is specified, a database entry with a
                # position of -1 must be inserted.  Otherwise, no entry
                # in the database means inheritance.
                if self._workload is None:
                    job.job_arguments = [job_arguments(-1, "")]
                else:
                    job.job_arguments = [job_arguments_s(-1, "",
                                                          self._workload)]
            else:
                arguments = list()
                for i in range(l):
                    item = c.getItem(c.getIdByIndex(i))
                    v = item.getItemProperty('argument').getValue()
                    if self._workload is None:
                        arguments.append(job_arguments(len(arguments) + 1, v))
                    else:
                        arguments.append(job_arguments_s(len(arguments) + 1,
                                                         v,
                                                         self._workload))
                job.job_arguments = arguments
        return None


class Selected(IValueChangeListener):

    """Callback when an item is selected by the user."""

    def __init__(self, obj):
        super(Selected, self).__init__()
        self._c = obj

    def valueChange(self, event):
        AutoRefresh.reset()
        self._c.set_state_button()


class UpClicked(IClickListener):
    def __init__(self, obj):
        super(UpClicked, self).__init__()
        self._c = obj

    def buttonClick(self, event):
        AutoRefresh.reset()
        selected_item_id = self._c._w.getValue()
        if selected_item_id is not None:
            c = self._c.get_current_datasource()
            prev_item_id = c.prevItemId(selected_item_id)
            if prev_item_id is not None:
                v = self._c.get_value_from_item_id(prev_item_id)
                c.removeItem(prev_item_id)
                item = c.addItemAfter(selected_item_id, c.generateId())
                item.getItemProperty('argument').setValue(v)
                self._c.set_state_button()


class DownClicked(IClickListener):
    def __init__(self, obj):
        super(DownClicked, self).__init__()
        self._c = obj

    def buttonClick(self, event):
        AutoRefresh.reset()
        selected_item_id = self._c._w.getValue()
        if selected_item_id is not None:
            c = self._c.get_current_datasource()
            next_item_id = c.nextItemId(selected_item_id)
            if next_item_id is not None:
                v = self._c.get_value_from_item_id(next_item_id)
                c.removeItem(next_item_id)
                idx = c.indexOfId(selected_item_id)
                item = c.addItemAt(idx, c.generateId())
                item.getItemProperty('argument').setValue(v)
                self._c.set_state_button()


class TopClicked(IClickListener):
    def __init__(self, obj):
        super(TopClicked, self).__init__()
        self._c = obj

    def buttonClick(self, event):
        AutoRefresh.reset()
        selected_item_id = self._c._w.getValue()
        if selected_item_id is not None:
            c = self._c.get_current_datasource()
            prev_item_id = c.prevItemId(selected_item_id)
            if prev_item_id is not None:
                v = self._c.get_value_from_item_id(selected_item_id)
                c.removeItem(selected_item_id)
                i = c.generateId()
                item = c.addItemAt(0, i)
                item.getItemProperty('argument').setValue(v)
                self._c._w.select(i)


class BottomClicked(IClickListener):
    def __init__(self, obj):
        super(BottomClicked, self).__init__()
        self._c = obj

    def buttonClick(self, event):
        AutoRefresh.reset()
        selected_item_id = self._c._w.getValue()
        if selected_item_id is not None:
            c = self._c.get_current_datasource()
            next_item_id = c.nextItemId(selected_item_id)
            if next_item_id is not None:
                v = self._c.get_value_from_item_id(selected_item_id)
                c.removeItem(selected_item_id)
                i = c.generateId()
                item = c.addItem(i)
                item.getItemProperty('argument').setValue(v)
                self._c._w.select(i)


class DelClicked(IClickListener):
    def __init__(self, obj):
        super(DelClicked, self).__init__()
        self._c = obj

    def buttonClick(self, event):
        AutoRefresh.reset()
        selected_item_id = self._c._w.getValue()
        # Get the item/argument before the selected one (or if there is no
        # argmument before, get the one after).  This way we can select
        # it after the removal.
        c = self._c.get_current_datasource()
        item_id = c.prevItemId(selected_item_id)
        if item_id is None:
            item_id = c.nextItemId(selected_item_id)
        self._c.remove_item(selected_item_id)
        if item_id is None:
            self._c._enable_buttons(False)
        else:
            self._c._w.setValue(item_id)


class EditClicked(IClickListener):
    def __init__(self, obj):
        super(EditClicked, self).__init__()
        self._c = obj

    def buttonClick(self, event):
        AutoRefresh.reset()
        selected_item_id = self._c._w.getValue()
        if selected_item_id is not None:
            EditPopup(self._c, event.getClientX(), event.getClientY(),
                      selected_item_id)


class NewClicked(IClickListener):
    def __init__(self, obj):
        super(NewClicked, self).__init__()
        self._c = obj

    def buttonClick(self, event):
        AutoRefresh.reset()
        EditPopup(self._c, event.getClientX(), event.getClientY())


class EditPopup(IClickListener):

    """Display the add/edit command argument popup."""

    def __init__(self, command_args_obj, x, y, item_id=None):
        """Create the popup.

        @param command_args_obj:
                the assosiated L{CommandArgs} object.
        @type command_args_obj: L{CommandArgs}
        @param x:
                the X coordinate of the mouse which is used to position
                the popup.
        @param y:
                the Y coordinate of the mouse which is used to position
                the popup.
        @param item_id:
                the ID of the edited argument (or None in the case of a
                new argument)
        """
        super(EditPopup, self).__init__()
        self._c = command_args_obj
        self._item_id = item_id

        if item_id is None:
            title = _('New argument')
        else:
            title = _('Edit argument')
            value = self._c.get_value_from_item_id(item_id)
        h = HorizontalLayout()
        h.setWidth('100%')
        h.setMargin(True)
        h.setSpacing(True)
        self._text = TextField()
        self._text.focus()
        self._text.setWidth('100%')
        if item_id is not None:
            self._text.setValue(value)
        h.addComponent(self._text)
        h.setExpandRatio(self._text, 1.0)
        ok = Button(_('OK'))
        ok.addListener(self, IClickListener)
        ok.setClickShortcut(KeyCode.ENTER)
        h.addComponent(ok)

        self._w = Window(title, h)
        self._w.setWidth('350px')
        self._c._application_window.addWindow(self._w)
        self._w.setPositionX(x)
        self._w.setPositionY(y)

    def buttonClick(self, event):
        AutoRefresh.reset()
        c = self._c.get_current_datasource()
        if self._item_id is None:
            item = c.addItem(c.generateId())
        else:
            item = c.getItem(self._item_id)
        # try clause to get around a strange exception in
        # indexed_container.py, in firePropertyValueChange:
        # self._singlePropertyValueChangeListeners.get(source._propertyId)
        # AttributeError: 'NoneType' object has no attribute 'get'
        try:
            item.getItemProperty('argument').setValue(self._text.getValue())
        except:
            pass
        self._c._application_window.removeWindow(self._w)
