# Schedwi
# Copyright (C) 2013 Herve Quatremain
#
# This file is part of Schedwi.
#
# Schedwi is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
#
# Schedwi is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

"""Calendar edit popup."""

from muntjac.api import (Window, Button, Alignment, HorizontalLayout, Label,
                         TextField, TextArea, VerticalSplitPanel,
                         VerticalLayout)
from muntjac.ui.button import IClickListener
from muntjac.ui.window import Notification
from muntjac.terminal.sizeable import ISizeable
from muntjac.data.property import IValueChangeListener
from muntjac.data.validators.abstract_validator import AbstractValidator

from tables.calendars import calendars
from tables.calendars_s import calendars_s
from web.selectcalwidget import SelectCalendar
from web.calendarview import CalendarView
from cmd_cal import calcomp
from path_cal import PathCal
from simple_queries_cal import sql_get_cal, sql_get_cal_children


class CalendarEditWindow(IClickListener, IValueChangeListener):

    """Display the add/edit calendar popup."""

    _bt_captions = [_('Cancel'), _('OK')]

    def __init__(self, obj, parent_id, cal_id=None, x=None, y=None):
        """Create the popup.

        @param obj:
                L{web.calendarwindow.CutCopyPasteMenuListener} object to
                retrieve the details required by the menu.
        @param parent_id:
                database ID of the parent directory.
        @param cal_id:
                when the window is used to edit an calendar,
                this is the database ID of this calendar.
        @param x:
                the X coordinate of the mouse which is used to position
                the popup.  By default, the popup is centered.
        @param y:
                the Y coordinate of the mouse which is used to position
                the popup.  By default, the popup is centered.
        """
        super(CalendarEditWindow, self).__init__()
        self._c = obj
        self._parent_id = parent_id
        self._cal_id = cal_id

        if cal_id is None:
            title = _('New calendar')
        else:
            title = _('Edit calendar')

        self._w = Window(title)
        self._w.setHeight('510px')
        self._w.setWidth('562px')

        v = self._w.getContent()
        v.setSizeFull()
        v.setMargin(False)
        v.setSpacing(False)

        vert = VerticalSplitPanel()
        vert.setSplitPosition(242, ISizeable.UNITS_PIXELS)
        vert.setSizeFull()
        v.addComponent(vert)
        v.setExpandRatio(vert, 1.0)

        v_top = VerticalLayout()
        v_top.setSizeFull()
        v_top.setMargin(True)
        v_top.setSpacing(True)
        vert.addComponent(v_top)

        # Name
        h = HorizontalLayout()
        h.setMargin(False)
        h.setSpacing(True)
        v_top.addComponent(h)

        h.addComponent(Label(_("Name:")))
        self._name = TextField()
        self._name.focus()
        h.addComponent(self._name)
        h.setExpandRatio(self._name, 1.0)

        # Formula and calendar list
        h = HorizontalLayout()
        h.setSizeFull()
        h.setMargin(False)
        h.setSpacing(True)
        v_top.addComponent(h)
        v_top.setExpandRatio(h, 1.0)

        t = TextArea(_('Formula:'))
        t.setDebugId('formula')
        t.setSizeFull()
        t.setDescription(_('Calendar formula'))
        t.setImmediate(True)
        t.addValidator(FormulaValidator())
        t.addListener(self, IValueChangeListener)
        h.addComponent(t)
        h.setExpandRatio(t, 2.0)
        self._formula = t

        b = Button("<<")
        b.setDescription(_("Append a formula from an existing calendar"))
        b.addListener(CopyFromExistingCalendar(self), IClickListener)
        h.addComponent(b)
        h.setComponentAlignment(b, Alignment.MIDDLE_CENTER)

        c = SelectCalendar(self._c._sql_session, self._c._workload)
        c.setSizeFull()
        h.addComponent(c)
        h.setExpandRatio(c, 1.0)
        self._calendar_select = c

        # Calendar preview
        c = CalendarView(text_area_id='formula')
        vert.addComponent(c)
        self._calendar_view = c

        # Fill the form
        if cal_id is not None:
            try:
                c = sql_get_cal(self._c._sql_session, cal_id,
                                self._c._workload)
            except:
                self._c._main_window.showNotification(
                                _("Cannot get the calendar details"),
                                _("<br/>Maybe someone else just removed it."),
                                Notification.TYPE_ERROR_MESSAGE)
                return
            self._old_name = c.name
            self._name.setValue(c.name.encode('utf-8'))
            self._formula.setValue(c.formula.encode('utf-8'))
            self._calendar_view.set_formula(c.formula)
        else:
            self._old_name = None

        # Button box
        h_bt = HorizontalLayout()
        h_bt.setMargin(True)
        h_bt.setSpacing(True)

        for caption in self._bt_captions:
            b = Button(_(caption))
            b.addListener(self, IClickListener)
            h_bt.addComponent(b)

        v.addComponent(h_bt)
        v.setComponentAlignment(h_bt, Alignment.BOTTOM_RIGHT)

        self._c._main_window.addWindow(self._w)
        if x and y:
            self._w.setPositionX(x)
            self._w.setPositionY(y)
        else:
            self._w.center()

    def copy_from_existing_calendar(self):
        """Append the existing selected calendar formula to the textarea."""
        cal_id = self._calendar_select.getSelected()
        if cal_id is not None:
            try:
                cal = sql_get_cal(self._c._sql_session, cal_id,
                                  self._c._workload, True)
            except:
                self._c._main_window.showNotification(
                                _("Cannot get the calendar details"),
                                _("<br/>Maybe someone else just removed it."),
                                Notification.TYPE_ERROR_MESSAGE)
                return
            # Get the full path to add it as a comment (description)
            pcal = PathCal(self._c._sql_session, id=cal_id,
                           workload=self._c._workload)
            self._formula.setValue(self._formula.getValue() +
                                   "\n# From " + str(pcal) +
                                   "\n" + cal.formula)

    def valueChange(self, event):
        """Callback for when the formula textarea is changed."""
        self._calendar_view.set_formula(self._formula.getValue())

    def buttonClick(self, event):
        """Callback for the `Cancel' and `OK' buttons."""
        # First button is Cancel
        if event.getButton().getCaption() != _(self._bt_captions[0]):
            # Name sanity checks
            name = self._name.getValue().strip()
            if not name:
                self._c._main_window.showNotification(
                            _("Name is empty"),
                            _("<br/>The calendar name cannot \
                                  be empty or contain just spaces."),
                            Notification.TYPE_ERROR_MESSAGE)
                self._name.focus()
                return
            session = self._c._sql_session.open_session()
            cals = filter(lambda i: i.name.encode('utf-8') == name,
                          sql_get_cal_children(session, self._parent_id,
                                               workload=self._c._workload))
            if cals and (self._cal_id is None or cals[0].id != self._cal_id):
                self._c._sql_session.cancel_session(session)
                self._c._main_window.showNotification(
                        _("The name is already used"),
                        _("<br/>The specified name is already taken."),
                        Notification.TYPE_ERROR_MESSAGE)
                self._name.focus()
                return
            formula = self._formula.getValue()
            if self._cal_id is None:
                # New calendar
                if self._c._workload is None:
                    cal = calendars(self._parent_id, name, 0, '', formula)
                else:
                    cal = calendars_s(self._parent_id, name, 0, '', formula,
                                      self._c._workload)
                session.add(cal)
            else:
                # Update existing calendar
                try:
                    cal = sql_get_cal(session, self._cal_id, self._c._workload,
                                      True)
                except:
                    self._c._sql_session.cancel_session(session)
                    self._c._main_window.showNotification(
                            _("Cannot get the edited calendar details"),
                            _("<br/>Maybe someone else just removed it."),
                            Notification.TYPE_ERROR_MESSAGE)
                    # Close the window
                    self._c._main_window.removeWindow(self._w)
                    return
                cal.name = name.decode('utf-8')
                cal.formula = formula.decode('utf-8')
            self._c._sql_session.close_session(session)
            self._c.repaint()
            self._c.refresh_main()
        # Close the window
        self._c._main_window.removeWindow(self._w)


class FormulaValidator(AbstractValidator):

    """Check that the provided calendar formula is valid."""

    def __init__(self):
        super(FormulaValidator, self).__init__(_("Syntax error"))

    def isValid(self, value):
        """Tests if the given value is a valid formula.

        None values are always accepted.

        @param value:
                 the calendar formula to check
        @return: True if the value is a valid calendar formula, False otherwise
        """
        if value is None:
            return True
        err, idx, c = calcomp.str2cal(value, 2013)
        if err == calcomp.CAL_NOERROR:
            return True
        line_number = value.count('\n', 0, idx) + 1
        if err == calcomp.CAL_EMPTYFIELD:
            msg = _("Empty field line %d") % line_number
        elif err == calcomp.CAL_BADMONTHNAME:
            msg = _("Invalid month name line %d") % line_number
        elif err == calcomp.CAL_BADDAYNAME:
            msg = _("Invalid day name line %d") % line_number
        elif err == calcomp.CAL_MONTHOOR:
            msg = _("Month number out of range line %d") % line_number
        elif err == calcomp.CAL_DAYOOR:
            msg = _("Day number out of range line %d") % line_number
        elif err == calcomp.CAL_BADOPTION:
            msg = _("Invalid option flag line %d") % line_number
        else:
            msg = _("Syntax error line %d") % line_number
        self.setErrorMessage(msg)
        return False


class CopyFromExistingCalendar(IClickListener):

    """Callback for the `<<' button."""

    def __init__(self, calendar_edit_window_obj):
        """Initialize the callback.

        @param calendar_edit_window_obj:
                            the associated L{CalendarEditWindow} object.
        """
        super(CopyFromExistingCalendar, self).__init__()
        self._c = calendar_edit_window_obj

    def buttonClick(self, event):
        self._c.copy_from_existing_calendar()
