# Schedwi
# Copyright (C) 2011-2013 Herve Quatremain
#
# This file is part of Schedwi.
#
# Schedwi is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
#
# Schedwi is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.


"""Parse configuration file."""

import re
import os
import os.path
import sys
import socket

import config
import system_utils


DBI_DRIVERNAME = config.DEFAULT_DBI_DRIVERNAME
DBI_DBNAME = config.DEFAULT_DBI_DBNAME
DBI_USER = config.DEFAULT_DBI_USER
DBI_PASSWORD = config.DEFAULT_DBI_PASSWORD
DBI_DBHOSTNAME = config.DEFAULT_DBI_DBHOSTNAME
DBI_DBDIR = config.DEFAULT_DBI_DBDIR
GUI_REFRESH = config.DEFAULT_GUI_REFRESH
GUI_USERNAME = config.DEFAULT_GUI_USERNAME
GUI_PASSWORD = config.DEFAULT_GUI_PASSWORD
GUI_IFACE_LISTEN = config.DEFAULT_GUI_IFACE_LISTEN
GUI_PORT = config.DEFAULT_GUI_PORT
USER_ID = None
GROUP_ID = None
GUI_LOG_FILE = config.DEFAULT_GUI_LOG_FILE
GUI_CONTEXT_ROOT = config.DEFAULT_GUI_CONTEXT_ROOT


def _parse_file(f, conf_file_name):
    """Parse the provided opened file."""
    global DBI_DRIVERNAME, DBI_DBNAME, DBI_USER, DBI_PASSWORD, \
           DBI_DBHOSTNAME, DBI_DBDIR, GUI_REFRESH, \
           GUI_USERNAME, GUI_PASSWORD, GUI_IFACE_LISTEN, GUI_PORT, \
           USER_ID, GROUP_ID, GUI_LOG_FILE, GUI_CONTEXT_ROOT
    for line in f:
        m = re.search(r'^\s*DBI_DRIVERNAME\s*=\s*(.+)$', line)
        if m:
            DBI_DRIVERNAME = m.group(1).rstrip()
            continue
        m = re.search(r'^\s*DBI_DBNAME\s*=\s*(.+)$', line)
        if m:
            DBI_DBNAME = m.group(1).rstrip()
            continue
        m = re.search(r'^\s*DBI_USER\s*=\s*(.+)$', line)
        if m:
            DBI_USER = m.group(1).rstrip()
            continue
        m = re.search(r'^\s*DBI_PASSWORD\s*=\s*(.+)$', line)
        if m:
            DBI_PASSWORD = m.group(1).rstrip()
            continue
        m = re.search(r'^\s*DBI_DBHOSTNAME\s*=\s*(.+)$', line)
        if m:
            DBI_DBHOSTNAME = m.group(1).rstrip()
            continue
        m = re.search(r'^\s*DBI_DBDIR\s*=\s*(.+)$', line)
        if m:
            DBI_DBDIR = m.group(1).rstrip()
            continue
        m = re.search(r'^\s*GUI_REFRESH\s*=\s*(.+)$', line)
        if m:
            try:
                i = int(m.group(1).rstrip())
            except:
                sys.stderr.write(
                        _("%s: erroneous value `%s' for GUI_REFRESH.") % \
                                (conf_file_name, m.group(1).rstrip()))
                sys.stderr.write("  ")
                sys.stderr.write(_("Using %d instead.\n") % GUI_REFRESH)
            else:
                GUI_REFRESH = i
            continue
        m = re.search(r'^\s*GUI_USERNAME\s*=\s*(.+)$', line)
        if m:
            GUI_USERNAME = m.group(1).rstrip()
            continue
        m = re.search(r'^\s*GUI_PASSWORD\s*=\s*(.+)$', line)
        if m:
            GUI_PASSWORD = m.group(1).rstrip()
            continue
        m = re.search(r'^\s*GUI_IFACE_LISTEN\s*=\s*(.+)$', line)
        if m:
            GUI_IFACE_LISTEN = m.group(1).rstrip()
            continue
        m = re.search(r'^\s*GUI_PORT\s*=\s*(.+)$', line)
        if m:
            try:
                p = socket.getaddrinfo(None, m.group(1).rstrip(),
                                       socket.AF_INET,
                                       socket.SOCK_STREAM)[0][4][1]
            except:
                sys.stderr.write(
                        _("%s: erroneous value `%s' for GUI_PORT.") % \
                                (conf_file_name, m.group(1).rstrip()))
                sys.stderr.write("  ")
                sys.stderr.write(_("Using %d instead.\n") % GUI_PORT)
            else:
                GUI_PORT = p
            continue
        m = re.search(r'^\s*USER\s*=\s*(.+)$', line)
        if m:
            try:
                uid = system_utils.get_user_id(m.group(1).rstrip())
            except:
                sys.stderr.write(
                        _("%s: unknown user `%s' for USER.") % \
                                (conf_file_name, m.group(1).rstrip()))
                sys.stderr.write("  ")
                sys.stderr.write(_("Using the current user instead.\n"))
            else:
                USER_ID = uid
            continue
        m = re.search(r'^\s*GROUP\s*=\s*(.+)$', line)
        if m:
            try:
                gid = system_utils.get_group_id(m.group(1).rstrip())
            except:
                sys.stderr.write(
                        _("%s: unknown group `%s' for GROUP.") % \
                                (conf_file_name, m.group(1).rstrip()))
                sys.stderr.write("  ")
                sys.stderr.write(_("Using the current group instead.\n"))
            else:
                GROUP_ID = gid
            continue
        m = re.search(r'^\s*GUI_LOG_FILE\s*=\s*(.+)$', line)
        if m:
            GUI_LOG_FILE = m.group(1).rstrip()
            continue
        m = re.search(r'^\s*GUI_CONTEXT_ROOT\s*=\s*(.+)$', line)
        if m:
            d = m.group(1).rstrip()
            if os.path.isdir(d):
                GUI_CONTEXT_ROOT = d
            else:
                sys.stderr.write(
                    _("%s: unknown directory `%s' for GUI_CONTEXT_ROOT.") % \
                                (conf_file_name, d))
                sys.stderr.write("  ")
                sys.stderr.write(_("Using the default path instead.\n"))
            continue


def parse(file=None):
    """Parse the configuration files."""
    for conf in (config.GLOBAL_CONFIG_FILE,
                 config.LOCAL_CONFIG_FILE,
                 os.path.expanduser(config.USER_CONFIG_FILE),
                 file):
        if conf is not None:
            try:
                f = open(conf, 'r')
            except IOError:
                pass
            else:
                _parse_file(f, conf)
                f.close()


if __name__ == "__main__":
    parse()
    print "DBI_DRIVERNAME=", DBI_DRIVERNAME
    print "DBI_DBNAME=", DBI_DBNAME
    print "DBI_USER=", DBI_USER
    print "DBI_PASSWORD=", DBI_PASSWORD
    print "DBI_DBHOSTNAME=", DBI_DBHOSTNAME
    print "DBI_DBDIR=", DBI_DBDIR
    print "GUI_REFRESH=", GUI_REFRESH
    print "GUI_USERNAME=", GUI_USERNAME
    print "GUI_PASSWORD=", GUI_PASSWORD
