# Schedwi
# Copyright (C) 2011-2013 Herve Quatremain
#
# This file is part of Schedwi.
#
# Schedwi is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
#
# Schedwi is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.


"""Module to set jobs or jobsets parameters."""

import getopt
import time
import sys

from sqlalchemy.orm.exc import NoResultFound, MultipleResultsFound

from tables.job_start_limit import job_start_limit
from tables.job_max_duration import job_max_duration
from tables.job_retries import job_retries
from tables.job_retries_interval import job_retries_interval
from tables.job_username import job_username
from tables.job_file_out import job_file_out
from tables.job_file_err import job_file_err
from tables.job_control_group import job_control_group
from tables.job_loadenv import job_loadenv
from tables.job_success_return_code import job_success_return_code
from tables.job_command import job_command
from tables.job_arguments import job_arguments
from tables.job_environment import job_environment
from tables.job_host import job_host
from tables.constraint_file import constraint_file
import path_cal
import host_utils
import environments_utils


def usage():
    """Print a usage message on STDOUT."""
    print _("""Usage: set [OPTION]... JOB|JOBSET
Change a job or jobset parameter.

Options:
  --enable=True|False     enable (True) or disable (False) the job/jobset.
  --host=HOSTNAME[:PORT]  host on which the job/jobset must be running.
  --delhost               remove the hostname so this is the one from the
                          parent jobset which is going to be used instead.
  --calendar=CAL_NAME     calendar name.
  --delcalendar           remove the calendar so the job/jobset will use the
                          same calendar as its parent jobset.
  --start=TIME            start time of the job (HH:MM).
  --delstart              use the start time of the parent jobset.
  --command=COMMAND       command to run (if set to `-', no command will be
                          started and the job will complete immediately)
  --delcommand            no command.  Use the one defined in the
                          parent jobset.
  --parameter=PARAMETER   a parameter to pass to the command.  This option can
                          be specified several times to define more than one
                          parameter.
  --delparameter          use the parent jobset parameters instead.
  --noparameter           no parameter at all for the command (the ones from
                          the parent jobset are ignored).
  --return=RETURN_CODE    a return code from the command equal or below this
                          value means success.
  --delreturn             use the return code value defined at the parent
                          jobset level.
  --user=USERNAME         username under which the command must be run.
  --deluser               use the username defined in the parent
                          jobset instead.
  --stdout=FILE           command output file (stored on the Schedwi agent).
  --delstdout             use the file name defined in the parent
                          jobset instead.
  --stderr=FILE           command error file (stored on the Schedwi agent).
  --delstderr             use the file name defined in the parent
                          jobset instead.
  --cgroup=NAME           Linux Control Group name to set for the job.  This
                          name is relative to the controllers mount point.
                          For instance, for the /cgroup/cpu/schedwi/test mount
                          point, NAME must be `schedwi/test'.
  --delcgroup             use the Linux Control Group defined by the parent
                          jobset instead.
  --duration=MINUTES      maximum number of minutes for the job/jobset to
                          complete once running.
  --delduration           use the value defined in the parent jobset instead.
  --retries=NUM           number of retries when the job/jobset fails.
  --delretries            use the value defined in the parent jobset instead.
  --retries-int=MINUTES   number of minutes between retries.
  --delretries-int        use the value defined in the parent jobset instead.
  --limit=MINUTES         number of minutes for the job to be started past its
                          start time (number of minutes or HH:MM)
  --dellimit              use the value defined in the parent jobset instead.
  --user-env=True|False   load (True) or not (False) the user account
                          environment (~/.profile or ~/.bashrc for instance) on
                          the remote host before running the job command.
  --deluser-env           use the value defined in the parent jobset instead.
  --addenv=ENV_NAME       add an environment group to the job/jobset.  This
                          option can be specified several times.
  --delenv=ENV_NAME       remove the provided environment group.
  --enableenv             enable the use of environment groups.
  --disableenv            do not use any environment group for this job/jobset.
  --addfile=HOSTNAME[:PORT],True|False,FILE     add a file to check before
                          starting the job/jobset.  The value is composed of
                          three fields separated by a comma.  The first one is
                          the remote agent hostname on which to check the file.
                          The second specifies if the file must be present
                          (True) or not (False).  The last one is the file or
                          directory to check.
  --delfile=HOSTNAME[:PORT],FILE    remove the provided file.
  --description=TEXT      a description.
  -h, --help              display this help.
    """)


def parse_args(argv):
    """Parse the argument list.

    Argument:
    argv --  argument list to parse

    Exception:
    getopt.getopterror -- option error (unknow option or missing parameter)

    """
    optlist, args = getopt.getopt(argv, "h",
            [
                "enable=",
                "host=",
                "delhost",
                "calendar=",
                "delcalendar",
                "start=",
                "delstart",
                "command=",
                "delcommand",
                "parameter=",
                "delparameter",
                "noparameter",
                "return=",
                "delreturn",
                "user=",
                "deluser",
                "stdout=",
                "delstdout",
                "stderr=",
                "delstderr",
                "cgroup=",
                "delcgroup",
                "duration=",
                "delduration",
                "retries=",
                "delretries",
                "retries-int=",
                "delretries-int",
                "limit=",
                "dellimit",
                "user-env=",
                "deluser-env",
                "addenv=",
                "delenv=",
                "enableenv",
                "disableenv",
                "description=",
                "addfile=",
                "delfile=",
                "help"
            ])
    return (optlist, args)


def set(session, current_cwd, current_cwd_cal, job, opts):
    """Change jobs and jobsets parameters.

    Arguments:
    session -- SQLAlchemy session
    current_cwd -- current working jobset (a path.Path object)
    current_cwd_cal -- current working calendar directory (a path_cal.PathCal
                       object)
    job -- job object to change
    opts -- option list (from getopt.getopt() - see parse_args())

    """
    arguments = list()
    constraint_files = list()
    envs = list()
    for o, a in opts:
        if o == "--enable":
            if a[0] in (_('Y'), _('y'), '1', _('t'), _('T')):
                job.enabled = 1
            else:
                job.enabled = 0
        elif o == "--host":
            try:
                host = host_utils.name2host(session, a)
            except NoResultFound:
                sys.stderr.write(_("No such host (--host=%s)\n") % a)
                return 1
            except MultipleResultsFound:
                sys.stderr.write(_("More than one host match the provided \
name (--host=%s)\n") % a)
                return 1
            job.job_host = job_host(host.id)

        elif o == "--delhost":
            del job.job_host

        elif o == "--calendar":
            c = path_cal.PathCal(session, path=a, cwd=current_cwd_cal)
            if not c:
                sys.stderr.write(_("Unknown calendar %s (--calendar)\n") % a)
                return 1
            job.cal_id = c.id[-1]

        elif o == "--delcalendar":
            job.cal_id = 0

        elif o == "--start":
            for fmt in ("%H:%M", "%I:%M %p", "%Hh%M", "%Ih%M %p"):
                try:
                    tstr = time.strptime(a, fmt)
                except ValueError:
                    continue
                else:
                    break
            else:
                sys.stderr.write(_("Unknown time format (--start).  Should be \
`H:M' or `H:M AM/PM'\n"))
                return 1
            job.start_time = tstr.tm_hour * 100 + tstr.tm_min

        elif o == "--delstart":
            job.start_time = -1

        elif o == "--command":
            job.job_command = job_command(a)

        elif o == "--delcommand":
            del job.job_command

        elif o == "--parameter":
            arguments.append(job_arguments(len(arguments) + 1, a))

        elif o == "--delparameter":
            arguments = list()
            job.job_arguments = []

        elif o == "--noparameter":
            arguments = list()
            job.job_arguments = [job_arguments(-1, "")]

        elif o == "--return":
            try:
                v = int(a)
                if v < 0 or v > 255:
                    raise ValueError
            except:
                sys.stderr.write(
                  _("Wrong value for --return.  Must be between 0 and 255\n"))
                return 1
            job.job_success_return_code = job_success_return_code(v)

        elif o == "--delreturn":
            del job.job_success_return_code
        elif o == "--user":
            job.job_username = job_username(a)
        elif o == "--deluser":
            del job.job_username
        elif o == "--stdout":
            job.job_file_out = job_file_out(a)
        elif o == "--delstdout":
            del job.job_file_out
        elif o == "--stderr":
            job.job_file_err = job_file_err(a)
        elif o == "--delstderr":
            del job.job_file_err
        elif o == "--cgroup":
            job.job_control_group = job_control_group(a)
        elif o == "--delcgroup":
            del job.job_control_group
        elif o == "--duration":
            try:
                v = int(a)
                if v < 0:
                    raise ValueError
            except:
                sys.stderr.write(_("Wrong value for --duration.  Must be a \
number of minutes (0 means no limit)\n"))
                return 1
            job.job_max_duration = job_max_duration(v)

        elif o == "--delduration":
            del job.job_max_duration

        elif o == "--retries":
            try:
                v = int(a)
                if v < 0:
                    raise ValueError
            except:
                sys.stderr.write(_("Wrong value for --retries\n"))
                return 1
            job.job_retries = job_retries(v)

        elif o == "--delretries":
            del job.job_retries

        elif o == "--retries-int":
            try:
                v = int(a)
                if v < 0:
                    raise ValueError
            except:
                sys.stderr.write(_("Wrong value for --retries-int.  Must be \
the number of minutes between retries\n"))
                return 1
            job.job_retries_interval = job_retries_interval(v)

        elif o == "--delretries-int":
            del job.job_retries_interval

        elif o == "--limit":
            for fmt in ("%H:%M", "%Hh%M"):
                try:
                    tstr = time.strptime(a, fmt)
                except ValueError:
                    continue
                else:
                    v = tstr.tm_hour * 60 + tstr.tm_min
                    break
            else:
                try:
                    v = int(a)
                    if v < 0:
                        raise ValueError
                except:
                    sys.stderr.write(_("Wrong value for --limit.  Must be the \
number of minutes for the job to be started past its start time \
(0 means no limit)\n"))
                    return 1
            job.job_start_limit = job_start_limit(v)

        elif o == "--dellimit":
            del job.job_start_limit

        elif o == "--user-env":
            if a[0] in (_('Y'), _('y'), '1', _('t'), _('T')):
                job.job_loadenv = job_loadenv(1)
            else:
                job.job_loadenv = job_loadenv(0)
        elif o == "--deluser-env":
            del job.job_loadenv

        elif o == "--addenv":
            try:
                env = environments_utils.name2env(session, a)
            except NoResultFound:
                sys.stderr.write(_("No such environment %s (--addenv)\n") % a)
                return 1
            try:
                envs.remove(env.id)
            except:
                pass
            envs.append(env.id)

        elif o == "--delenv":
            try:
                env = environments_utils.name2env(session, a)
            except NoResultFound:
                sys.stderr.write(_("No such environment %s (--delenv)\n") % a)
                return 1
            for e in job.job_environment:
                if env.id == e.env_id:
                    job.job_environment.remove(e)
                    break
            envs = filter(lambda id: id != env.id, envs)

        elif o == "--enableenv":
            job.has_env = 1

        elif o == "--disableenv":
            job.has_env = 0

        elif o == "--description":
            job.description = a.decode('utf-8')

        elif o == "--addfile":
            try:
                h, exist, filename = a.split(',', 2)
            except:
                sys.stderr.write(_("Wrong format for --addfile.  Must be \
<host>,<0|1>,<file>\n"))
                return 1
            try:
                host = host_utils.name2host(session, h)
            except NoResultFound:
                sys.stderr.write(_("No such host %s (--addfile=%s)\n")
                                 % (h, a))
                return 1
            except MultipleResultsFound:
                sys.stderr.write(_("More than one host matches the provided \
name %s (--addfile=%s)\n") % (h, a))
                return 1
            try:
                constraint_files.remove(constraint_file(host.id, filename, 1))
            except:
                pass
            if exist[0] in (_('Y'), _('y'), '1', _('t'), _('T')):
                constraint_files.append(constraint_file(host.id, filename, 1))
            else:
                constraint_files.append(constraint_file(host.id, filename, 0))

        elif o == "--delfile":
            try:
                h, exist, filename = a.split(',', 2)
            except:
                try:
                    h, filename = a.split(',', 1)
                except:
                    sys.stderr.write(
                    _("Wrong format for --delfile.  Must be <host>,<file>\n"))
                    return 1
            try:
                host = host_utils.name2host(session, h)
            except NoResultFound:
                sys.stderr.write(_("No such host %s (--delfile=%s)\n")
                                 % (h, a))
                return 1
            except MultipleResultsFound:
                sys.stderr.write(_("More than one host match the provided \
name %s (--delfile=%s)\n") % (h, a))
                return 1
            try:
                job.constraint_file.remove(constraint_file(host.id,
                                                           filename, 1))
            except:
                pass
            try:
                constraint_files.remove(constraint_file(host.id, filename, 1))
            except:
                pass
        elif o in ("-h", "--help"):
            usage()

    if arguments:
        job.job_arguments = arguments
    if constraint_files:
        for f in job.constraint_file:
            try:
                constraint_files.remove(f)
            except:
                pass
        job.constraint_file.extend(constraint_files)
    if envs:
        new_envs = list()
        last_position = 0
        for e in job.job_environment:
            try:
                envs.index(e.env_id)
            except:
                pass
            else:
                continue
            if e.position > last_position:
                last_position = e.position
            new_envs.append(e)
        for env_id in envs:
            last_position += 1
            new_envs.append(job_environment(env_id, last_position))
        job.job_environment = new_envs
    return 0
