# Schedwi
# Copyright (C) 2011-2013 Herve Quatremain
#
# This file is part of Schedwi.
#
# Schedwi is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
#
# Schedwi is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.


"""Module to define a new host."""

import sys
import getopt

import sqlalchemy.exc

import set_options
from tables.hosts import hosts
import host_utils
from help import print_trim


def usage():
    """Print a usage message on STDOUT."""
    print_trim(_("""Usage: mk [OPTION]... HOST...
        Create new hosts.  HOST must have the following format:
           HOSTNAME[:PORT]
        For IPv6, the address must be enclosed between square brackets (ie.
        [fe80::210:a2ff:fa1d:aabb:ccdd]:2006)
        Notice that the recommended way to create new agents is to use the
        schedwica command (see man 8 schedwica for more details)"""))
    set_options.usage_options(with_hostname=False)


def mk(sql_session, arguments):
    """Create a new host.

    Arguments:
    sql_session -- SQLAlchemy session
    arguments -- list of arguments given to the mk command (list of hosts)

    """
    # Parse the parameters
    try:
        optlist, args = set_options.parse_args(arguments)
    except getopt.GetoptError, err:
        sys.stderr.write(_("mk: ") + str(err) + "\n")
        return 1
    for o, a in optlist:
        if o in ("-h", "--help"):
            usage()
            return 0
    if not args:
        sys.stderr.write(_("mk: missing the host name to create\n"))
        return 1
    session = sql_session.open_session()
    error = False
    for a in args:
        hostname, port = host_utils.parse_host(a)
        if hostname is None:
            sys.stderr.write(_("mk: unknown host `%s'\n") % a)
            sql_session.cancel_session(session)
            return 1
        host = hosts(hostname, port if port is not None else 2006,
                     0, '', '')
        session.add(host)
        if set_options.set(session, host, optlist) != 0:
            sql_session.cancel_session(session)
            return 1
    try:
        sql_session.close_session(session)
    except sqlalchemy.exc.IntegrityError:
        sql_session.cancel_session(session)
        sys.stderr.write(
                _("mk: a host with this name and this port already exists\n"))
        return 1
    return 0
