/* Schedwi
   Copyright (C) 2007-2010 Herve Quatremain

   This file is part of Schedwi.

   Schedwi is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 3 of the License, or
   (at your option) any later version.

   Schedwi is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

/*
 * startjob_request.c -- Build the string which contains the network request
 *                       to start a job
 */

#include <schedwi.h>

#if STDC_HEADERS
#include <stdlib.h>
#include <string.h>
#else
#if HAVE_STDLIB_H
#include <stdlib.h>
#endif
#if HAVE_STRING_H
#include <string.h>
#endif
#endif

#if HAVE_ASSERT_H
#include <assert.h>
#endif

#include <lwc_log.h>
#include <lib_functions.h>
#include <startjob_request.h>
#include <utils.h>


/*
 * Create a new startjob_request_ptr structure
 *
 * Return:
 *   The new structure (to be freed by startjob_request_destroy()) or
 *   NULL in case of memory allocation error
 */
startjob_request_ptr
startjob_request_new ()
{
	startjob_request_ptr ptr;

	ptr = (startjob_request_ptr ) malloc (sizeof (startjob_request));
	if (ptr != NULL) {
		ptr->parameters = ptr->env = ptr->args = NULL;
	}
	return ptr;
}


/*
 * Destroy (free) the provided startjob_request_ptr structure
 */
void
startjob_request_destroy (startjob_request_ptr ptr)
{
	if (ptr != NULL) {
		lwc_delLL (ptr->parameters, (void (*)(const void *))free);
		lwc_delLL (ptr->env, (void (*)(const void *))free);
		lwc_delLL (ptr->args, (void (*)(const void *))free);
		free (ptr);
	}
}


/*
 * Add an argument to the startjob_request object
 *
 * Return:
 *   0 --> No error
 *  -1 --> Memory allocation error (lwc_writeLog() is used to log an
 *         error message)
 */
int
startjob_request_add_argument (	startjob_request_ptr ptr,
				const char *pos, unsigned int len_pos,
				const char *val, unsigned int len_val)
{
	char *tmp;

#if HAVE_ASSERT_H
	assert (ptr != NULL && pos != NULL && val != NULL);
#endif

	if (ptr->args == NULL) {
		ptr->args = lwc_newLL ();
		if (ptr->args == NULL) {
			lwc_writeLog (LOG_CRIT, _("Memory allocation error"));
			return -1;
		}
	}

	tmp = escape_string (val, len_val, 1);
	if (tmp == NULL || lwc_addEndLL (ptr->args, tmp) != 0) {
		if (tmp != NULL) {
			free (tmp);
		}
		lwc_writeLog (LOG_CRIT, _("Memory allocation error"));
		return -1;
	}

	return 0;
}


/*
 * Add an environment variable to the startjob_request object
 *
 * Return:
 *   0 --> No error
 *  -1 --> Memory allocation error (lwc_writeLog() is used to display an
 *         error message)
 */
int
startjob_request_add_environment (	startjob_request_ptr ptr,
					const char *pos,
					const char *key, const char *value)
{
	unsigned int len_key, len_value;
	char *tmp, *s;

#if HAVE_ASSERT_H
	assert (ptr != NULL && pos != NULL && key != NULL && value != NULL);
#endif

	if (ptr->env == NULL) {
		ptr->env = lwc_newLL ();
		if (ptr->env == NULL) {
			lwc_writeLog (LOG_CRIT, _("Memory allocation error"));
			return -1;
		}
	}

	/* Compute the size of the strings */
	len_key   = schedwi_strlen (key);
	len_value = schedwi_strlen (value);

	s = (char *) malloc (len_key + len_value + 2);
	if (s == NULL) {
		lwc_writeLog (LOG_CRIT, _("Memory allocation error"));
		return -1;
	}
	strcpy (s, key);
	s[len_key] = '=';
	strcpy (s + len_key + 1, value);
	tmp = escape_string (s, len_key + len_value + 1, 1);
	free (s);
	if (tmp == NULL || lwc_addEndLL (ptr->env, tmp) != 0) {
		if (tmp != NULL) {
			free (tmp);
		}
		lwc_writeLog (LOG_CRIT, _("Memory allocation error"));
		return -1;
	}

	return 0;
}


/*
 * Add a string parameter to the startjob_request object
 *
 * Return:
 *   0 --> No error
 *  -1 --> Memory allocation error (lwc_writeLog() is used to display an
 *         error message)
 */
int
startjob_request_add_parameter_string (	startjob_request_ptr ptr,
					const char *name,
					const char *val, unsigned int len_val)
{
	unsigned int len_name;
	char *tmp, *s;

#if HAVE_ASSERT_H
	assert (ptr != NULL && name != NULL && val != NULL);
#endif

	if (ptr->parameters == NULL) {
		ptr->parameters = lwc_newLL ();
		if (ptr->parameters == NULL) {
			lwc_writeLog (LOG_CRIT, _("Memory allocation error"));
			return -1;
		}
	}

	tmp = escape_string (val, len_val, 1);
	if (tmp == NULL) {
		lwc_writeLog (LOG_CRIT, _("Memory allocation error"));
		return -1;
	}

	len_name = schedwi_strlen (name);
	s = (char *) malloc (len_name + schedwi_strlen (tmp) + 5);
	if (s == NULL) {
		free (tmp);
		lwc_writeLog (LOG_CRIT, _("Memory allocation error"));
		return -1;
	}

	s[0] = '"';
	strcpy (s + 1, name);
	s[len_name + 1] = '"';
	s[len_name + 2] = ':';
	strcpy (s + len_name + 3, tmp);
	free (tmp);

	if (lwc_addEndLL (ptr->parameters, s) != 0) {
		free (s);
		lwc_writeLog (LOG_CRIT, _("Memory allocation error"));
		return -1;
	}

	return 0;
}


/*
 * Add a boolean parameter to the startjob_request object
 *
 * Return:
 *   0 --> No error
 *  -1 --> Memory allocation error (lwc_writeLog() is used to display an
 *         error message)
 */
int
startjob_request_add_parameter_bool (startjob_request_ptr ptr,
					const char *name, char val)
{
	unsigned int len_name;
	char *s;

#if HAVE_ASSERT_H
	assert (ptr != NULL && name != NULL);
#endif

	if (ptr->parameters == NULL) {
		ptr->parameters = lwc_newLL ();
		if (ptr->parameters == NULL) {
			lwc_writeLog (LOG_CRIT, _("Memory allocation error"));
			return -1;
		}
	}

	len_name = schedwi_strlen (name);
	s = (char *) malloc (len_name + 10);
	if (s == NULL) {
		lwc_writeLog (LOG_CRIT, _("Memory allocation error"));
		return -1;
	}

	s[0] = '"';
	strcpy (s + 1, name);
	s[len_name + 1] = '"';
	s[len_name + 2] = ':';
	strcpy (s + len_name + 3, (val == 0) ? "false" : "true");

	if (lwc_addEndLL (ptr->parameters, s) != 0) {
		free (s);
		lwc_writeLog (LOG_CRIT, _("Memory allocation error"));
		return -1;
	}

	return 0;
}


/*
 * Add the Job ID parameter to the string
 *
 * Return:
 *   0 --> No error
 *  -1 --> Memory allocation error (lwc_writeLog() is used to display an
 *         error message)
 */
int
startjob_request_add_jobid (	startjob_request_ptr ptr,
				const char *workload,
				const char *job_id_str,
				unsigned int len_job_id_str)
{
	unsigned int len_workload, len_key;
	char *s;
	char *json_key = "\"job id\":";

#if HAVE_ASSERT_H
	assert (ptr != NULL && workload != NULL);
#endif

	if (ptr->parameters == NULL) {
		ptr->parameters = lwc_newLL ();
		if (ptr->parameters == NULL) {
			lwc_writeLog (LOG_CRIT, _("Memory allocation error"));
			return -1;
		}
	}

	len_key = schedwi_strlen (json_key);
	len_workload = schedwi_strlen (workload);
	s = (char *) malloc (len_key + len_workload + len_job_id_str + 5);
	if (s == NULL) {
		lwc_writeLog (LOG_CRIT, _("Memory allocation error"));
		return -1;
	}

	strcpy (s, json_key);
	s[len_key] = '"';
	strcpy (s + len_key + 1, workload);
	s[len_key + len_workload + 1] = SCHEDWI_WORKLOAD_SEPARATOR;
	strncpy (s + len_key + len_workload + 2, job_id_str, len_job_id_str);
	s[len_key + len_workload + 2 + len_job_id_str] = '"';
	s[len_key + len_workload + 3 + len_job_id_str] = '\0';

	if (lwc_addEndLL (ptr->parameters, s) != 0) {
		free (s);
		lwc_writeLog (LOG_CRIT, _("Memory allocation error"));
		return -1;
	}

	return 0;
}


/*
 * Send the resquest on the network
 */
int
net_write_request (	startjob_request_ptr ptr,
			schedwi_BIO *b,
			const char *module_name)
{
	int ret;
	char *s, first;

#if HAVE_ASSERT_H
	assert (ptr != NULL && module_name != NULL);
#endif

	s = "[ \"";
	ret = net_write (b, s, schedwi_strlen (s));
	if (ret != 0) {
		return ret;
	}

	ret = net_write (b, module_name, schedwi_strlen (module_name));
	if (ret != 0) {
		return ret;
	}

	s = "\", { ";
	ret = net_write (b, s, schedwi_strlen (s));
	if (ret != 0) {
		return ret;
	}

	/* Send the global parameters */
	lwc_rewindLL (ptr->parameters);
	first = 1;
	while ((s = (char *)lwc_nextLL (ptr->parameters)) != NULL) {
		if (first != 1) {
			ret = net_write (b, ", ", 2);
			if (ret != 0) {
				return ret;
			}
		}
		ret = net_write (b, s, schedwi_strlen (s));
		if (ret != 0) {
			return ret;
		}
		first = 0;
	}

	/* Send the arguments */
	if (lwc_getNumNode (ptr->args) != 0) {
		if (first != 1) {
			s = ", ";
			ret = net_write (b, s, schedwi_strlen (s));
			if (ret != 0) {
				return ret;
			}
		}
		s = "\"args\":[";
		ret = net_write (b, s, schedwi_strlen (s));
		if (ret != 0) {
			return ret;
		}

		lwc_rewindLL (ptr->args);
		first = 1;
		while ((s = (char *)lwc_nextLL (ptr->args)) != NULL) {
			if (first != 1) {
				ret = net_write (b, ",", 1);
				if (ret != 0) {
					return ret;
				}
			}
			ret = net_write (b, s, schedwi_strlen (s));
			if (ret != 0) {
				return ret;
			}
			first = 0;
		}
		ret = net_write (b, "]", 1);
		if (ret != 0) {
			return ret;
		}
	}

	/* Send the environment */
	if (lwc_getNumNode (ptr->env) != 0) {
		if (first != 1) {
			s = ", ";
			ret = net_write (b, s, schedwi_strlen (s));
			if (ret != 0) {
				return ret;
			}
		}
		s = "\"env\":[";
		ret = net_write (b, s, schedwi_strlen (s));
		if (ret != 0) {
			return ret;
		}

		lwc_rewindLL (ptr->env);
		first = 1;
		while ((s = (char *)lwc_nextLL (ptr->env)) != NULL) {
			if (first != 1) {
				ret = net_write (b, ",", 1);
				if (ret != 0) {
					return ret;
				}
			}
			ret = net_write (b, s, schedwi_strlen (s));
			if (ret != 0) {
				return ret;
			}
			first = 0;
		}
		ret = net_write (b, "]", 1);
		if (ret != 0) {
			return ret;
		}
	}

	s = " } ]";
	ret = net_write (b, s, schedwi_strlen (s));
	if (ret != 0) {
		return ret;
	}

	return 0;
}

/*-----------------============== End Of File ==============-----------------*/
