# Schedwi
# Copyright (C) 2011 Herve Quatremain
# 
# This file is part of Schedwi.
# 
# Schedwi is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
# 
# Schedwi is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.


"""Job and jobset status."""

import time

import sqlalchemy.orm.session

from tables.job_status import job_status
import colors

WAITING, RUNNING, COMPLETED, FAILED = 1, 2, 3, 4

def status2string(st):
    """Convert a status number to a string."""
    if st == WAITING:
        return "waiting"
    elif st == RUNNING:
        return "running"
    elif st == COMPLETED:
        return "completed"
    else:
        return "failed"

def string2status(str_status):
    """Convert a string to a status number."""
    c = str_status[0].lower()
    if c == 'w':
       return WAITING
    elif c == 'r':
       return RUNNING
    elif c == 'c':
       return COMPLETED
    elif c == 'f':
       return FAILED
    return 0

def status2char(st):
    """Convert a status number to a single character."""
    if st == WAITING:
        return "w"
    elif st == RUNNING:
        return "r"
    elif st == COMPLETED:
        return "c"
    else:
        return "f"

def status2color(st):
    """Convert a status number to a color code."""
    if st == WAITING:
        return colors.YELLOW
    elif st == RUNNING:
        return colors.BLUE
    elif st == COMPLETED:
        return colors.GREEN
    else:
        return colors.RED

def get_status(sql_session, job_id, workload):
    """Return the job_status object of the specified job_id."""
    if not isinstance(sql_session, sqlalchemy.orm.session.Session):
        session = sql_session.open_session()
    else:
        session = sql_session
    query = session.query(job_status).filter(job_status.job_id == job_id)
    query = query.filter(job_status.workload_date == workload)
    try:
        status = query.one()
    except:
        if not isinstance(sql_session, sqlalchemy.orm.session.Session):
            sql_session.close_session(session)
        return ''
    if not isinstance(sql_session, sqlalchemy.orm.session.Session):
        sql_session.close_session(session)
    return status

def get_status_string(sql_session, job_id, workload):
    """Return a string representing the status of the given job_id."""
    status = get_status(sql_session, job_id, workload)
    if not status:
        return "waiting"
    return "%s (%s - try %d%s)" % (
           status2string(status.status),
           time.strftime("%Hh%M", time.localtime(status.start_time)),
           status.retry_num,
           (" - %s" % status.error_msg) if status.error_msg else '')

