# Schedwi
# Copyright (C) 2011 Herve Quatremain
# 
# This file is part of Schedwi.
# 
# Schedwi is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
# 
# Schedwi is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.


"""Functions to get parameters from the job/jobset hierarchy."""

from sqlalchemy.orm.exc import NoResultFound
import sqlalchemy.orm.session

def get_by_id(sql_session, ids, obj, workload=None):
    """Get the parameter from the job/jobset hierarchy.

    Arguments:
    sql_session -- SQLAlchemy session
    ids -- list of IDs of the object
    obj -- database object to retrive
    workload -- workload to use

    Return the tuple (from_parents, obj) with from_parents to True if the
    parameter has been retrieved from the object itself or False if it is not
    defined in the object but has been retrieved from a parent (inherited)

    """
    if not isinstance(sql_session, sqlalchemy.orm.session.Session):
        session = sql_session.open_session()
    else:
        session = sql_session
    # First, try to get the data from the ID of the object
    query = session.query(obj).filter_by(job_id=ids[-1])
    if workload is not None:
        query = query.filter_by(workload_date=workload)
    try:
        query = query.one()
    except NoResultFound:
        pass
    else:
        if not isinstance(sql_session, sqlalchemy.orm.session.Session):
            sql_session.close_session(session)
        return (True, query)
    # Otherwise, look in the parent hierarchy
    for i in range(len(ids) - 2, -1, -1):
        query = session.query(obj).filter_by(job_id=ids[i])
        if workload is not None:
            query = query.filter_by(workload_date=workload)
        try:
            query = query.one()
        except NoResultFound:
            pass
        else:
            if not isinstance(sql_session, sqlalchemy.orm.session.Session):
                sql_session.close_session(session)
            return (False, query)
    if not isinstance(sql_session, sqlalchemy.orm.session.Session):
        sql_session.close_session(session)
    return (False, None)

def get(sql_session, path, obj, workload=None):
    """Get the parameter from the job/jobset hierarchy.

    Arguments:
    sql_session -- SQLAlchemy session
    path -- Path object of the job/jobset
    obj -- database object to retrive
    workload -- workload to use

    Return the tuple (from_parents, obj) with from_parents to True if the
    parameter has been retrieved from the object itself or False if it is not
    defined in the object but has been retrieved from a parent (inherited)

    """
    return get_by_id(sql_session, path.id, obj, workload)

def get_list(sql_session, path, obj, order=None, workload=None):
    """Get parameters from the job/jobset hierarchy.

    Arguments:
    sql_session -- SQLAlchemy session
    path -- Path object of the job/jobset
    obj -- database object to retrive
    order -- column name to sort results
    workload -- workload to use

    Return the tuple (from_parents, obj_lst) with from_parents to True if the
    parameters have been retrieved from the object itself or False if they are
    not defined in the object but have been retrieved from a parent (inherited)

    """
    session = sql_session.open_session()
    # First, try to get the data from the ID of the object
    query = session.query(obj).filter_by(job_id=path.id[-1])
    if workload is not None:
        query = query.filter_by(workload_date=workload)
    if order is not None:
        query = query.order_by(order)
    res = query.all()
    if res:
        sql_session.close_session(session)
        return (True, res)
    # Otherwise, look in the parent hierarchy
    for i in range(len(path.id) - 2, -1, -1):
        query = session.query(obj).filter_by(job_id=path.id[i])
        if workload is not None:
            query = query.filter_by(workload_date=workload)
        if order is not None:
            query = query.order_by(order)
        res = query.all()
        if res:
            sql_session.close_session(session)
            return (False, res)
    sql_session.close_session(session)
    return (False, None)

