# Schedwi
# Copyright (C) 2011 Herve Quatremain
# 
# This file is part of Schedwi.
# 
# Schedwi is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
# 
# Schedwi is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.


"""Module to convert job/jobset path strings to job objects."""

from sqlalchemy.orm.exc import NoResultFound
import sqlalchemy.orm.session

from tables.job_main import job_main
from tables.job_main_s import job_main_s

def _sql_path_multi(session, h, id, res, only_dir=False, workload=None):
    if not h:
        # Check for duplicates
        for lst in res:
            for i in range(0, len(id)):
                if id[i] != lst[i]:
                    break
            else:
                return res
        res.append(id)
        return res
    job_name = h.pop(0)
    if not job_name or job_name == '.':
        return _sql_path_multi(session, h, id, res, only_dir, workload)
    if job_name == '..':
        if len(id) > 1:
            id.pop()
        return _sql_path_multi(session, h, id, res, only_dir, workload)
    # Glob to SQL
    jn = job_name.replace("*","%").replace("?","_")
    if workload is None:
        q = session.query(job_main.id).filter(job_main.name.like(jn))
    else:
        q = session.query(job_main_s.id).filter(job_main_s.name.like(jn))
        q = q.filter(job_main_s.workload_date == workload)
    q = q.filter_by(parent=id[-1])
    if only_dir:
        q = q.filter_by(type=0)
    for i, in q:
        myid = list(id)
        myid.append(i)
        try:
            _sql_path_multi(session, list(h), myid, res, only_dir, workload)
        except NoResultFound:
            pass
    return res

def _name2id(session, name_list, cwd=None, only_dir=False, workload=None):
    """Convert a job/jobset name to its list of IDs.

    Arguments:
    session -- SQLAlchemy session
    name_list -- list of job/jobset name parts
    cwd -- current working jobset
    only_dir -- whether to only return the IDs if it's a jobset
    workload -- workload ID to search

    """
    if cwd is None:
        ids = [ 1 ]
    else:
        ids = list(cwd.id)
    for job_name in name_list:
        if not job_name or job_name == '.':
            continue
        if job_name == '..':
            try:
                ids.pop()
            except:
                pass
            if not ids:
                ids = [ 1 ]
            continue
        # Glob to SQL
        name = job_name.replace("*","%").replace("?","_")
        if workload is None:
            q = session.query(job_main.id).filter(job_main.name.like(name))
        else:
            q = session.query(job_main_s.id).filter(job_main_s.name.like(name))
            q = q.filter_by(workload_date=workload)
        if only_dir:
            q = q.filter_by(type=0)
        res = q.filter_by(parent=ids[-1]).first()
        if res is None:
            raise NoResultFound()
        ids.append(res[0])
    return ids

def _id2name(session, ids, workload=None):
    """Convert a job/jobset ID to its name."""
    names = list()
    for i in ids:
        if i > 1:
            try:
                if workload is None:
                    q = session.query(job_main.name)
                else:
                    q = session.query(job_main_s.name)
                    q = q.filter_by(workload_date=workload)
                path, = q.filter_by(id=i).one()
            except NoResultFound:
                return None
            names.append(path)
    return '/' + '/'.join(names)


class Path(object):

    def __init__(self, sql_session, path=None, cwd=None, only_dir=False,
                 id=None, workload=None):
        """Build a new Path object.

        Arguments:
        sql_session -- SQLAlchemy session
        path -- job/jobset path (only if `id' is None)
        cwd -- current working jobset.  By default `/'
        only_dir -- whether to only consider jobsets
        id -- job ID list (from `/' to the job/jobset)
        workload -- workload to consider

        """
        self.workload = workload
        if not isinstance(sql_session, sqlalchemy.orm.session.Session):
            session = sql_session.open_session()
        else:
            session = sql_session
        if id is None:
            if cwd is None or path[0] == '/':
                self.id = _name2id(session, path.split('/'), None,
                                   only_dir, workload)
            else:
                self.id = _name2id(session, path.split('/'), cwd,
                                   only_dir, workload)
        else:
            self.id = id
        self.path = _id2name(session, self.id, workload)
        if not isinstance(sql_session, sqlalchemy.orm.session.Session):
            sql_session.close_session(session)

    def __str__(self):
        return self.path

    def __cmp__(self, other):
        return cmp(self.id[-1], other.id[-1])

    def __nonzero__(self):
        if self.path is not None and self.id is not None:
            return True
        return False

def get_paths(sql_session, path, cwd=None, only_dir=False, workload=None):
    """Return a list of jobs/jobsets matching the given path."""
    session = sql_session.open_session()
    jobs = list()
    if cwd is None or path[0] == '/':
        _sql_path_multi(session, path.split('/'), [1], jobs, only_dir,
                        workload)
    else:
        _sql_path_multi(session, path.split('/'), list(cwd.id), jobs, only_dir,
                        workload)
    sql_session.close_session(session)
    res = list()
    for job in jobs:
        res.append(Path(sql_session, id=job, workload=workload))
    return res

def id2path(sql_session, id, workload=None):
    """Convert a job/jobset ID to a Path object.

    Exceptions:
    sqlalchemy.orm.exc.NoResultFound -- the given id is not in the database
    sqlalchemy.orm.exc.MultipleResultsFound -- more than one id matches

    """
    if not isinstance(sql_session, sqlalchemy.orm.session.Session):
        session = sql_session.open_session()
    else:
        session = sql_session
    ids = [ id ]
    while id > 0:
        if workload is None:
            q = session.query(job_main.parent)
        else:
            q = session.query(job_main_s.parent)
            q = q.filter_by(workload_date=workload)
        parent_id, = q.filter_by(id=id).one()
        ids.append(parent_id)
        id = parent_id
    ids.reverse()
    p = Path(session, id=ids, workload=workload)
    if not isinstance(sql_session, sqlalchemy.orm.session.Session):
        sql_session.close_session(session)
    return p

def _split_path(path):
    try:
        pos = path.rindex("/")
    except ValueError:
        return ("./", path)
    return (path[:pos] if pos > 0 else '/', path[pos+1:])

def get_new_paths(sql_session, path, cwd=None):
    """Split the given path in its basename and jobset.

    Return the tuple (basename, list_jobset) with basename the job base name
    and list_jobset a list of Path objects.

    """
    base, name = _split_path(path)
    res = get_paths(sql_session, base, cwd)
    return (name, res)

