# Schedwi
# Copyright (C) 2011 Herve Quatremain
# 
# This file is part of Schedwi.
# 
# Schedwi is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
# 
# Schedwi is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.


"""Module to convert environment group names to environment group objects."""

import sqlalchemy.orm.session
from sqlalchemy.orm.exc import NoResultFound, MultipleResultsFound

from tables.environments import environments
from tables.environments_s import environments_s

def name2env_list(sql_session, name, workload=None):
    """Convert an environment group name to a list of environment group objects.

    Arguments:
    sql_session -- SQLAlchemy session
    name -- environment group name (can contain wildcard `*' and `?')
    workload -- workload to use

    """
    if not isinstance(sql_session, sqlalchemy.orm.session.Session):
        session = sql_session.open_session()
    else:
        session = sql_session
    e = name.replace("*","%").replace("?","_")
    if workload is None:
        q = session.query(environments).filter(environments.name.like(e))
        q = q.order_by(environments.name)
    else:
        q = session.query(environments_s).filter(environments_s.name.like(e))
        q = q.filter(environments_s.workload_date == workload)
        q = q.order_by(environments_s.name)
    env_list = q.all()
    if not isinstance(sql_session, sqlalchemy.orm.session.Session):
        sql_session.close_session(session)
    return env_list

def name2env(sql_session, name, workload=None):
    """Convert an environment group name to an environment group object.

    Arguments:
    sql_session -- SQLAlchemy session
    name -- environment group name (can contain wildcard `*' and `?')
    workload -- workload to use

    Exceptions:
    sqlalchemy.orm.exc.NoResultFound -- environment group not found
    sqlalchemy.orm.exc.MultipleResultsFound -- more than one environment group
                                               matches the given name

    """
    env_list = name2env_list(sql_session, name, workload)
    if not env_list:
        raise NoResultFound
    if len(env_list) > 1:
        raise MultipleResultsFound
    return env_list[0]

