# Schedwi
# Copyright (C) 2011 Herve Quatremain
# 
# This file is part of Schedwi.
# 
# Schedwi is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
# 
# Schedwi is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.


"""Module to list workloads and to change the current workload."""

import sys
import getopt
import time
import datetime

from sqlalchemy.orm.exc import NoResultFound
from sqlalchemy import desc

from tables.job_status import job_status
import status_utils
from help import print_trim

def usage():
    """Print a usage message on STDOUT."""
    print_trim("""Usage: wl [-h] [WORKLOAD]
    List all workloads, or, if WORKLOAD is given, enter the specified workload
    (all subsequent commands will apply to this workload objects - type `exit'
    to leave this mode).
    WORKLOAD can be:
      - a workload date YYYYMMDD
      - 0 or a negative number.  0 means today's workload, -1 means
        yesterdays, ...
      - the keywords `Today' and `Yesterday'

    Option:
      -h, --help  display this help
    """)

def parse(sql_session, w):
    """Parse the given workload string

    Returns the tuple (workload, error_message) with workload the parsed
    workload or None in case of error.  In this later case, error_message
    contains an error message.

    """
    try:
        delta = int(w)
    except:
        s = w.lower() 
        if s == "today":
            delta = 0
        elif s == "yesterday":
            delta = -1
        else:
            return (None, "`%s': wrong workload format" % w)
    if delta <= 0:
        workload_date = datetime.date.today() + datetime.timedelta(delta)
        workload = workload_date.strftime("%Y%m%d")
    else:
        #  Proper workload date has been provided (ie. 20101224)
        workload = str(delta)
    # Check that the workload exists
    session = sql_session.open_session()
    q = session.query(job_status).filter(job_status.job_id == 1)
    try:
        q = q.filter(job_status.workload_date == workload).one()
    except NoResultFound:
        sql_session.close_session(session)
        return (None, "`%s': no such workload" % w)
    sql_session.close_session(session)
    return (workload, "")

def _print_workload(js):
    """Print a workload."""
    print "%s - %s - %s" % (js.workload_date,
                        time.strftime("%a %x",
                                      time.strptime(str(js.workload_date),
                                                    "%Y%m%d")),
                        status_utils.status2string(js.status)) 

def wl(sql_session, arguments):
    """List workloads and change the current workload.

    Arguments:
    sql_session -- SQLAlchemy session
    arguments -- list of arguments given to the wl command.  Only the first
                 one is used and specifies the workload.  Is no argument is
                 provided, simply list all the available workloads.

    """
    try:
        optlist, args = getopt.getopt(arguments, 'h', ["help"])
    except getopt.GetoptError, err:
        sys.stderr.write("wl: " + str(err) + "\n")
        return None
    for o, a in optlist:
        if o in ("-h", "--help"):
            usage()
            return None
    if not args:
        # List current workloads
        session = sql_session.open_session()
        q = session.query(job_status).filter(job_status.job_id == 1)
        for js in q.order_by(desc(job_status.workload_date)):
            _print_workload(js)
        sql_session.close_session(session)
        return None
    workload, err_msg = parse(sql_session, args[0])
    if workload is None:
        sys.stderr.write("wl: " + err_msg + "\n")
        return None
    return workload

