# Schedwi
# Copyright (C) 2011 Herve Quatremain
# 
# This file is part of Schedwi.
# 
# Schedwi is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
# 
# Schedwi is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.


"""Module to list jobs and jobsets in a tree-like format."""

import sys
import getopt

from sqlalchemy.orm.exc import NoResultFound

import path
import status_utils
import colors
from tables.job_main import job_main
from tables.job_main_s import job_main_s
from tables.job_status import job_status
from help import print_trim

def usage():
    """Print a usage message on STDOUT."""
    print_trim("""Usage: tree [OPTION]... [JOB|JOBSET]...
List contents of jobsets in a tree-like format (the current jobset by default).

Options:
  -d             list jobsets only.
  -f             prints the full path prefix for each job.
  -F             append  a `/' for jobsets.
  -i             makes tree not print the indentation lines, useful when used
                 in conjunction with the -f option.
  --jobsetsfirst list jobsets before jobs.
  -L LEVEL       max display depth of the jobset tree.
  -n             do not display job/jobset current status.
  --noreport     omits printing of the job and jobset report at the end of the
                 tree listing.
  -t             sort the output by start time instead of alphabetically.
  -h, --help     display this help.

    """)

def _get_status(session, id, options, workload):
    """Get the status of a job/jobset.

    Arguments:
    session -- SQLAlchemy session
    id -- job/jobset ID
    options -- list of options given to the tree command
    workload -- workload to use

    Return the list (status, start_char, status_char, end_char)

    """
    if workload is None:
        return (0, '', '', '')
    space = ' ' * (1 - colors.CHAR_COLOR_LEN)
    query = session.query(job_status.status)
    query = query.filter(job_status.job_id == id)
    query = query.filter(job_status.workload_date == workload)
    try:
        s, = query.one()
    except NoResultFound:
        if options['show_status']:
            return (0, '', space * 3, '')
        else:
            return (0, '', '', '')

    if not options['show_status']:
        return (s, '', '', '')
    return (s, status_utils.status2color(s),
               space + status_utils.status2char(s) + space,
               colors.NORMAL)

def _print_job(session, job, id, last, prefix, counters, options, workload):
        if options['full_path']:
            name = path.Path(session, id=id, workload=workload)
        else:
            name = job.name
        if options['append_type'] and job.type == 0:
            type_char = '/'
        else:
            type_char = ''
        s, status_color, status_char, status_reset = _get_status(session,
                                                                 id[-1],
                                                                 options,
                                                                 workload)
        counters['status'][s] += 1
        if not options['indentation']:
            print "%s%s%s%s%s" % (status_color, status_char, status_reset,
                                   name, type_char)
        else:
            if last:
                print "%s`--%s%s%s%s%s" % (prefix,
                                   status_color, status_char, status_reset,
                                   name, type_char)
            else:
                print "%s|--%s%s%s%s%s" % (prefix,
                                   status_color, status_char, status_reset,
                                   name, type_char)

def _print_recur(session, id, prefix, level, counters, options, workload):
    if workload is None:
        query = session.query(job_main)
        query = query.filter(job_main.parent == id[-1])
        if options['only_dir']:
            query = query.filter(job_main.type == 0)
        if options['sort_by_jobsets']:
            query = query.order_by(job_main.type)
        if options['sort_by_time']:
            query = query.order_by(job_main.start_time)
        query = query.order_by(job_main.name)
    else:
        query = session.query(job_main_s)
        query = query.filter(job_main_s.parent == id[-1])
        query = query.filter(job_main_s.workload_date == workload)
        if options['only_dir']:
            query = query.filter(job_main_s.type == 0)
        if options['sort_by_jobsets']:
            query = query.order_by(job_main_s.type)
        if options['sort_by_time']:
            query = query.order_by(job_main_s.start_time)
        query = query.order_by(job_main_s.name)
    jobs = query.all()
    l = len(jobs)
    for i in range(l):
        job = jobs[i]
        if job.type == 0:
            counters["jobsets"] += 1
        else:
            counters["jobs"] += 1
        job_ids = id + [job.id]
        _print_job(session, job, job_ids, True if i+1 == l else False,
                   prefix, counters, options, workload) 
        if i + 1 == l:
            sub_prefix = prefix + '    '
        else:
            sub_prefix = prefix + '|   '
        if options['max_level'] == 0 or level < options['max_level']:
            _print_recur(session, job_ids, sub_prefix, level + 1, counters,
                         options, workload)

def tree(sql_session, current_cwd, arguments, workload=None):
    """List jobs and jobsets in a tree-like format.

    Arguments:
    sql_session -- SQLAlchemy session
    current_cwd -- current working jobset (a path.Path object)
    arguments -- list of arguments given to the tree command (list
                 of jobs/jobsets)
    workload -- workload to use

    """
    try:
        optlist, args = getopt.getopt(arguments, 'dfFintL:h',
                                      ["help", "noreport", "jobsetsfirst"])
    except getopt.GetoptError, err:
        sys.stderr.write("tree: " + str(err) + "\n")
        return 1
    options = { 'only_dir': False,
                'full_path': False,
                'print_report': True,
                'append_type': False,
                'max_level': 0,
                'sort_by_time': False,
                'sort_by_jobsets': False,
                'indentation': True,
                'show_status': True
              }
    for o, a in optlist:
        if o == "-d":
            options['only_dir'] = True
        elif o == "-f":
            options['full_path'] = True
        elif o == "-F":
            options['append_type'] = True
        elif o == "-i":
            options['indentation'] = False
        elif o == "-L":
            try:
                level = int(a)
            except ValueError:
                level = -1
            if level < 1:
                sys.stderr.write(
                            "tree: Invalid level, must be greater than 0.\n")
                return 1
            options['max_level'] = level
        elif o == "-n":
            options['show_status'] = False
        elif o == "-t":
            options['sort_by_time'] = True
        elif o == "--jobsetsfirst":
            options['sort_by_jobsets'] = True
        elif o == "--noreport":
            options['print_report'] = False
        elif o in ("-h", "--help"):
            usage()
            return 0
    paths = list()
    if args:
        for arg in args:
            p = path.get_paths(sql_session, arg, current_cwd, only_dir=True,
                               workload=workload)
            if not p:
                sys.stderr.write("tree: `%s': no such jobset\n" % arg)
            else:
                paths.extend(p)
    else:
        paths.append(current_cwd)
    if not paths:
        sys.stderr.write("tree: no such jobset\n")
        return 1
    session = sql_session.open_session()
    counters = { "jobsets": 0, "jobs": 0, "status": [0, 0, 0, 0, 0] }
    for p in paths:
        s, status_color, status_char, status_reset = _get_status(session,
                                                                 p.id[-1],
                                                                 options,
                                                                 workload)
        print "%s%s%s%s" % (status_color, status_char, status_reset, p)
        _print_recur(session, p.id, '', 1, counters, options, workload)
    if options['print_report']:
        if options['only_dir']:
            print "\n%d jobsets" % counters["jobsets"]
        else:
            print "\n%d jobsets, %d jobs" % (counters["jobsets"],
                                             counters["jobs"])
        if workload is not None:
            print "%4d %s%s%s" % (counters['status'][status_utils.WAITING],
                             status_utils.status2color(status_utils.WAITING),
                             status_utils.status2string(status_utils.WAITING),
                             colors.NORMAL)
            print "%4d %s%s%s" % (counters['status'][status_utils.RUNNING],
                             status_utils.status2color(status_utils.RUNNING),
                             status_utils.status2string(status_utils.RUNNING),
                             colors.NORMAL)
            print "%4d %s%s%s" % (counters['status'][status_utils.COMPLETED],
                             status_utils.status2color(status_utils.COMPLETED),
                             status_utils.status2string(status_utils.COMPLETED),
                             colors.NORMAL)
            print "%4d %s%s%s" % (counters['status'][status_utils.FAILED],
                             status_utils.status2color(status_utils.FAILED),
                             status_utils.status2string(status_utils.FAILED),
                            colors.NORMAL)
    sql_session.close_session(session)
    return 0

