# Schedwi
# Copyright (C) 2011 Herve Quatremain
# 
# This file is part of Schedwi.
# 
# Schedwi is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
# 
# Schedwi is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.


"""Module to set jobs or jobsets parameters in workloads."""

import getopt
import time
import sys

from sqlalchemy.orm.exc import NoResultFound, MultipleResultsFound

from tables.job_start_limit_s import job_start_limit_s
from tables.job_max_duration_s import job_max_duration_s
from tables.job_retries_s import job_retries_s
from tables.job_retries_interval_s import job_retries_interval_s

from tables.job_username_s import job_username_s
from tables.job_file_out_s import job_file_out_s
from tables.job_file_err_s import job_file_err_s
from tables.job_loadenv_s import job_loadenv_s
from tables.job_success_return_code_s import job_success_return_code_s
from tables.job_command_s import job_command_s
from tables.job_arguments_s import job_arguments_s
from tables.job_environment_s import job_environment_s
from tables.job_host_s import job_host_s
from tables.commands import commands
from tables.constraint_file_s import constraint_file_s
from tables.job_status import job_status
import cmd_wl.start
import host_utils
import environments_utils
import status_utils
import parse_config
import init

def usage():
    """Print a usage message on STDOUT."""
    print """Usage: set [OPTION]... JOB|JOBSET
Change a job or jobset parameter.

Options:
  --host=HOSTNAME[:PORT]  host on which the job/jobset must be running.
  --delhost               remove the hostname so this is the one from the
                          parent jobset which is going to be used instead.
  --start=TIME            start time of the job (HH:MM).
  --delstart              use the start time of the parent jobset.
  --command=COMMAND       command to run (if set to `-', no command will be
                          started and the job will complete immediately)
  --delcommand            no command.  Use the one defined in the
                          parent jobset.
  --parameter=PARAMETER   a parameter to pass to the command.  This option can
                          be specified several times to define more than one
                          parameter.
  --delparameter          use the parent jobset parameters instead.
  --noparameter           no parameter at all for the command (the ones from
                          the parent jobset are ignored).
  --return=RETURN_CODE    a return code from the command equal or below this
                          value means success.
  --delreturn             use the return code value defined at the parent
                          jobset level.
  --user=USERNAME         username under which the command must be run.
  --deluser               use the username defined in the parent
                          jobset instead.
  --stdout=FILE           command output file (stored on the Schedwi agent).
  --delstdout             use the file name defined in the parent
                          jobset instead.
  --stderr=FILE           command error file (stored on the Schedwi agent).
  --delstderr             use the file name defined in the parent
                          jobset instead.
  --duration=MINUTES      maximum number of minutes for the job/jobset to
                          complete once running.
  --delduration           use the value defined in the parent jobset instead.
  --retries=NUM           number of retries when the job/jobset fails.
  --delretries            use the value defined in the parent jobset instead.
  --retries-int=MINUTES   number of minutes between retries.
  --delretries-int        use the value defined in the parent jobset instead.
  --limit=MINUTES         number of minutes for the job to be started past its
                          start time (number of minutes or HH:MM)
  --dellimit              use the value defined in the parent jobset instead.

  --user-env=True|False   load (True) or not (False) the user account
                          environment (~/.profile or ~/.bashrc for instance) on
                          the remote host before running the job command.
  --deluser-env           use the value defined in the parent jobset instead.
  --addenv=ENV_NAME       add an environment group to the job/jobset.  This
                          option can be specified several times.
  --delenv=ENV_NAME       remove the provided environment group.
  --noenv                 do not use any environment group for this job/jobset.
  --addfile=HOSTNAME[:PORT],True|False,FILE     add a file to check before
                          starting the job/jobset.  The value is composed of
                          three fields separated by a comma.  The first one is
                          the remote agent hostname on which to check the file.
                          The second specifies if the file must be present
                          (True) or not (False).  The last one is the file or
                          directory to check.
  --delfile=HOSTNAME[:PORT],FILE    remove the provided file.
  --status=STATUS         change the current status. STATUS is
                            `running'   to start the job/jobset immediately.
                            `completed' to consider the job/jobset completed.
                            `failed'    to consider the job/jobset failed.
                            `waiting'   to reschedule the job now.
                          The force stop a running job/jobset rather use the
                          `stop' command.  To start a job, use the `start'
                          command.
  --description=TEXT      a description.
  -h, --help              display this help.
    """

def parse_args(argv):
    """Parse the argument list.

    Argument:
    argv --  argument list to parse

    Exception:
    getopt.getopterror -- option error (unknow option or missing parameter)
    
    """
    optlist, args = getopt.getopt(argv, "h",
            [
                "host=",
                "delhost",
                "start=",
                "delstart",
                "command=",
                "delcommand",
                "parameter=",
                "delparameter",
                "noparameter",
                "return=",
                "delreturn",
                "user=",
                "deluser",
                "stdout=",
                "delstdout",
                "stderr=",
                "delstderr",
                "duration=",
                "delduration",
                "retries=",
                "delretries",
                "retries-int=",
                "delretries-int",
                "limit=",
                "dellimit",
                "user-env=",
                "deluser-env",
                "addenv=",
                "delenv=",
                "noenv",
                "description=",
                "addfile=",
                "delfile=",
                "status=",
                "help"
            ])
    return (optlist, args)

def set(session, job, opts, workload):
    """Change jobs and jobsets parameters.

    Arguments:
    session -- SQLAlchemy session
    job -- job object to change
    opts -- option list (from getopt.getopt() - see parse_args())
    workload -- workload to use

    """
    workload = int(workload)
    arguments = list()
    constraint_files = list()
    envs = list()
    status_code = 0
    for o, a in opts:
        if o == "--host":
            try:
                host = host_utils.name2host(session, a)
            except NoResultFound:
                sys.stderr.write("No such host (--host=%s)\n" % a)
                return 1
            except MultipleResultsFound:
                sys.stderr.write("More than one host matches the provided name (--host=%s)\n" % a)
                return 1
            job.job_host = job_host_s(host.id, workload)

        elif o == "--delhost":
            del job.job_host

        elif o == "--start":
            for fmt in ("%H:%M", "%I:%M %p", "%Hh%M", "%Ih%M %p"):
                try:
                    tstr = time.strptime(a, fmt)
                except ValueError:
                    continue
                else:
                    break
            else:
                sys.stderr.write("Unknown time format (--start).  Should be `H:M' or `H:M AM/PM'\n")  
                return 1
            job.start_time = tstr.tm_hour * 100 + tstr.tm_min

        elif o == "--delstart":
            job.start_time = -1

        elif o == "--command":
            job.job_command = job_command_s(a, workload)

        elif o == "--delcommand":
            del job.job_command

        elif o == "--parameter":
            arguments.append(job_arguments_s(len(arguments) + 1, a, workload))

        elif o == "--delparameter":
             arguments = list()
             job.job_arguments = [ ]

        elif o == "--noparameter":
             arguments = list()
             job.job_arguments = [ job_arguments_s(-1, "", workload) ]

        elif o == "--return":
            try:
                v = int(a)
                if v < 0 or v > 255:
                    raise ValueError
            except:
                sys.stderr.write(
                    "Wrong value for --return.  Must be between 0 and 255\n")  
                return 1
            job.job_success_return_code = job_success_return_code_s(v, workload)

        elif o == "--delreturn":
            del job.job_success_return_code
        elif o == "--user":
            job.job_username = job_username_s(a, workload)
        elif o == "--deluser":
            del job.job_username
        elif o == "--stdout":
            job.job_file_out = job_file_out_s(a, workload)
        elif o == "--delstdout":
            del job.job_file_out
        elif o == "--stderr":
            job.job_file_err = job_file_err_s(a, workload)
        elif o == "--delstderr":
            del job.job_file_err
        elif o == "--duration":
            try:
                v = int(a)
                if v < 0:
                    raise ValueError
            except:
                sys.stderr.write("Wrong value for --duration.  Must be a number of minutes (0 means no limit)\n")  
                return 1
            job.job_max_duration = job_max_duration_s(v, workload)

        elif o == "--delduration":
            del job.job_max_duration

        elif o == "--retries":
            try:
                v = int(a)
                if v < 0:
                    raise ValueError
            except:
                sys.stderr.write("Wrong value for --retries\n")  
                return 1
            job.job_retries = job_retries_s(v, workload)

        elif o == "--delretries":
            del job.job_retries

        elif o == "--retries-int":
            try:
                v = int(a)
                if v < 0:
                    raise ValueError
            except:
                sys.stderr.write("Wrong value for --retries-int.  Must be the number of minutes between retries\n")
                return 1
            job.job_retries_interval = job_retries_interval_s(v, workload)

        elif o == "--delretries-int":
            del job.job_retries_interval

        elif o == "--limit":
            for fmt in ("%H:%M", "%Hh%M"):
                try:
                    tstr = time.strptime(a, fmt)
                except ValueError:
                    continue
                else:
                    v = tstr.tm_hour * 60 + tstr.tm_min
                    break
            else:
                try:
                    v = int(a)
                    if v < 0:
                        raise ValueError
                except:
                    sys.stderr.write("Wrong value for --limit.  Must be the number of minutes allowed for the job to complete (0 means no limit)\n")  
                    return 1
            job.job_start_limit = job_start_limit_s(v, workload)

        elif o == "--dellimit":
            del job.job_start_limit

        elif o == "--user-env":
            if a[0] in ('Y', 'y', '1', 't', 'T'):
                job.job_loadenv = job_loadenv_s(1, workload)
            else:
                job.job_loadenv = job_loadenv_s(0, workload)
        elif o == "--deluser-env":
            del job.job_loadenv

        elif o == "--addenv":
            try:
                env = environments_utils.name2env(session, a, workload)
            except NoResultFound:
                sys.stderr.write("No such environment %s (--addenv)\n" % a)
                return 1
            try:
                envs.remove(env.id)
            except:
                pass
            envs.append(env.id)

        elif o == "--delenv":
            try:
                env = environments_utils.name2env(session, a, workload)
            except NoResultFound:
                sys.stderr.write("No such environment %s (--delenv)\n" % a)
                return 1
            for e in job.job_environment:
                if env.id == e.env_id:
                    job.job_environment.remove(e)
                    break
            envs = filter(lambda id: id!=env.id, envs)

        elif o == "--noenv":
            envs = list()
            job.job_environment = [ job_environment_s(0, -1, workload) ]

        elif o == "--addfile":
            try:
                h, exist, filename = a.split(',', 2)
            except:
                sys.stderr.write("Wrong format for --addfile.  Must be <host>,<0|1>,<file>\n")
                return 1
            try:
                host = host_utils.name2host(session, h)
            except NoResultFound:
                sys.stderr.write("No such host %s (--addfile=%s)\n" % (h, a))
                return 1
            except MultipleResultsFound:
                sys.stderr.write("More than one host matches the provided name %s (--addfile=%s)\n" % (h, a))
                return 1
            try:
                constraint_files.remove(constraint_file_s(host.id, filename,
                                                          1, workload))
            except:
                pass
            if exist[0] in ('Y', 'y', '1', 't', 'T'):
                constraint_files.append(constraint_file_s(host.id, filename,
                                                          1, workload))
            else:
                constraint_files.append(constraint_file_s(host.id, filename,
                                                          0, workload)) 

        elif o == "--delfile":
            try:
                h, exist, filename = a.split(',', 2)
            except:
                try:
                    h, filename = a.split(',', 1)
                except:
                    sys.stderr.write(
                        "Wrong format for --delfile.  Must be <host>,<file>\n")
                    return 1
            try:
                host = host_utils.name2host(session, h)
            except NoResultFound:
                sys.stderr.write("No such host %s (--delfile=%s)\n" % (h, a))
                return 1
            except MultipleResultsFound:
                sys.stderr.write("More than one host matches the provided name %s (--delfile=%s)\n" % (h, a))
                return 1
            try:
                job.constraint_file.remove(constraint_file_s(host.id, filename,
                                                             1, workload))
            except:
                pass
            try:
                constraint_files.remove(constraint_file_s(host.id, filename,
                                                          1, workload))
            except:
                pass

        elif o == "--status":
            status_code = status_utils.string2status(a)
            if not status_code:
                sys.stderr.write("Unknown status %s (--status)\n" % a)
                return 1

        elif o == "--description":
            job.description = a

        elif o in ("-h", "--help"):
            usage()

    if arguments:
        job.job_arguments = arguments
    if constraint_files:
        for f in job.constraint_file:
            try:
                constraint_files.remove(f)
            except:
                pass
        job.constraint_file.extend(constraint_files)
    if envs:
        new_envs = list()
        last_position = 0
        for e in job.job_environment:
            if e.position < 0: 
                continue
            try:
                envs.index(e.env_id)
            except:
                pass
            else:
                continue
            if e.position > last_position:
                last_position = e.position
            new_envs.append(e)
        for env_id in envs: 
            last_position += 1
            new_envs.append(job_environment_s(env_id, last_position, workload))
        job.job_environment = new_envs
    if status_code:
        if status_code == status_utils.RUNNING:
            cmd_wl.start.start_id(session, job.id, workload)
        else:
            # Check if the same command is already in the database
            query = session.query(commands).filter(commands.job_id == job.id)
            query = query.filter(commands.workload_date == workload)
            query = query.filter(commands.command_status == 0)
            query = query.filter(commands.command == 1)
            query = query.filter(commands.parameter == status_code)
            if not query.count():
                # Now check the current status of the job
                query = session.query(job_status)
                query = query.filter(job_status.job_id == job.id)
                query = query.filter(job_status.workload_date == workload)
                query = query.filter(job_status.status == status_code)
                if not query.count():
                    c = commands(job.id, parse_config.DBI_USER, 0, 1,
                                 status_code, workload)
                    session.add(c)
                    print "It may take a minute for the new status to be applied"
                else:
                    print "This is already the current status"
    return 0

