# Schedwi
# Copyright (C) 2011 Herve Quatremain
# 
# This file is part of Schedwi.
# 
# Schedwi is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
# 
# Schedwi is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.


"""Module to create dependency links between jobs and jobsets."""

import sys
import getopt

import path
import status_utils
from tables.job_main import job_main
from tables.link import link
from help import print_trim

def usage():
    """Print a usage message on STDOUT."""
    print_trim("""Usage: ln [OPTION] JOB|JOBSET DEST...
Create a dependency link between JOB|JOBSET and DESTs.  If no option is given,
all the DEST jobs/jobsets must be completed for the JOB|JOBSET to start.

Options:
  -c, --completed  the DEST jobs/jobsets must be completed for JOB|JOBSET
                   to start.  This is the default.
  -f, --failed     the DEST jobs/jobsets must have failed for JOB|JOBSET
                   to start.
  -r, --running    the DEST jobs/jobsets must be running for JOB|JOBSET
                   to start.
  -w, --waiting    the DEST jobs/jobsets must be waiting for JOB|JOBSET
                   to start.
  -h, --help       display this help.

See also:
    `cat JOB|JOBSET' to print the dependency links.
    """)

def _detect_loop(session, src_id, dst_id):
    """Detect a dependency link loop."""
    for lnk in session.query(link).filter_by(job_id_source=dst_id):
        if lnk.job_id_destination == src_id:
            return True
        if _detect_loop(session, src_id, lnk.job_id_destination):
            return True
    return False

def ln(sql_session, current_cwd, arguments):
    """Add a link between job/jobset.

    Arguments:
    sql_session -- SQLAlchemy session
    current_cwd -- current working jobset (a path.Path object)
    arguments -- list of arguments given to the ln command (list
                 of jobs/jobsets)

    """
    try:
        optlist, args = getopt.getopt(arguments, "cfrwh",
                       ["completed", "failed", "running", "waiting", "help"])
    except getopt.GetoptError, err:
       sys.stderr.write("ln: " + str(err) + "\n")
       return 1
    st = status_utils.COMPLETED
    for o, a in optlist:
        if o in ("-c", "--completed"):
            st = status_utils.COMPLETED
        elif o in ("-f", "--failed"):
            st = status_utils.FAILED
        elif o in ("-r", "--running"):
            st = status_utils.RUNNING
        elif o in ("-w", "--waiting"):
            st = status_utils.WAITING
        elif o in ("-h", "--help"):
            usage()
            return 0
    paths = list()
    if args:
        for arg in args:
            p = path.get_paths(sql_session, arg, current_cwd)
            if not p:
                sys.stderr.write("ln: `%s': no such job or jobset\n" % arg)
                return 1
            paths.extend(p)
    else:
        sys.stderr.write("ln: missing job or jobset operand\n")
        return 1
    if not paths:
        sys.stderr.write("ln: no such job or jobset\n")
        return 1
    if len(paths) < 2:
        sys.stderr.write("ln: no enough operand or unknown job/jobset\n")
        return 1
    source = paths.pop(0)
    error = False
    session = sql_session.open_session()
    try:
        source_job = session.query(job_main).filter_by(id=source.id[-1]).one()
    except:
        sys.stderr.write("ln: no such source job or jobset\n")
        sql_session.cancel_session(session)
        return 1
    for p in paths:
        if filter(lambda i: i.job_id_destination == p.id[-1], source_job.links):
            sys.stderr.write(
                "ln: error: a link already exists between `%s' and `%s'\n" % \
                (source, p))
            error = True
            continue
        if _detect_loop(session, source.id[-1], p.id[-1]):
            sys.stderr.write(
                "ln: error: loop detected between `%s' and `%s'\n" % \
                (source, p))
            error = True
            continue
        # Check that the destination job exists
        try:
            src = session.query(job_main).filter_by(id=p.id[-1]).one()
        except:
            sys.stderr.write("ln: no such destination job or jobset\n")
            error = True
            continue
        source_job.links.append(link(p.id[-1], st))
    if error:
        sql_session.cancel_session(session)
        return 1
    sql_session.close_session(session)
    return 0

