# Schedwi
# Copyright (C) 2011 Herve Quatremain
# 
# This file is part of Schedwi.
# 
# Schedwi is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
# 
# Schedwi is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.


"""Module to change the current jobset."""

import sys
import getopt

from sqlalchemy.orm.exc import NoResultFound

import path
from help import print_trim

def usage():
    """Print a usage message on STDOUT."""
    print_trim("""Usage: cd [JOBSET]
        Change the current jobset to JOBSET.
        If JOBSET is not specified, change to the root jobset (`/').
        If JOBSET is `-', change to the previous working jobset.

        Option:
          -h, --help  display this help.
    """)

def cd(sql_session, current_cwd, previous_cwd, arguments, workload=None):
    """Change the current jobset.

    Arguments:
    sql_session -- SQLAlchemy session
    current_cwd -- current working jobset (a path.Path object)
    previous_cwd -- previous working jobset (a path.Path object)
    arguments -- list of arguments given to the cd command (list
                 of jobs/jobsets)
    workload -- workload to use

    """
    try:
        optlist, args = getopt.getopt(arguments, "h", ["help"])
    except getopt.GetoptError, err:
        sys.stderr.write("cd: " + str(err) + "\n")
        return None 
    for o, a in optlist:
        if o in ("-h", "--help"):
            usage()
            return None
    if not args:
        new_path = '/'
    else:
        new_path = args[0]
    if new_path == '-':
        if previous_cwd is not None:
            return previous_cwd;
        sys.stderr.write("cd: no previous jobset\n")
        return None
    try:
        new_cwd = path.Path(sql_session, new_path, current_cwd,
                            only_dir=True, workload=workload)
        if not new_cwd:
            sys.stderr.write("cd: no such jobset\n")
            return current_cwd
        return new_cwd
    except NoResultFound:
        sys.stderr.write("cd: `%s': no such jobset\n" % new_path)
        return None

