# Schedwi
# Copyright (C) 2011 Herve Quatremain
# 
# This file is part of Schedwi.
# 
# Schedwi is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
# 
# Schedwi is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.


"""Module to list jobs and jobsets that are using a calendar."""

import sys
import getopt

import path
import path_cal
from tables.calendars import calendars
from tables.calendars_s import calendars_s
from tables.job_main import job_main
from tables.job_main_s import job_main_s
from help import print_trim

def usage():
    """Print a usage message on STDOUT."""
    print_trim("""Usage: whatuses CAL|DIR...
        List the jobs and jobsets that use CAL or DIR.  For DIR, recursively
        list the jobs and jobsets.
        
        Option:
          -h, --help  display this help.
    """)

def is_used(session, cal, workload=None):
    """Return whether a calendar is used by a job or jobset."""
    if workload is None:
        q = session.query(job_main)
    else:
        q = session.query(job_main_s)
        q = q.filter(job_main_s.workload_date == workload)
    if q.filter_by(cal_id=cal.id).count():
        return True
    return False

_FIELD_WIDTH = 10

def _print_val(title, fmt=None, val=None):
    """Print a parameter value.

    Arguments:
    title -- parameter name
    fmt -- format string for the parameter value
    val -- value

    """
    if fmt is not None:
        if val is not None:
            print "%s :" % title.rjust(_FIELD_WIDTH), fmt % val
        else:
            print "%s :" % title.rjust(_FIELD_WIDTH), fmt
    else:
        print "%s :" % title.rjust(_FIELD_WIDTH)

def _print_whatuses(session, cal, p, only_if_used=False, workload=None):
    """Print a list of jobs/jobsets that are using the given calendar.
    
    Arguments:
    session -- SQLAlchemy session
    cal -- calendar object
    p -- calendar path
    only_if_used -- whether to print the calendar name if it is not in used
                    by any job or jobset.
    workload -- workload to use
    
    """
    first = True
    if workload is None:
        q = session.query(job_main).filter(job_main.cal_id == cal.id)
        q = q.order_by(job_main.parent)
    else:
        q = session.query(job_main_s).filter(job_main_s.cal_id == cal.id)
        q = q.filter(job_main_s.workload_date == workload)
        q = q.order_by(job_main_s.parent)
    for job in q:
        if first:
            print "\n%s:" % p
            first = False
        _print_val("Job" if job.type else "Jobset", "%s",
                   path.id2path(session, job.id, workload))
    if first and not only_if_used:
        print "\n%s:" % p

def _print_whatuses_dir(session, cal, p, workload=None):
    """Recursively print jobs and jobsets that use a calendar directory.

    Arguments:
    session -- SQLAlchemy session
    cal -- calendar directory object
    p -- calendar directory path
    workload -- workload to use

    """
    if workload is None:
        q = session.query(calendars).filter(calendars.parent == cal.id)
        q = q.order_by(calendars.name)
    else:
        q = session.query(calendars_s).filter(calendars_s.parent == cal.id)
        q = q.filter(calendars_s.workload_date == workload)
        q = q.order_by(calendars_s.name)
    for c in q:
        if c.entry_type == 0:
            _print_whatuses(session,c, path_cal.PathCal(session, id=c.id,
                                                        workload=workload),
                            True, workload=workload)
        else:
            _print_whatuses_dir(session, c, path_cal.PathCal(session, id=c.id,
                                                            workload=workload),
                                workload=workload)

def whatuses(sql_session, current_cwd, arguments, workload=None):
    """Show what is referencing a calendar.

    Arguments:
    sql_session -- SQLAlchemy session
    current_cwd -- current working directory (a path_cal.PathCal object)
    arguments -- list of arguments given to the whatuses command (list
                 of calendars)
    workload -- workload to use

    """
    try:
        optlist, args = getopt.getopt(arguments, "h", ["help"])
    except getopt.GetoptError, err:
        sys.stderr.write("whatuses: " + str(err) + "\n")
        return 1
    for o, a in optlist:
        if o in ("-h", "--help"):
            usage()
            return 0
    paths = list()
    if args:
        for arg in args:
            p = path_cal.get_paths(sql_session, arg, current_cwd,
                                   workload=workload)
            if not p:
                sys.stderr.write(
                      "whatuses: `%s': no such calendar or directory\n" % arg)
            else:
                paths.extend (p)
    else:
        sys.stderr.write("whatuses: missing operand\n")
        return 1
    if not paths:
        return 1
    session = sql_session.open_session()
    error = False
    for p in paths:
        if workload is None:
            q = session.query(calendars)
        else:
            q = session.query(calendars_s)
            q = q.filter(calendars_s.workload_date == workload)
        try:
            cal = q.filter_by(id=p.id[-1]).one()
        except:
            sys.stderr.write("whatuses: no such calendar or directory\n")
            error = True
            continue
        if cal.entry_type == 0:
            _print_whatuses(session, cal, p, workload=workload)
        else:
            _print_whatuses_dir(session, cal, p, workload=workload)
    sql_session.close_session(session)
    return 0 if not error else 1

