# Schedwi
# Copyright (C) 2011 Herve Quatremain
# 
# This file is part of Schedwi.
# 
# Schedwi is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
# 
# Schedwi is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.


"""Module to set calendars or calendar directories parameters."""

import getopt
import sys

import calcomp

def usage_options():
    """Print option description on STDOUT."""
    print """
Options:
  --formula=FORMULA       set the calendar formula.  See `help formula' for
                          more details on the syntax.
  --formula-add=FORMULA   add a formula at the end of the existing one.  See
                          `help formula' for more details on the syntax.
  --description=TEXT      a description.
  -h, --help              display this help.
    """

def usage():
    """Print a usage message on STDOUT."""
    print """Usage: set [OPTION]... CALENDAR|DIRECTORY
Change a calendar or directory parameter."""
    usage_options()

def parse_args(argv):
    """Parse the argument list.

    Argument:
    argv --  argument list to parse

    Exception:
    getopt.getopterror -- option error (unknow option or missing parameter)
    
    """
    optlist, args = getopt.getopt(argv, "h",
            [
                "formula=",
                "formula-add=",
                "description=",
                "help"
            ])
    return (optlist, args)

def print_calcomp_error(err, idx, f):
    """Print an error message given the calcomp error code."""
    if err == calcomp.CAL_EMPTYFIELD:
        sys.stderr.write("%s: syntax error: empty field\n" % f)
    elif err == calcomp.CAL_BADMONTHNAME:
        sys.stderr.write("%s: syntax error: invalid month name\n" % f)
    elif err == calcomp.CAL_BADDAYNAME:
        sys.stderr.write("%s: syntax error: invalid day name\n" % f)
    elif err == calcomp.CAL_MONTHOOR:
        sys.stderr.write("%s: syntax error: month number out of range\n" %f)
    elif err == calcomp.CAL_DAYOOR:
        sys.stderr.write("%s: syntax error: day number out of range\n" % f)
    elif err == calcomp.CAL_BADOPTION:
        sys.stderr.write("%s: syntax error: invalid option flag\n" % f)
    elif err != calcomp.CAL_NOERROR:
        sys.stderr.write("%s: syntax error\n" % f)
    if err != calcomp.CAL_NOERROR:
        sys.stderr.write("%s^\n" % (' ' * idx))

def set(session, cal, opts):
    """Change calendars and calendar groups parameters.

    Arguments:
    session -- SQLAlchemy session
    cal -- calendar object to change
    opts -- option list (from getopt.getopt() - see parse_args())

    """
    formulas = list()
    formulas_add = list()
    description = None
    for o, a in opts:
        if o == "--description":
            description = a
        elif o == "--formula":
            formulas.append(a)
        elif o == "--formula-add":
            formulas_add.append(a)
        elif o in ("-h", "--help"):
            usage()
            return 0
    error = False
    for f in formulas:
        if f.strip()[0] != '#':
            err, idx, c = calcomp.calcomp(f, 2010)
            print_calcomp_error(err, idx, f)
            if err != calcomp.CAL_NOERROR:
                error = True
    for f in formulas_add:
        if f.strip()[0] != '#':
            err, idx, c = calcomp.calcomp(f, 2010)
            print_calcomp_error(err, idx, f)
            if err != calcomp.CAL_NOERROR:
                error = True
    if not error:
        if description is not None:
            cal.description = description
        if formulas:
            cal.formula = '\n'.join(formulas)
        if formulas_add:
            cal.formula += '\n' + '\n'.join(formulas_add)
        return 0
    return 1

