# Schedwi
# Copyright (C) 2011 Herve Quatremain
# 
# This file is part of Schedwi.
# 
# Schedwi is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
# 
# Schedwi is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.


"""Module to print the parameters of calendars and calendar directories."""

import sys
import getopt

import path_cal
from tables.calendars import calendars
from tables.calendars_s import calendars_s
from help import print_trim

def usage():
    """Print a usage message on STDOUT."""
    print_trim("""Usage: cat [-h] [CAL|DIR]...
        Print calendar or calendar directory parameters.
        If CAL|DIR is not specified, print the parameters of the current
        calendar directory.

        Option:
          -h, --help  display this help. 

        See also:
            `show CAL'     to highlight the days corresponding to the formula
                           on a calendar.
            `help formula' for mode details on the formula syntax.
    """)

_FIELD_WIDTH = 13

def _print_val(title, fmt=None, val=None):
    """Print a parameter value.

    Arguments:
    title -- parameter name
    fmt -- format string for the parameter value
    val -- value

    """
    if fmt is not None:
        if val is not None:
            print "%s :" % title.rjust(_FIELD_WIDTH), fmt % val
        else:
            print "%s :" % title.rjust(_FIELD_WIDTH), fmt
            
    else:
        print "%s :" % title.rjust(_FIELD_WIDTH)

def cat(sql_session, current_cwd, arguments, workload=None):
    """Print calendar/directory parameters.

    Arguments:
    sql_session -- SQLAlchemy session
    current_cwd -- current working directory (a path_cal.PathCal object)
    arguments -- list of arguments given to the cat command (list of calendars)
    workload -- workload to use

    """
    try:
        optlist, args = getopt.getopt(arguments, "h", ["help"])
    except getopt.GetoptError, err:
        sys.stderr.write("cat: " + str(err) + "\n")
        return 1
    for o, a in optlist:
        if o in ("-h", "--help"):
            usage()
            return 0
    paths = list()
    if args:
        for arg in args:
            paths.extend (path_cal.get_paths(sql_session, arg, current_cwd,
                                             workload=workload))
    else:
        paths.append(current_cwd)
    if not paths:
        sys.stderr.write("cat: no such calendar or directory\n")
        return 1
    first = True
    error = False
    for p in paths:
        if first == False:
            sys.stdout.write("\n")
        else:
            first = False

        session = sql_session.open_session()
        if workload is None:
            q = session.query(calendars)
        else:
            q = session.query(calendars_s)
            q = q.filter(calendars_s.workload_date == workload)
        try:
            cal = q.filter_by(id=p.id[-1]).one()
        except:
            sys.stderr.write("cat: no such calendar or directory\n")
            error = True
            continue
        sql_session.close_session(session)

        _print_val("Name", "%s", cal.name if cal.id > 1 else '/')
        _print_val("Type", "%s", "Directory" if cal.entry_type else "Calendar")
        if not cal.entry_type:
            _print_val("Formula", "%s", cal.formula)
        _print_val("Description","%s", cal.description)
    return 0 if not error else 1

