/* Schedwi
   Copyright (C) 2007-2010 Herve Quatremain

   This file is part of Schedwi.

   Schedwi is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 3 of the License, or
   (at your option) any later version.

   Schedwi is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

/* net_module_runjob.c -- Module to start a job */

#include <schedwi.h>

#if STDC_HEADERS
#include <stdlib.h>
#include <string.h>
#endif

#if HAVE_CTYPE_H
#include <ctype.h>
#endif

#if HAVE_UNISTD_H
#include <unistd.h>
#endif

#if HAVE_ASSERT_H
#include <assert.h>
#endif

#include <lib_functions.h>
#include <job_parameters.h>
#include <lwc_log.h>
#include <child_mgnt.h>
#include <job_launcher.h>
#include <net_module_runjob.h>
#include <JSON_parser.h>

enum steps {
	S_START,
	S_OUTER_ARRAY,
	S_OUTER_ARRAY_MODULE,
	S_DATA,
	S_KEY_JOBID,
	S_KEY_COMMAND,
	S_KEY_USER,
	S_KEY_STDOUT,
	S_KEY_STDERR,
	S_KEY_ARGS,
	S_ARG_ARRAY_START,
	S_KEY_USERENV,
	S_KEY_ENV,
	S_ENV_ARRAY_START,
	S_END
};

struct json_cb {
	enum steps step;
	char error;
	char *jobid;
	environment_t env;
	job_parameters_t job;
};


/*
 * JSON callback function
 */
static int
json_char_callback (void* ctx, int type, const JSON_value* value)
{
	struct json_cb *s = (struct json_cb *)ctx;
	char *v;

	switch (type) {
		case JSON_T_ARRAY_BEGIN:
			if (s->step == S_START) {
				s->step = S_OUTER_ARRAY;
			}
			else if (s->step == S_KEY_ARGS) {
				s->step = S_ARG_ARRAY_START;
			}
			else if (s->step == S_KEY_ENV) {
				s->step = S_ENV_ARRAY_START;
			}
			break;
		case JSON_T_STRING:
			if (s->step == S_OUTER_ARRAY) {
				s->step = S_OUTER_ARRAY_MODULE;
			}
			else if (s->step == S_KEY_JOBID) {
				v = (char *)malloc (value->vu.str.length + 1);
				if (v == NULL) {
					s->error = 1;
					return 1;
				}
				strncpy (v, value->vu.str.value,
						value->vu.str.length);
				v[value->vu.str.length] = '\0';
				if (s->jobid != NULL) {
					free (s->jobid);
				}
				s->jobid = v;
				s->step = S_DATA;
			}
			else if (s->step == S_KEY_COMMAND) {
				if (add_command_job_parameters (&(s->job),
						value->vu.str.value,
						value->vu.str.length) != 0)
				{
					s->error = 1;
					return 1;
				}
				s->step = S_DATA;
			}
			else if (s->step == S_KEY_USER) {
				if (add_username_job_parameters (&(s->job),
						value->vu.str.value,
						value->vu.str.length) != 0)
				{
					s->error = 1;
					return 1;
				}
				s->step = S_DATA;
			}
			else if (s->step == S_KEY_STDOUT) {
				if (add_stdout_job_parameters (&(s->job),
						value->vu.str.value,
						value->vu.str.length) != 0)
				{
					s->error = 1;
					return 1;
				}
				s->step = S_DATA;
			}
			else if (s->step == S_KEY_STDERR) {
				if (add_stderr_job_parameters (&(s->job),
						value->vu.str.value,
						value->vu.str.length) != 0)
				{
					s->error = 1;
					return 1;
				}
				s->step = S_DATA;
			}
			else if (s->step == S_ARG_ARRAY_START) {
				if (add_arg_job_parameters (&(s->job),
						value->vu.str.value,
						value->vu.str.length) != 0)
				{
					s->error = 1;
					return 1;
				}
			}
			else if (s->step == S_ENV_ARRAY_START) {
				v = strchr (value->vu.str.value, '=');
				if (	   v != NULL
					&& add_env_s (&(s->env),
						value->vu.str.value,
						v - value->vu.str.value,
						v + 1,
					value->vu.str.length
					- (v + 1 - value->vu.str.value)) != 0)
				{
					s->error = 1;
					return 1;
				}
			}
			break;
		case JSON_T_TRUE:
			if (s->step == S_KEY_USERENV) {
				add_loadenv_job_parameters (&(s->job), 1);
				s->step = S_DATA;
			}
			break;
		case JSON_T_FALSE:
			if (s->step == S_KEY_USERENV) {
				add_loadenv_job_parameters (&(s->job), 0);
				s->step = S_DATA;
			}
			break;
		case JSON_T_OBJECT_BEGIN:
			if (s->step == S_OUTER_ARRAY_MODULE) {
				s->step = S_DATA;
			}
			break;
		case JSON_T_KEY:
			if (s->step == S_DATA) {
				if (schedwi_strcasecmp (value->vu.str.value,
							"job id") == 0)
				{
					s->step = S_KEY_JOBID;
				}
				else
				if (schedwi_strcasecmp (value->vu.str.value,
							"command") == 0)
				{
					s->step = S_KEY_COMMAND;
				}
				else
				if (schedwi_strcasecmp (value->vu.str.value,
							"user") == 0)
				{
					s->step = S_KEY_USER;
				}
				else
				if (schedwi_strcasecmp (value->vu.str.value,
							"stdout") == 0)
				{
					s->step = S_KEY_STDOUT;
				}
				else
				if (schedwi_strcasecmp (value->vu.str.value,
							"stderr") == 0)
				{
					s->step = S_KEY_STDERR;
				}
				else
				if (schedwi_strcasecmp (value->vu.str.value,
							"user env") == 0)
				{
					s->step = S_KEY_USERENV;
				}
				else
				if (schedwi_strcasecmp (value->vu.str.value,
							"args") == 0)
				{
					s->step = S_KEY_ARGS;
				}
				else
				if (schedwi_strcasecmp (value->vu.str.value,
							"env") == 0)
				{
					s->step = S_KEY_ENV;
				}
			}
			break;
		case JSON_T_ARRAY_END:
			if (	   s->step == S_ARG_ARRAY_START
				|| s->step == S_ENV_ARRAY_START)
			{
				s->step = S_DATA;
			}
			break;
		case JSON_T_OBJECT_END:
			if (s->step == S_DATA) {
				s->step = S_END;
			}
			break;
	}

	return 1;
}


#if CYGWIN_IN_USE
/*
 * For Cygwin, some environment variables must be set to fork/exec the
 * command job.  This function get the required variables from the current
 * environment and copy them to the environment of the new job
 *
 * Return:
 *   0 --> No error
 *  -1 --> Memory allocation error
 */
static int
cygwin_required_env (job_parameters_t *job)
{
	char *e;

	if (job == NULL) {
		return 0;
	}

	e = getenv ("PATH");
	if (e != NULL && add_env_job_parameters (job, "PATH", e) != 0) {
		return -1;
	}
	return 0;
}
#endif


/*
 * Parse the server request (in buffer) and run the specified job
 *
 * Return:
 *   0 --> No error
 *  -1 --> Memory allocation error (lwc_writeLog() is used to display an error
 *         message)
 *  -2 --> Syntax error in the request from the server (lwc_writeLog() is used
 *         to display an error message)
 *  -3 --> Error starting the job (lwc_writeLog() is used to display an error
 *         message)
 */
int
run_job (net_id *sock, char *buffer, size_t size)
{
	char *err_msg, *s;
	int ret, i;
        JSON_config config;
        struct JSON_parser_struct *jc;
	struct json_cb cb_data;
#if HAVE_PID_T
	pid_t child;
#else
	int child;
#endif


#if HAVE_ASSERT_H
	assert (sock != NULL && buffer != NULL);
#endif

	/* JSON parser initialization */
	init_JSON_config (&config);

	init_environment (&(cb_data.env));
	init_job_parameters (&(cb_data.job));
	cb_data.step                  = S_START;
	cb_data.error                 = 0;
	cb_data.jobid                 = NULL;
	config.depth                  = 20;
	config.callback               = &json_char_callback;
	config.allow_comments         = 0;
	config.handle_floats_manually = 0;
	config.callback_ctx           = &cb_data;

	jc = new_JSON_parser (&config);

	/* Parse the JSON string */
	for (i = 0; i < size && cb_data.step != S_END; i++) {
		/*
		 * No need to check the return code as the JSON string has
		 * already been parsed once in net_parse.c
		 */
		JSON_parser_char (jc, buffer[i]);
		if (cb_data.error != 0) {
			delete_JSON_parser (jc);
			destroy_environment (&(cb_data.env));
			destroy_job_parameters (&(cb_data.job));
			if (cb_data.jobid != NULL) {
				free (cb_data.jobid);
			}
			s = "{ \"success\" : false, \"data\" : \"";
			net_write (sock, s, schedwi_strlen (s));
			s = _("Memory allocation error");
			net_write (sock, s, schedwi_strlen (s));
			lwc_writeLog (LOG_CRIT, s);
			s = "\" }";
			net_write (sock, s, schedwi_strlen (s));
			return -1;
		}
	}
	delete_JSON_parser (jc);

	/* Check that the job_id (JID) has been provided */
	if (cb_data.jobid == NULL) {
		destroy_environment (&(cb_data.env));
		destroy_job_parameters (&(cb_data.job));
		s = "{ \"success\" : false, \"data\" : \"";
		net_write (sock, s, schedwi_strlen (s));
		s = _("Syntax error: job ID not provided");
		net_write (sock, s, schedwi_strlen (s));
		lwc_writeLog (LOG_ERR, s);
		s = "\" }";
		net_write (sock, s, schedwi_strlen (s));
		return -2;
	}

	/*
	 * For Cygwin, some environment variables are required to fork/exec
	 * the job command
	 */
#if CYGWIN_IN_USE
	if (cygwin_required_env (&(cb_data.job)) != 0) {
		destroy_environment (&(cb_data.env));
		destroy_job_parameters (&(cb_data.job));
		free (cb_data.jobid);
		s = "{ \"success\" : false, \"data\" : \"";
		net_write (sock, s, schedwi_strlen (s));
		s = _("Memory allocation error");
		net_write (sock, s, schedwi_strlen (s));
		lwc_writeLog (LOG_CRIT, s);
		s = "\" }";
		net_write (sock, s, schedwi_strlen (s));
		return -1;
	}
#endif

	/* Build the environment. SCHEDWI_JOBID must be first */
	if (	   add_env_job_parameters (	&(cb_data.job),
					"SCHEDWI_JOBID", cb_data.jobid) != 0
		|| concat_env_job_parameters (	&(cb_data.job),
						&(cb_data.env)) != 0)
	{
		destroy_environment (&(cb_data.env));
		destroy_job_parameters (&(cb_data.job));
		free (cb_data.jobid);
		s = "{ \"success\" : false, \"data\" : \"";
		net_write (sock, s, schedwi_strlen (s));
		s = _("Memory allocation error");
		net_write (sock, s, schedwi_strlen (s));
		lwc_writeLog (LOG_CRIT, s);
		s = "\" }";
		net_write (sock, s, schedwi_strlen (s));
		return -1;
	}

	destroy_environment (&(cb_data.env));

	/* Start the job */
	ret = submit_job (&(cb_data.job), cb_data.jobid, &child, &err_msg);
	destroy_job_parameters (&(cb_data.job));
	if (ret != 0) {
		s = "{ \"success\" : false, \"data\" : \"";
		net_write (sock, s, schedwi_strlen (s));
		if (err_msg != NULL) {
			net_write (sock, err_msg, schedwi_strlen (err_msg));
			lwc_writeLog (LOG_ERR, err_msg);
		}
		else {
			s = _("Fail to start the job");
			net_write (sock, s, schedwi_strlen (s));
			lwc_writeLog (LOG_ERR,
				_("Job %s: Fail to start"), cb_data.jobid);
		}
		free (cb_data.jobid);
		s = "\" }";
		net_write (sock, s, schedwi_strlen (s));
		return (ret == -2) ? -3: ret;
	}
	else {
		lwc_writeLog (	LOG_INFO, _("Job %s: Started (PID %d)"),
				cb_data.jobid, child);
	}

	/* Add the job to the list of the running jobs */
	if (add_child (child, cb_data.jobid) != 0) {
		lwc_writeLog (LOG_CRIT, _("Memory allocation error"));
	}
	free (cb_data.jobid);

	s = "{ \"success\" : true }";
	net_write (sock, s, schedwi_strlen (s));

	return 0;
}

/*-----------------============== End Of File ==============-----------------*/
