/* Schedwi
   Copyright (C) 2011 Herve Quatremain

   This file is part of Schedwi.

   Schedwi is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 3 of the License, or
   (at your option) any later version.

   Schedwi is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

/* cert_client.c -- Check agent key and certificate or register */

#include <schedwi.h>

#if STDC_HEADERS
#include <stdlib.h>
#include <string.h>
#else
#if HAVE_STDLIB_H
#include <stdlib.h>
#endif
#if HAVE_STRING_H
#include <string.h>
#endif
#endif

#if HAVE_SYS_STAT_H
#include <sys/stat.h>
#endif

#if HAVE_UNISTD_H
#include <unistd.h>
#endif

#if HAVE_STDIO_H
#include <stdio.h>
#endif

#if HAVE_TIME_H
#include <time.h>
#endif

#if HAVE_ERRNO_H
#include <errno.h>
#endif
#ifndef errno
extern int errno;
#endif

#if HAVE_ASSERT_H
#include <assert.h>
#endif

#include <lib_functions.h>
#include <lwc_log.h>
#include <utils.h>
#include <cert_utils.h>
#include <reg_getmycrt_clnt.h>
#include <reg_whoami_clnt.h>
#include <reg_getca_clnt.h>
#include <reg_register_clnt.h>
#include <net_utils_sock.h>
#include <cert_client.h>


/*
 * Open a socket connected to the schedwireg server
 *
 * Return:
 *   The socket (to close with net_close_sock ())
 */
static int
get_socket_retries (const char *registrar_server, const char *registrar_port)
{
	int sock, count;
#if HAVE_NANOSLEEP
	struct timespec req;

	req.tv_sec = 10;
	req.tv_nsec = 0;
#endif

	count = 0;
	while (1) {
		sock = net_client_sock (registrar_port, registrar_server);
		if (sock >= 0) {
			return sock;
		}
		if (count == 0) {
			lwc_writeLog (	LOG_INFO,
					_("I'll recheck a little bit latter"));
		}
		else
		if (count % 60 == 0) {
			lwc_writeLog (	LOG_INFO,
			_("Server still not available.  I'll recheck latter"));
		}
		count++;
#if HAVE_NANOSLEEP
		nanosleep (&req, NULL);
#elif HAVE_SLEEP
		sleep (10);
#elif HAVE_USLEEP
		usleep (10 * 1000000);
#endif
	}
}


#if HAVE_LIBGNUTLS

/*
 * Free the NULL-terminated array of hostnames.
 */
static void
free_host_names (char **lst)
{
	int i;

	if (lst == NULL) {
		return;
	}
	for (i = 0; lst[i] != NULL; i++) {
		free (lst[i]);
		lst[i] = NULL;
	}
	free (lst);
}


/*
 * Get the certificate from the server (schedwireg) and store it in
 * the provided `file_cert' file.
 *
 * Return:
 *   0 --> No error
 *  -1 --> Error (a message has been logged)
 *   1 --> The server doesn't have a CSR for this agent.  A CSR must be sent
 *         first
 */
static int
wait_for_cert (	const char *registrar_server, const char *registrar_port,
		const char *local_port, const char *file_cert)
{
	int sock, count, ret;
	char *s, *crt;
	error_reason_t reason;
	size_t size;
	FILE *file;
#if HAVE_NANOSLEEP
	struct timespec req;

	req.tv_sec = 10;
	req.tv_nsec = 0;
#endif


#if HAVE_ASSERT_H
	assert (file_cert != NULL);
#endif

	count = 0;
	while (1) {
		sock =  get_socket_retries (registrar_server, registrar_port);

		s = NULL;
		ret = getmycrt (sock, local_port, &s, &reason);
		net_close_sock (sock);
		if (ret < 0) {
			return -1;
		}
		if (ret == 0) {
			crt = string2PEM (s);
			free (s);
			if (crt == NULL) {
				lwc_writeLog (	LOG_CRIT,
						_("Memory allocation error"));
				return -1;
			}
			file = fopen (file_cert, "wb");
			if (file == NULL) {
				lwc_writeLog (	LOG_ERR, _("%s: %s"),
						file_cert, strerror (errno));
				free (crt);
				return -1;
			}
			size = schedwi_strlen (crt);
			if (fwrite (crt, 1, size, file) != size) {
				lwc_writeLog (	LOG_ERR,
						_("%s: write error: %s"),
						file_cert, strerror (errno));
				fclose (file);
				my_unlink (file_cert);
				free (crt);
				return -1;
			}
			free (crt);
			if (fclose(file) != 0) {
				my_unlink (file_cert);
				lwc_writeLog (	LOG_ERR, _("%s: %s"),
						file_cert, strerror (errno));
				return -1;
			}
			return 0;
		}
		/* Server side error */
		switch (reason) {
		  case REG_NOT_READY_YET:
			if (count == 0) {
				lwc_writeLog (	LOG_INFO,
				_("Certificate not yet ready on the server"));
				lwc_writeLog (	LOG_INFO,
						_("I'll retry latter"));
			}
			else
			if (count % 60 == 0) {
				lwc_writeLog (	LOG_INFO,
		_("Certificate still not ready.  I'll retry latter"));
			}
			break;

		  case REG_NO_CSR:
			if (s != NULL) {
				free (s);
			}
			return 1;
			break;

		  default:
			if (count == 0) {
				if (s == NULL) {
					lwc_writeLog (	LOG_INFO,
		_("Cannot get the certificate: server-side error"));
				}
				else {
					lwc_writeLog (	LOG_INFO,
		_("Cannot get the certificate: server-side error: %s"),
							s);
				}
				lwc_writeLog (LOG_INFO, _("I'll retry latter"));
			}
			else
			if (count % 60 == 0) {
				lwc_writeLog (	LOG_INFO,
		_("Still get errors from the server.  I'll retry latter"));
			}
			break;
		}
		count++;
		if (s != NULL) {
			free (s);
		}
#if HAVE_NANOSLEEP
		nanosleep (&req, NULL);
#elif HAVE_SLEEP
		sleep (10);
#elif HAVE_USLEEP
		usleep (10 * 1000000);
#endif
	}
}


/*
 * Get the CA certificate from the server (schedwireg) and store it in
 * the provided `file_ca_cert' file.
 *
 * Return:
 *   0 --> No error
 *  -1 --> Error (a message has been logged)
 */
static int
get_ca (const char *registrar_server, const char *registrar_port,
	const char *file_ca_cert)
{
	int sock, count, ret;
	char *err_msg;
#if HAVE_NANOSLEEP
	struct timespec req;

	req.tv_sec = 10;
	req.tv_nsec = 0;
#endif


#if HAVE_ASSERT_H
	assert (file_ca_cert != NULL);
#endif

	count = 0;
	while (1) {
		sock =  get_socket_retries (registrar_server, registrar_port);

		err_msg = NULL;
		ret = getCA (sock, file_ca_cert, &err_msg);
		net_close_sock (sock);
		if (ret < 0) {
			return -1;
		}
		if (ret == 0) {
			return 0;
		}
		/* Server side error */
		if (count == 0) {
			if (err_msg == NULL) {
				lwc_writeLog (	LOG_INFO,
		_("Cannot get the CA certificate: server-side error"));
			}
			else {
				lwc_writeLog (	LOG_INFO,
		_("Cannot get the CA certificate: server-side error: %s"),
						err_msg);
				free (err_msg);
			}
			lwc_writeLog (LOG_INFO, _("I'll retry latter"));
		}
		else
		if (count % 60 == 0) {
			if (err_msg != NULL) {
				free (err_msg);
			}
			lwc_writeLog (	LOG_INFO,
		_("Still get errors from the server.  I'll retry latter"));
		}
		count++;
#if HAVE_NANOSLEEP
		nanosleep (&req, NULL);
#elif HAVE_SLEEP
		sleep (10);
#elif HAVE_USLEEP
		usleep (10 * 1000000);
#endif
	}
}


/*
 * Generate the certificate signig request (CSR) and send it for validation
 * to the server (schedwireg)
 *
 * Return:
 *   0 --> No error
 *  -1 --> Error (a message has been logged)
 *   1 --> A CSR or certificate for this agent is already on the server.
 */
static int
generate_and_send_csr (	const char *file_key,
			const char *registrar_server,
			const char *registrar_port,
			const char *local_port)
{
	char buffer[4 * 1024];
	size_t size;
	gnutls_x509_privkey_t key;
	gnutls_x509_crq_t rq;
	int sock, count, ret, names_size;
	char **names, *err_msg, *ip;
	error_reason_t reason;
#if HAVE_NANOSLEEP
	struct timespec req;

	req.tv_sec = 10;
	req.tv_nsec = 0;
#endif


#if HAVE_ASSERT_H
	assert (file_key != NULL);
#endif

	/*
	 * Get the names of the current agent as seen from the server side.
	 * These names will be used in the certificate signing request.
	 */
	count = 0;
	while (1) {
		sock =  get_socket_retries (registrar_server, registrar_port);

		names = NULL;
		err_msg = NULL;
		names_size = 0;
		ret = whoami (sock, &names, &names_size, &err_msg, NULL);
		net_close_sock (sock);
		if (ret < 0) {
			return -1;
		}
		if (ret == 0) {
			break;
		}
		/* Server side error */
		if (count == 0) {
			if (err_msg == NULL) {
				lwc_writeLog (	LOG_INFO,
		_("Cannot get the agent DNS names: server-side error"));
			}
			else {
				lwc_writeLog (	LOG_INFO,
		_("Cannot get the agent DNS names: server-side error: %s"),
						err_msg);
				free (err_msg);
			}
			lwc_writeLog (LOG_INFO, _("I'll retry latter"));
		}
		else
		if (count % 60 == 0) {
			if (err_msg != NULL) {
				free (err_msg);
			}
			lwc_writeLog (	LOG_INFO,
		_("Still get errors from the server.  I'll retry latter"));
		}
		count++;
#if HAVE_NANOSLEEP
		nanosleep (&req, NULL);
#elif HAVE_SLEEP
		sleep (10);
#elif HAVE_USLEEP
		usleep (10 * 1000000);
#endif
	}

	/* Generate the CSR */
	key = load_private_key (file_key);
	if (key == NULL) {
		free_host_names (names);
		return -1;
	}
	ip = names[names_size - 1];
	names[names_size - 1] = NULL;
	rq = generate_request (key, names, ip);
	free (ip);
	free_host_names (names);
	if (rq == NULL) {
		gnutls_x509_privkey_deinit (key);
		return -1;
	}

	size = sizeof (buffer);
	ret = gnutls_x509_crq_export (rq, GNUTLS_X509_FMT_PEM, buffer, &size);
	if (ret < 0) {
		lwc_writeLog (	LOG_ERR,
				_("SSL: Failed to export the request: %s"),
				gnutls_strerror (ret));
		gnutls_x509_crq_deinit (rq);
		gnutls_x509_privkey_deinit (key);
		return -1;
	}
	gnutls_x509_crq_deinit (rq);
	gnutls_x509_privkey_deinit (key);
	buffer[size] = '\0';

	/* Send the CSR to the server */
	count = 0;
	while (1) {
		sock =  get_socket_retries (registrar_server, registrar_port);

		err_msg = NULL;
		ret = send_request (	sock, local_port, 1, buffer,
					&err_msg, &reason);
		net_close_sock (sock);
		if (ret < 0) {
			return -1;
		}
		if (ret == 0) {
			return 0;
		}
		/* Server side error */
		if (reason != REG_SYSTEM_ERROR) {
			if (err_msg != NULL) {
				free (err_msg);
			}
			return 1;
		}
		if (count == 0) {
			if (err_msg == NULL) {
				lwc_writeLog (	LOG_INFO,
		_("Cannot register the agent: server-side error"));
			}
			else {
				lwc_writeLog (	LOG_INFO,
		_("Cannot register the agent: server-side error: %s"),
						err_msg);
				free (err_msg);
			}
			lwc_writeLog (LOG_INFO, _("I'll retry latter"));
		}
		else
		if (count % 60 == 0) {
			if (err_msg != NULL) {
				free (err_msg);
			}
			lwc_writeLog (	LOG_INFO,
		_("Still get errors from the server.  I'll retry latter"));
		}
		count++;
#if HAVE_NANOSLEEP
		nanosleep (&req, NULL);
#elif HAVE_SLEEP
		sleep (10);
#elif HAVE_USLEEP
		usleep (10 * 1000000);
#endif
	}
	return 0;
}

#endif /* HAVE_LIBGNUTLS  */


/*
 * Register with the server (schedwireg)
 *
 * Return:
 *   0 --> No error
 *  -1 --> Error (a message has been logged)
 */
static int
register_no_ssl (	const char *registrar_server,
			const char *registrar_port,
			const char *local_port)
{
	int sock, count, ret;
	char *err_msg;
	error_reason_t reason;
#if HAVE_NANOSLEEP
	struct timespec req;

	req.tv_sec = 10;
	req.tv_nsec = 0;
#endif

	count = 0;
	while (1) {
		sock =  get_socket_retries (registrar_server, registrar_port);

		err_msg = NULL;
		ret = send_request (	sock, local_port, 0, NULL,
					&err_msg, &reason);
		net_close_sock (sock);
		if (ret < 0) {
			return -1;
		}
		if (ret == 0) {
			return 0;
		}
		/* Server side error */
		if (reason != REG_SYSTEM_ERROR) {
			if (err_msg != NULL) {
				free (err_msg);
			}
			return 0;
		}
		if (count == 0) {
			if (err_msg == NULL) {
				lwc_writeLog (	LOG_INFO,
		_("Cannot register the agent: server-side error"));
			}
			else {
				lwc_writeLog (	LOG_INFO,
		_("Cannot register the agent: server-side error: %s"),
						err_msg);
				free (err_msg);
			}
			lwc_writeLog (LOG_INFO, _("I'll retry latter"));
		}
		else
		if (count % 60 == 0) {
			if (err_msg != NULL) {
				free (err_msg);
			}
			lwc_writeLog (	LOG_INFO,
		_("Still get errors from the server.  I'll retry latter"));
		}
		count++;
#if HAVE_NANOSLEEP
		nanosleep (&req, NULL);
#elif HAVE_SLEEP
		sleep (10);
#elif HAVE_USLEEP
		usleep (10 * 1000000);
#endif
	}
	return 0;
}


/*
 * Check that the certificate and key are available or register the agent with
 * the server
 *
 * Return:
 *   0 --> No error
 *  -1 --> Error.  A message has been logged
 */
int
client_cert_check_and_create (	char ssl,
				const char *file_ca_cert,
				const char *file_cert,
				const char *file_key,
				char quick_random,
				const char *registrar_server,
				const char *registrar_port,
				const char *local_port)
{
#if HAVE_ASSERT_H
	assert (   registrar_server != NULL
		&& registrar_port != NULL && local_port != NULL);
#endif

#if HAVE_LIBGNUTLS
	if (ssl != 0) {
		int missing_file_ca_cert, missing_file_cert,
			missing_file_key, ret;
		struct stat buf;
		gnutls_x509_privkey_t key;


#if HAVE_ASSERT_H
	assert (file_ca_cert != NULL && file_cert != NULL && file_key != NULL);
#endif

		/* Check which files already exist */
		missing_file_key     = stat (file_key, &buf);
		missing_file_cert    = stat (file_cert, &buf);
		missing_file_ca_cert = stat (file_ca_cert, &buf);

		if (missing_file_key) {
			if (quick_random == 0) {
				lwc_writeLog (LOG_INFO,
			_("Creating SSL key (may take several minutes)..."));
			}
			key = generate_private_key (file_key, NULL, NULL);
			if (key == NULL) {
				return -1;
			}
			gnutls_x509_privkey_deinit (key);
			/* We need to (re)asked the server a certificate */
			missing_file_cert = 1;

			if (quick_random == 0) {
				lwc_writeLog (LOG_INFO, _("Done"));
			}

			/*
			 * A CSR must be generated and sent to the server to
			 * get a certificate
			 */
			ret = generate_and_send_csr (	file_key,
							registrar_server,
							registrar_port,
							local_port);
			if (ret < 0) {
				return -1;
			}

			/*
			 * As we've just generated a new private key, if the
			 * server already has a CSR or CRT for this agent, they
			 * cannot be used any more and they need to be removed
			 * from the server.
			 */
			if (ret > 0) {
				lwc_writeLog (	LOG_ERR,
				_("The server has an old pending request or certificate for this agent.  They must be removed.  See the log file on the server side for more details"));
				return -1;
			}
		}

		if (missing_file_cert) {
			ret = wait_for_cert (registrar_server, registrar_port,
						local_port, file_cert);
			if (ret < 0) {
				return -1;
			}

			/* The server doesn't have a CSR for this agent */
			if (ret > 0) {
				if (generate_and_send_csr (file_key,
							registrar_server,
							registrar_port,
							local_port) < 0)
				{
					return -1;
				}
				ret = wait_for_cert (	registrar_server,
							registrar_port,
							local_port, file_cert);
				if (ret < 0) {
					return -1;
				}
				if (ret > 0) {
					lwc_writeLog (	LOG_ERR,
					_("Cannot register with the server"));
					return -1;
				}
			}
		}

		if (	   missing_file_ca_cert
			&& get_ca (	registrar_server, registrar_port,
					file_ca_cert) != 0)
		{
			return -1;
		}
	}
	else
#endif /* HAVE_LIBGNUTLS  */
	if (register_no_ssl (	registrar_server, registrar_port,
				local_port) != 0)
	{
		return -1;
	}
	return 0;
}

/*-----------------============== End Of File ==============-----------------*/
