/* Schedwi
   Copyright (C) 2007 Herve Quatremain

   This file is part of Schedwi.

   Schedwi is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 3 of the License, or
   (at your option) any later version.

   Schedwi is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

/* workload.c -- Workloads management functions */

#include <schedwi.h>

#include <schedwi_interface.h>

#if STDC_HEADERS
#include <string.h>
#else
#if HAVE_STRING_H
#include <string.h>
#endif
#endif

#include <sql_status.h>
#include <message_windows.h>
#include <gschedwi_jobtree.h>
#include <child_job.h>
#include <cursor.h>
#include <main_cb.h>
#include <workload.h>

#define DATE_LEN 100


/*
 * Redraw the provided job on the canvas
 */
static void
workload_update_job (gpointer data, gpointer user_data)
{
	jobtree_ptr job_status_tree = (jobtree_ptr)user_data;
	child_job_t *job = (child_job_t *)data;
	schedwi_jobtree_node_ptr ptr, tmp;

	if (data == NULL || user_data == NULL) {
		return;
	}

	ptr = jobtree_find (	job_status_tree,
				g_ascii_strtoull (job->id, NULL, 0));
	if (ptr == NULL) {
		job->state = JOB_STATUS_STATE_UNDEFINED;
	}
	else {
		for (	tmp = ptr;
			tmp != NULL && tmp->run_time <= 0;
			tmp = tmp->parent);
		if (tmp == NULL) {
			job->run_time = 0;
		}
		else {
			job->run_time = tmp->run_time;
		}

		job->state            = ptr->status;
		job->start_time       = ptr->start_time;
		job->start_limit      = ptr->start_limit;
		job->max_duration     = ptr->max_duration;
		job->retries          = ptr->retries;
		job->retry_num        = ptr->retry_num;
		job->retries_interval = ptr->retries_interval;
		job->average_duration = ptr->average_duration;
		job->completion_pct   = ptr->run_pct;
		g_free (job->status_message);
		job->status_message   = g_strdup (ptr->status_message);
	}
	child_job_draw (job, job->parent_jobset->group_jobs);
}


/*
 * Update the jobs in the canvas
 */
static void
workload_merge (jobtree_ptr job_status_tree, GSList *jobs)
{
	g_slist_foreach (jobs, workload_update_job, job_status_tree);
}


/*
 * Callback function called when the selected item in the combo box is changed
 */
void
workload_changed (GtkComboBox *combo)
{
	jobset_item_t *current_jobset;
	gint i;
	GArray *workloads;
	guint32 val;
	schedwi_date workload_date;
	jobtree_ptr ptr;


	cursor_busy (NULL);

	/* Retrieve the currently displayed jobset */
	current_jobset = (jobset_item_t *) g_object_get_data (G_OBJECT (combo),
								"jobset");

	/* Retrieve the selected item */
	i =  gtk_combo_box_get_active (combo);

	/* Retrieve the workload list */
	workloads = (GArray *) g_object_get_data (	G_OBJECT (combo),
							"workloads");

	if (current_jobset == NULL || i < 0 || workloads == NULL) {
		cursor_normal (NULL);
		return;
	}

	/* Convert the workload date */
	val = g_array_index (workloads, guint32, i);
	if (schedwi_date_from_int (val, &workload_date) != 0) {
		cursor_normal (NULL);
		return;
	}

	/* Build the jobtree_ptr object for the selected workload */
	ptr = new_jobtree (workload_date);
	if (ptr == NULL) {
		cursor_normal (NULL);
		return;
	}

	/* Update the working area */
	workload_merge (ptr, current_jobset->jobs);

	free_jobtree (ptr);
	cursor_normal (NULL);
}


/*
 * Free the provided array
 */
static void
workload_free_workload_array (gpointer data)
{
	GArray *workloads = (GArray *)data;

	g_array_free (workloads, TRUE);
}


/*
 * Compare two arrays
 *
 * Return:
 *    TRUE --> The two arrays are the same
 *   FALSE --> The arrays are different
 */
static gboolean
workload_compare_workloads (GArray *a, GArray *b)
{
	guint i;

	if (a == NULL) {
		if (b == NULL) {
			return TRUE;
		}
		else {
			return FALSE;
		}
	}
	if (b == NULL) {
		return FALSE;
	}

	if (a->len != b->len) {
		return FALSE;
	}

	for (i = 0; i < a->len; i++) {
		if (g_array_index (a, guint32, i)
				!= g_array_index (b, guint32, i))
		{
			return FALSE;
		}
	}
	return TRUE;
}


/*
 * Convert the the provided date (YYYYMMDD) to a human format
 *
 * Return:
 *    A human readable date string to be freed by the caller by g_free()
 */
static gchar *
workload_to_date (guint32 val)
{
	GDate date;
	GDateDay day;
	GDateMonth month;
	GDateYear year;
	gchar *s;

	day = val % 100;
	val /= 100;
	month = val % 100;
	year = val / 100;

	/* Check that the date is valid and convert it to a string */
	if (g_date_valid_dmy (day, month, year) == TRUE) {
		g_date_clear (&date, 1);
		g_date_set_dmy (&date, day, month, year);
		s = g_malloc (DATE_LEN + 1);
		if (g_date_strftime (	s, DATE_LEN,
					_("%A %B %e, %Y"), &date) == 0)
		{
			/* Buffer to small.  Just convert val to a string */
			g_free (s);
			s = g_strdup_printf ("%u", val);
		}
	}
	else {
		/* The date is wrong, simply convert val to a string */
		s = g_strdup_printf ("%u", val);
	}

	return s;
}


/*
 * Associate a jobset to the combo box and redaw the jobs
 */
void
workload_set_jobset (GtkComboBox *combo, jobset_item_t *current_jobset)
{
	/* Add the jobset_item_t object to the combo */
	g_object_set_data (G_OBJECT (combo), "jobset", current_jobset);

	/* Draw the jobs */
	workload_changed (combo);
}


/*
 * Initialize the provided combo box with the workload dates in the job_status
 * database table
 *
 * Return:
 *    TRUE --> No error
 *   FALSE --> Error (an error message has been displayed)
 */
gboolean
workload_init_combo (GtkComboBox *combo)
{
	GArray *new_workloads, *old_workloads;
	lwc_LL *rows;
	char **row;
	gchar *date_str;
	guint i;
	gint active, j;
	guint32 val, old_active;

	/* Retrieve the workload list from the database */
	if (sql_status_workload_list (&rows,
				(void (*)(void *, const char*, unsigned int))
					error_window_ignore_errno,
				_("Database error")) != 0)
	{
		return FALSE;
	}

	/* Build the array of workloads */ 
	new_workloads = g_array_sized_new (	FALSE, FALSE, sizeof (guint32),
						lwc_getNumNode (rows));
	while ((row = (char **)lwc_delEndLL (rows)) != NULL) {
		/*
		 * row[0] is the workload date (YYYYMMDD)
		 */
		val = (guint32) g_ascii_strtoull (row[0], NULL, 0);
		g_array_append_val (new_workloads, val);
		sql_free_row (row);
	}
	lwc_delLL (rows, NULL);

	/* Remove the associated jobset_item_t object */
	g_object_set_data (G_OBJECT (combo), "jobset", NULL);

	/* Retrieve the current displayed workloads */
	old_workloads = (GArray *) g_object_get_data (	G_OBJECT (combo),
							"workloads");

	/* No previous workloads in the combo box */
	if (old_workloads == NULL) {
		/* Just add the workloads */ 
		for (i = 0; i < new_workloads->len; i++) {
			date_str = workload_to_date (
				g_array_index (new_workloads, guint32, i));
			gtk_combo_box_append_text (combo, date_str);
			g_free (date_str);
		}
		g_object_set_data_full (G_OBJECT (combo),
					"workloads", new_workloads,
					workload_free_workload_array);
		gtk_combo_box_set_active (combo, 0);
	}

	/* Previous workloads in the combo box */
	else {
		/* If the workloads are the same, nothing to do */
		if (workload_compare_workloads (new_workloads,
						old_workloads) == TRUE)
		{
			g_array_free (new_workloads, TRUE);
			return TRUE;
		}

		/* Currently active item */
		active = gtk_combo_box_get_active (combo);
		if (active >= 0) {
			old_active = g_array_index (	old_workloads, guint32,
							active);
		}
		else {
			old_active = 0;
		}

		/* Clear the combo box */
		for (j = old_workloads->len - 1; j >= 0; j--) {
			gtk_combo_box_remove_text (combo, j);
		}

		/* Fill the combo box with the new workloads */
		j = 0;
		for (i = 0; i < new_workloads->len; i++) {
			val = g_array_index (new_workloads, guint32, i);
			date_str = workload_to_date (val);
			gtk_combo_box_append_text (combo, date_str);
			g_free (date_str);
			if (val == old_active) {
				j = i;
			}
		}
		g_object_set_data_full (G_OBJECT (combo),
					"workloads", new_workloads,
					workload_free_workload_array);
		gtk_combo_box_set_active (combo, j);
	}

	return TRUE;
}


/*
 * Clear the provided combo box
 */
void
workload_clear (GtkComboBox *combo)
{
	GArray *old_workloads;
	guint *old_source_id;
	gint j;

	/* Retrieve the current displayed workloads */
	old_workloads = (GArray *) g_object_get_data (	G_OBJECT (combo),
							"workloads");

	if (old_workloads != NULL) {
		/* Clear the combo box */
		for (j = old_workloads->len - 1; j >= 0; j--) {
			gtk_combo_box_remove_text (combo, j);
		}

		/* Remove (and free) the array from the combo box widget */
		g_object_set_data_full (G_OBJECT (combo), "workloads", NULL,
					workload_free_workload_array);
	}

	/* Remove the associated jobset_item_t object */
	g_object_set_data (G_OBJECT (combo), "jobset", NULL);

	/* Remove the timer */
	old_source_id = (guint *) g_object_get_data (	G_OBJECT (combo),
							"source_id");
	if (old_source_id != NULL) {
		g_source_remove (*old_source_id);
	}
	g_object_set_data_full (G_OBJECT (combo), "source_id", NULL, g_free);
}


/*
 * Refresh the jobs and jobsets status.  This function is called at
 * update intervals specified by the spin button
 */
static gboolean
workload_timer (gpointer data)
{
	GtkComboBox *combo = (GtkComboBox *)data;

	/* Rebuild the combo */
	workload_init_combo (GTK_COMBO_BOX (combo));

	jobset_list_refresh (application_main);
	return TRUE;
}


/*
 * Change in the update interval
 */
void
workload_update_interval (GtkSpinButton *spinbutton)
{
	GtkWidget *combo;
	guint *new_source_id, *old_source_id;
	gint interval;

	combo = lookup_widget (	GTK_WIDGET (spinbutton),
				"combobox_main_workload");

	/* Remove the old timer */
	old_source_id = (guint *) g_object_get_data (	G_OBJECT (combo),
							"source_id");
	if (old_source_id != NULL) {
		g_source_remove (*old_source_id);
	}

	/* Set the new time */
	interval = gtk_spin_button_get_value_as_int (spinbutton);
	new_source_id = g_new (guint, 1);
	*new_source_id = g_timeout_add (interval * 1000,
					workload_timer, combo);
	
	/* Store the new ID in the combo object */
	g_object_set_data_full (G_OBJECT (combo), "source_id",
				new_source_id, g_free);
}


/*
 * Get the currently displayed workload
 */
guint32
workload_get_workload (GtkWidget *w)
{
	GtkWidget *combo;
	gint i;
	GArray *workloads;

	combo = lookup_widget (w, "combobox_main_workload");

	/* Retrieve the selected item */
	i =  gtk_combo_box_get_active (GTK_COMBO_BOX (combo));

	/* Retrieve the workload list */
	workloads = (GArray *) g_object_get_data (	G_OBJECT (combo),
							"workloads");

	if (i < 0 || workloads == NULL) {
		return 0;
	}

	/* Convert the workload date */
	return g_array_index (workloads, guint32, i);
}

/*-----------------============== End Of File ==============-----------------*/
