/* Schedwi
   Copyright (C) 2007 Herve Quatremain

   This file is part of Schedwi.

   Schedwi is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 3 of the License, or
   (at your option) any later version.

   Schedwi is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

/* link.c -- Structure to draw links between jobs into a GnomeCanvas */

#include <schedwi.h>

#include <schedwi_interface.h>

#include <math.h>

#include <message_windows.h>
#include <cache_pixbuf.h>
#include <sql_link.h>
#include <schedwi_style.h>
#include <cursor.h>
#include <main_cb.h>
#include <link.h>

#include <job_select.h>

#define ARROW_SHAPE_A	10
#define ARROW_SHAPE_B	10
#define ARROW_SHAPE_C	6
#define LINK_LINE_WIDTH	5
#define LINK_LINE_SELECT_WIDTH	10


/*
 * Destroy a link_t object
 */
void
destroy_link (link_t *ptr)
{
	if (ptr != NULL) {
		if (ptr->line_item != NULL) {
			gtk_object_destroy (GTK_OBJECT (ptr->line_item));
		}
		if (ptr->line_item_select != NULL) {
			gtk_object_destroy (GTK_OBJECT (ptr->line_item_select));
		}
		g_free (ptr);
	}
}


/*
 * Destroy an element from a list
 */
static void
destroy_element_link_list (gpointer data, gpointer user_data)
{
        destroy_link ((link_t *)data);
}


/*
 * Destroy the provided list of links
 */
void
destroy_link_list (GSList *list)
{
        g_slist_foreach (list, destroy_element_link_list, NULL);
        g_slist_free (list);
}


/*
 * Create and return a new link_t object
 *
 * Return:
 *   The new object (must be freed by the caller by destroy_link())
 */
link_t *
new_link_jobs (child_job_t *src, child_job_t *dst, job_status_state link_type)
{
	link_t *ptr;

	ptr = g_new0 (link_t, 1);
	ptr->src = src;
	ptr->dst = dst;
	child_job_add_link_out (src, ptr);
	child_job_add_link_in (dst, ptr);
	get_color_from_cache (	link_type,
				&(ptr->line_color[DEFAULT]),
				&(ptr->line_color[HIGHLIGHTED]),
				NULL, NULL);
	ptr->line_color[SELECTED] = ptr->line_color[DEFAULT];
	ptr->line_color[SELECTED_HIGHLIGHTED] = ptr->line_color[HIGHLIGHTED];
	ptr->line_color[CUT] = ptr->line_color[DEFAULT];


	ptr->link_type = link_type;
	ptr->selected = FALSE;
	return ptr;
}


/*
 * Create and return a new link_t object
 *
 * Return:
 *   The new object (must be freed by the caller by destroy_link())
 */
link_t *
new_link (	GSList *jobs, const gchar *id_src, const gchar *id_dst,
		job_status_state link_type)
{
	return new_link_jobs (	child_job_find (jobs, id_src),
				child_job_find (jobs, id_dst),
				link_type);
}


/*
 * Create a new link and add it to the provided list.
 * This function is the callback of the sql_link_list() function
 */
static int
add_link_to_list (	void *user_data,
			const char *id_src, const char *id_dst,
			const char *status)
{
	GSList **lists = user_data;
	link_t *ptr;

	ptr = new_link (lists[1], id_src, id_dst,
			job_status_state_str2status (status));
	if (ptr == NULL) {
		return -1;
	}
	lists[0] = g_slist_prepend (lists[0], ptr);
	return 0;
}


/*
 * Build and return the list of all the links connected to the jobs
 * contained in the provided list
 *
 * Return:
 *   The new list (to be freed by the caller by destroy_link_list()).
 *   NULL may be returned in case of error (an error message has been
 *   displayed) or if no links are associated with the provided jobs
 */
GSList *
get_link_list (GSList *jobs, int workload_date)
{
	gchar *jobs_string;
	GSList *lists[2];

	if (jobs == NULL) {
		return NULL;
	}

	jobs_string = get_child_job_list_ids (jobs);

	lists[0] = NULL;
	lists[1] = jobs;
	if (sql_link_list (	workload_date,
				jobs_string, add_link_to_list, lists,
				(void (*)(void *, const char*, unsigned int))
					error_window_ignore_errno,
				_("Database error")) != 0)
        {
		g_free (jobs_string);
                destroy_link_list (lists[0]);
                return NULL;
        }
	g_free (jobs_string);
	return lists[0];
}


/*
 * Change the link aspect according to the provided mode
 */
static void
link_set_mode (link_t *ptr, pixbuf_highlight_t new_mode)
{
	if (ptr == NULL) {
		return;
	}
	gnome_canvas_item_set (	ptr->line_item,
				"fill-color-gdk", ptr->line_color[new_mode],
				NULL);
	if (new_mode == SELECTED || new_mode == SELECTED_HIGHLIGHTED) {
		gnome_canvas_item_show (ptr->line_item_select);
	}
	else {
		gnome_canvas_item_hide (ptr->line_item_select);
	}
}


/*
 * Display a message in the application bar
 */
static void
link_status_appbar_push (link_t *ptr)
{
	gchar *s;

	if (ptr != NULL && application_statusbar != NULL) {
		s = g_strdup_printf (_("%s needs to be %s for %s to start"),
				ptr->dst->name,
				job_status_state2str (ptr->link_type),
				ptr->src->name);
		gnome_appbar_push (application_statusbar, s);
		g_free (s);
	}
}


/*
 * Remove the message from the application bar
 */
static void
link_status_appbar_pop (link_t *ptr)
{
	if (ptr != NULL && application_statusbar != NULL) {
		gnome_appbar_pop (application_statusbar);
	}
}


/*
 * Callback for the `Change link type' item in the popup menu
 */
void
link_menu_change_type (GtkMenuItem *item, job_status_state new_link_type)
{
	link_t *ptr;

	ptr = (link_t *) g_object_get_data (	G_OBJECT (application_main),
						"link");
	cursor_busy (NULL);

	/* Delete from the database */
	if (sql_link_type (	ptr->src->id, ptr->dst->id,
				(long int)new_link_type,
				(void (*)(void *, const char*, unsigned int))
						error_window_ignore_errno,
				_("Database error")) != 0)
	{
		cursor_normal (NULL);
		return;
	}

	/* Refresh the canvas */
	cursor_normal (NULL);
	jobset_list_refresh (application_main);
}


/*
 * Delete the provided link
 *
 * Return:
 *    TRUE --> The link has been deleted
 *   FALSE --> Error or the operation has been cancelled
 */
gboolean
link_delete (link_t *ptr)
{
	if (ptr == NULL) {
		return TRUE;
	}

	/* Ask the user if she really wants to delete the selected link */
	if (question_delete_window (application_main,
		_("Are you sure you want to delete the selected link?"))
				== FALSE)
	{
		return FALSE;
	}

	cursor_busy (NULL);

	/* Delete from the database */
	if (sql_link_del1 (	ptr->src->id, ptr->dst->id,
				(void (*)(void *, const char*, unsigned int))
						error_window_ignore_errno,
				_("Database error")) != 0)
	{
		cursor_normal (NULL);
		return FALSE;
	}

	cursor_normal (NULL);

	/* Refresh the canvas */
	jobset_list_refresh (application_main);
	return TRUE;
}


/*
 * Callback for the `Delete' item in the popup menu
 */
void
link_menu_delete (GtkMenuItem *item)
{
	link_t *ptr;

	ptr = (link_t *) g_object_get_data (	G_OBJECT (application_main),
						"link");
	link_delete (ptr);
}


/*
 * Event callback
 */
static gboolean
link_event (	GnomeCanvasItem *canvasitem,
		GdkEvent *event,
		gpointer user_data)
{
	link_t *ptr = (link_t *)user_data;

	switch (event->type) {
		case GDK_ENTER_NOTIFY:
			link_set_mode (ptr, (ptr->selected == FALSE)
						? HIGHLIGHTED
						: SELECTED_HIGHLIGHTED);
			link_status_appbar_push (ptr);
			return TRUE;
	
		case GDK_LEAVE_NOTIFY:
			link_set_mode (ptr, (ptr->selected == FALSE)
						? DEFAULT : SELECTED);
			link_status_appbar_pop (ptr);
			return TRUE;

		default:
			return FALSE;
	}
}


/*
 * Draw a link in the provided canvas group
 */
void
link_draw (link_t *ptr, GnomeCanvasGroup *group)
{
	gdouble angle, dst_radius;
	GnomeCanvasPoints *points;
	GdkColor select_color;

	if (ptr->src == NULL || ptr->dst == NULL) {
		return;
	}

	/* Get the color of the selection */
	schedwi_style_get_base_color (GTK_STATE_SELECTED, &select_color);

	dst_radius = ptr->dst->radius;
	angle = atan2 (ptr->src->y - ptr->dst->y, ptr->src->x - ptr->dst->x); 
	points = gnome_canvas_points_new (2);
	points->coords[0] = ptr->src->x;
	points->coords[1] = ptr->src->y;
	points->coords[2] = ptr->dst->x + cos (angle) * dst_radius;
	points->coords[3] = ptr->dst->y + sin (angle) * dst_radius;

	/* Draw the selection mask */
	if (ptr->line_item_select != NULL) {
		gtk_object_destroy (GTK_OBJECT (ptr->line_item_select));
	}
	ptr->line_item_select = gnome_canvas_item_new (
				group,
				gnome_canvas_line_get_type (),
				"points", points,
				"last-arrowhead", TRUE,
				"arrow-shape-a", (gdouble)ARROW_SHAPE_A
						+ LINK_LINE_SELECT_WIDTH,
				"arrow-shape-b", (gdouble)ARROW_SHAPE_B
						+ LINK_LINE_SELECT_WIDTH,
				"arrow-shape-c", (gdouble)ARROW_SHAPE_C
						+ LINK_LINE_SELECT_WIDTH * 0.5,
				"width-units", (gdouble)LINK_LINE_SELECT_WIDTH,
				"fill-color-gdk", &select_color,
				NULL);
	if (ptr->selected  == FALSE) {
		gnome_canvas_item_hide (ptr->line_item_select);
	}
	g_signal_connect (	(gpointer)ptr->line_item_select, "event",
				G_CALLBACK (link_event), ptr);

	/* Draw the link */
	if (ptr->line_item != NULL) {
		gtk_object_destroy (GTK_OBJECT (ptr->line_item));
	}
	dst_radius += 0.5 * LINK_LINE_SELECT_WIDTH;
	points->coords[2] = ptr->dst->x + cos (angle) * dst_radius;
	points->coords[3] = ptr->dst->y + sin (angle) * dst_radius;
	ptr->line_item = gnome_canvas_item_new (
				group,
				gnome_canvas_line_get_type (),
				"points", points,
				"last-arrowhead", TRUE,
				"arrow-shape-a", (gdouble)ARROW_SHAPE_A,
				"arrow-shape-b", (gdouble)ARROW_SHAPE_B,
				"arrow-shape-c", (gdouble)ARROW_SHAPE_C,
				"width-units", (gdouble)LINK_LINE_WIDTH,
				"fill-color-gdk", ptr->line_color[DEFAULT],
				NULL);
	gnome_canvas_points_free (points);
	g_signal_connect (	(gpointer)ptr->line_item, "event",
				G_CALLBACK (link_event), ptr);
}


/*
 * Draw all the links in the list to the provided canvas group
 */
void
link_draw_list (GSList *list, GnomeCanvasGroup *group)
{
	g_slist_foreach (list, (GFunc)link_draw, group);
}


/*
 * Unselect the provided link
 */
void
link_unselect (link_t *ptr)
{
	ptr->selected = FALSE;
	link_set_mode (ptr, DEFAULT);
}


/*
 * Select the provided link
 */
void
link_select (link_t *ptr)
{
	ptr->selected = TRUE;
	link_set_mode (ptr, SELECTED);
}


/*
 * Find the link connecting the two provided job
 *
 * Return:
 *   The corresponding link or
 *   NULL if not found
 */
link_t *
link_find (GSList *list, const gchar *job_src, const char *job_dst)
{
	link_t *ptr;

	while (list != NULL) {
		ptr = (link_t *)list->data;

		if (	   ptr->src != NULL
			&& strcmp (ptr->src->id, job_src) == 0
			&& ptr->dst != NULL
			&& strcmp (ptr->dst->id, job_dst) == 0)
		{
			return ptr;
		}
		list = g_slist_next (list);
	}
	return NULL;
}

/*-----------------============== End Of File ==============-----------------*/
