/* Schedwi
   Copyright (C) 2007 Herve Quatremain

   This file is part of Schedwi.

   Schedwi is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 3 of the License, or
   (at your option) any later version.

   Schedwi is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

/*
 * grid.c -- Structure to draw a grid on a GnomeCanvas
 *
 * Usage:
 *
 *    #include <grid.h>
 *    grid_t *grid_ptr;
 *
 *    // Create a new Grid object
 *
 *    grid_ptr = new_grid ();
 *
 *
 *    // Initialize the grid:
 *    // - gnome_canvas_group is a GnomeCanvasGroup in which
 *    //   the grid will be drawn 
 *    // - 800 and 600 are the width and height of the canvas
 *
 *    set_grid_parent_group (grid_ptr, gnome_canvas_group, 800, 600);
 *
 *
 *    // Draw the grid
 *    show_grid (grid_ptr);
 *
 *
 *    // The the `Snap to grid' flag
 *    snap_to_grid (grid_ptr, TRUE);
 *
 *
 *    // Change the size of the canvas (the grid is redrawn)
 *    set_grid_parent_group (grid_ptr, gnome_canvas_group, 1024, 768);
 *
 *
 *    // Hide the grid
 *
 *    hide_grid (grid_ptr);
 *
 *
 *    // Convert the provided mouse coordinates to coordinates snaped to
 *    // grid.  The width and height of the object to draw at this position
 *    // must be provided (for calculation purpose)
 *    gdouble new_x, new_y;
 *    grid_get_coordinate (grid_ptr, icon_width, icon_height,
 *                         mouse_x, mouse_y,
 *                         &new_x, &new_y);
 *
 *
 *    // Destroy the grid
 *
 *    destroy_grid (grid_ptr);
 *
 *
 * Other parameters (like color, line style, ...) are obtained with gconf
 */

#include <schedwi.h>

#include <schedwi_gconf.h>
#include <message_windows.h>
#include <grid.h>

#define WIDTH_PIXEL 1
#define CROSSHAIRS_LENGHT 10

#define DEFAULT_GRID_WIDTH 50
#define DEFAULT_GRID_HEIGHT 50
#define DEFAULT_GRID_COLOR "#808080"
#define DEFAULT_GRID_LINE_STYLE LINE_STYLE_SOLID


/*
 * Destroy the provided grid_t object
 */
void
destroy_grid (grid_t *ptr)
{
	if (ptr != NULL) {
		gconf_client_notify_remove (schedwi_gconf, ptr->gconf_notify);
		if (ptr->canvas_grid != NULL) {
			gtk_object_destroy (GTK_OBJECT (ptr->canvas_grid));
		}
		g_free (ptr);
	}
}


/*
 * Set/unset the `Snap to grid' flag
 */
void
snap_to_grid (grid_t *ptr, gboolean snap)
{
	if (ptr != NULL) {
		ptr->snap = snap;
	}
}


/*
 * Set the parent canvas group and the canvas size
 */
void
set_grid_parent_group (	grid_t *ptr,
			GnomeCanvasGroup *parent_group,
			gint canvas_width, gint canvas_height)
{
	if (ptr != NULL) {
		ptr->parent_group = parent_group;
		ptr->canvas_width = canvas_width;
		ptr->canvas_height = canvas_height;
		if (ptr->show == TRUE) {
			show_grid (ptr);
		}
	}
}


/*
 * Hide the grid
 */
void
hide_grid (grid_t *ptr)
{
	if (ptr != NULL) {
		ptr->show = FALSE;
		if (ptr->canvas_grid != NULL) {
			gtk_object_destroy (GTK_OBJECT (ptr->canvas_grid));
			ptr->canvas_grid = NULL;
		}
	}
}


/*
 * Show the grid
 */
void
show_grid (grid_t *ptr)
{
	GdkLineStyle style;
	GnomeCanvasPoints *points;
	guint x, y;

	if (	   ptr == NULL || ptr->parent_group == NULL
		|| ptr->spacing_width <= 0 || ptr->spacing_height <= 0)
	{
		return;
	}

	ptr->show = TRUE;
	if (ptr->canvas_grid != NULL) {
		gtk_object_destroy (GTK_OBJECT (ptr->canvas_grid));
	}

	ptr->canvas_grid = (GnomeCanvasGroup *)gnome_canvas_item_new (
				ptr->parent_group,
				gnome_canvas_group_get_type (),
				"x", (gdouble)0,
				"y", (gdouble)0,
				NULL);

	points = gnome_canvas_points_new (2);
	style = GDK_LINE_SOLID;
	switch (ptr->style) {
		case LINE_STYLE_DASH:
			style = GDK_LINE_ON_OFF_DASH;

		case LINE_STYLE_SOLID:
			points->coords[1] = (gdouble)0;
			points->coords[3] = (gdouble)ptr->canvas_height;
			for (	x = ptr->spacing_width;
				x < ptr->canvas_width;
				x += ptr->spacing_width)
			{
				points->coords[0] =
					points->coords[2] = (gdouble)x;
		
				gnome_canvas_item_new (
					ptr->canvas_grid,
					gnome_canvas_line_get_type (),
					"points", points,
					"width-pixels", WIDTH_PIXEL,
					"fill-color-gdk", &(ptr->color),
					"line-style", style,
					NULL);
			}

			points->coords[0] = (gdouble)0;
			points->coords[2] = (gdouble)ptr->canvas_width;
			for (	y = ptr->spacing_height;
				y < ptr->canvas_height;
				y += ptr->spacing_height)
			{
				points->coords[1] =
					points->coords[3] = (gdouble)y;
		
				gnome_canvas_item_new (
					ptr->canvas_grid,
					gnome_canvas_line_get_type (),
					"points", points,
					"width-pixels", WIDTH_PIXEL,
					"fill-color-gdk", &(ptr->color),
					"line-style", style,
					NULL);
			}
			break;

		case LINE_STYLE_CROSS:
			for (	x = ptr->spacing_width;
				x < ptr->canvas_width;
				x += ptr->spacing_width)
			{
				for (	y = ptr->spacing_height;
					y < ptr->canvas_height;
					y += ptr->spacing_height)
				{
					points->coords[0] =
						points->coords[2] = (gdouble)x;
					points->coords[1] = (gdouble)
						(y - CROSSHAIRS_LENGHT / 2); 
					points->coords[3] = (gdouble)
						(y + CROSSHAIRS_LENGHT / 2); 
					gnome_canvas_item_new (
						ptr->canvas_grid,
						gnome_canvas_line_get_type (),
						"points", points,
						"width-pixels", WIDTH_PIXEL,
						"fill-color-gdk", &(ptr->color),
						"line-style", style,
						NULL);

					points->coords[1] =
						points->coords[3] = (gdouble)y;
					points->coords[0] = (gdouble)
						(x - CROSSHAIRS_LENGHT / 2); 
					points->coords[2] = (gdouble)
						(x + CROSSHAIRS_LENGHT / 2); 
					gnome_canvas_item_new (
						ptr->canvas_grid,
						gnome_canvas_line_get_type (),
						"points", points,
						"width-pixels", WIDTH_PIXEL,
						"fill-color-gdk", &(ptr->color),
						"line-style", style,
						NULL);
				}
			}
			break;
	}
	gnome_canvas_points_free (points);
}


/*
 * Get the coordinate of an icon aligned to the grid (if the flag
 * `snap to grip' is set).  mouse_x and mouse_y must be in canvas world
 * coordinate
 *
 * out_x and out_y are set by this function.  They contain the x and y
 * coordinates aligned to the grid (if the flag `snap to grip' is set otherwise
 * they are equal to the mouse coordinates - mouse_x and mouse_y)
 */
void
grid_get_coordinate (	grid_t *ptr,
			gint icon_width, gint icon_height,
			gdouble mouse_x, gdouble mouse_y,
			gdouble *out_x, gdouble *out_y)
{
	if (ptr == NULL || ptr->snap == FALSE) {
		*out_x = mouse_x;
		*out_y = mouse_y;
		return;
	}


	/*
	 * X
	 */

	/* Set *out_x to the closest grip point */
	if ((gint)mouse_x % ptr->spacing_width > ptr->spacing_width / 2) {
		*out_x = 	  (((gint)mouse_x / ptr->spacing_width) + 1)
				* ptr->spacing_width;
	}
	else {
		*out_x =	  ((gint)mouse_x / ptr->spacing_width)
				* ptr->spacing_width;
	}

	/* Adjust *out_x to align icon edge with the grip point */
	if (*out_x - mouse_x >= icon_width) {
		*out_x -= icon_width / 2;
	}
	else {
		if (mouse_x - *out_x >= icon_width) {
			*out_x += icon_width / 2;
		}
	}


	/*
	 * Y
	 */

	/* Set *out_y to the closest grip point */
	if ((gint)mouse_y % ptr->spacing_height > ptr->spacing_height / 2) {
		*out_y =	  (((gint)mouse_y / ptr->spacing_height) + 1)
				* ptr->spacing_height;
	}
	else {
		*out_y =	  ((gint)mouse_y / ptr->spacing_height)
				* ptr->spacing_height;
	}

	/* Adjust *out_y to align icon edge with the grip point */
	if (*out_y - mouse_y >= icon_height) {
		*out_y -= icon_height / 2;
	}
	else {
		if (mouse_y - *out_y >= icon_height) {
			*out_y += icon_height / 2;
		}
	}
}


/*
 * Convert a string to a line_style_t value
 *
 * Return:
 *   TRUE --> No error.  style contains the converted value
 *  FALSE --> Unknow line style in line_style_str
 */
static gboolean
str_to_line_style (const gchar *line_style_str, line_style_t *style)
{
	if (g_ascii_strncasecmp (line_style_str, "SOLID", 5) == 0) {
		*style = LINE_STYLE_SOLID;
		return TRUE;
	}
	if (g_ascii_strncasecmp (line_style_str, "DASH", 4) == 0) {
		*style = LINE_STYLE_DASH;
		return TRUE;
	}
	if (g_ascii_strncasecmp (line_style_str, "CROSS", 5) == 0) {
		*style = LINE_STYLE_CROSS;
		return TRUE;
	}
	return FALSE;
}


/*
 * GConf notify function for the grid keys
 */
static void
grid_notify (	GConfClient *client, guint cnxn_id,
		GConfEntry *entry, gpointer user_data)
{
	grid_t *ptr = user_data;

	if (strcmp (	entry->key,
			GCONF_PATH "/ui/grid/width") == 0)
	{
		ptr->spacing_width = gconf_value_get_int (entry->value);
	}
	else {
	if (strcmp (	entry->key,
			GCONF_PATH "/ui/grid/height") == 0)
	{
		ptr->spacing_height = gconf_value_get_int (entry->value);
	}
	else {
	if (strcmp (	entry->key,
			GCONF_PATH "/ui/grid/line_style") == 0)
	{
		if (str_to_line_style (	gconf_value_get_string (entry->value),
					&(ptr->style)) != TRUE)
		{
			ptr->style = DEFAULT_GRID_LINE_STYLE;
		}
	}
	else {
	if (strcmp (	entry->key,
			GCONF_PATH "/ui/grid/color") == 0)
	{
		if (gdk_color_parse (	gconf_value_get_string (entry->value),
					&(ptr->color)) != TRUE)
		{
			gdk_color_parse (DEFAULT_GRID_COLOR, &(ptr->color));
		}
	}
	}}}

	if (ptr->show == TRUE) {
		show_grid (ptr);
	}
}


/*
 * Create and return a new grid_t object
 *
 * Return:
 *   The new object (to be freed by the caller by destroy_grid())
 */
grid_t *
new_grid ()
{
	grid_t *ptr;
	gchar *s, *color_str, *line_style_str;
	GError *err;

	ptr = g_new0 (grid_t, 1);

	ptr->show = FALSE;
	ptr->snap = FALSE;

	/*
	 * Retrieve grid spacing witdh from GConf
	 */
	err = NULL;
	ptr->spacing_width = gconf_client_get_int (schedwi_gconf,
						GCONF_PATH "/ui/grid/width",
						&err);
	if (ptr->spacing_width == 0) {
		if (err != NULL) {
			s = g_strdup_printf (
					_("Invalid grid width in GConf: %s"),
					err->message);
			g_error_free (err);
			warning_window (s,
				"A default width will be used instead.");
			g_free (s);
		}
		ptr->spacing_width = DEFAULT_GRID_WIDTH;
	}

	/*
	 * Retrieve grid spacing height from GConf
	 */
	err = NULL;
	ptr->spacing_height = gconf_client_get_int (schedwi_gconf,
						GCONF_PATH "/ui/grid/height",
						&err);
	if (ptr->spacing_height == 0) {
		if (err != NULL) {
			s = g_strdup_printf (
					_("Invalid grid height in GConf: %s"),
					err->message);
			g_error_free (err);
			warning_window (s,
				"A default height will be used instead.");
			g_free (s);
		}
		ptr->spacing_height = DEFAULT_GRID_HEIGHT;
	}
	
	/*
	 * Retrieve grid color from GConf
	 */
	err = NULL;
	color_str = gconf_client_get_string (	schedwi_gconf,
						GCONF_PATH "/ui/grid/color",
						&err);
	if (color_str == NULL) {
		if (err != NULL) {
			s = g_strdup_printf (
					_("Invalid grid color in GConf: %s"),
					err->message);
			g_error_free (err);
			warning_window (s,
				"A default color will be used instead.");
			g_free (s);
		}
		gdk_color_parse (DEFAULT_GRID_COLOR, &(ptr->color));
	}
	else {
		if (gdk_color_parse (color_str, &(ptr->color)) != TRUE) {
			s = g_strdup_printf (
					_("Invalid grid color `%s' in GConf"),
					color_str);
			warning_window (s,
				"A default color will be used instead.");
			g_free (s);
			gdk_color_parse (DEFAULT_GRID_COLOR, &(ptr->color));
		}
		g_free (color_str);
	}

	/*
	 * Retrieve the line style from GConf
	 */
	err = NULL;
	line_style_str = gconf_client_get_string (schedwi_gconf,
					GCONF_PATH "/ui/grid/line_style",
					&err);
	if (line_style_str == NULL) {
		if (err != NULL) {
			s = g_strdup_printf (
				_("Invalid grid line style in GConf: %s"),
				err->message);
			g_error_free (err);
			warning_window (s,
				"A default line style will be used instead.");
			g_free (s);
		}
		ptr->style = DEFAULT_GRID_LINE_STYLE;
	}
	else {
		if (str_to_line_style (line_style_str, &(ptr->style)) != TRUE)
		{
			s = g_strdup_printf (
				_("Invalid grid line style `%s' in GConf"),
				line_style_str);
			warning_window (s,
				"A default line style will be used instead.");
			g_free (s);
			ptr->style = DEFAULT_GRID_LINE_STYLE;

		}
		g_free (line_style_str);
	}

	ptr->gconf_notify = gconf_client_notify_add (schedwi_gconf,
					GCONF_PATH "/ui/grid",
					grid_notify,
					ptr,
					NULL,
					NULL);
	return ptr;
}

/*-----------------============== End Of File ==============-----------------*/
